# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Contains functions for verifying the SWI/X signature.
# This separate library is used in CLI code (for example) because
# zipfile and M2Crypto, which VerifySwi program uses, are CLI memory hogs.
# That's also why for continued use in CLI code, zipfile and etc cannot
# be imported globally in this library.

from __future__ import absolute_import, division, print_function

SIG_FILE_NAME = 'swi-signature'
SWIX_SIG_FILE_NAME = 'swix-signature'

def isSwixFile( filename ):
   ''' Returns boolean of whether file is a SWIX '''
   import zipfile
   return zipfile.is_zipfile( filename ) and filename.endswith( '.swix' )

def signatureFileName( filename ):
   ''' Figure out which signature file to use based on the filename '''
   return SWIX_SIG_FILE_NAME if isSwixFile( filename ) else SIG_FILE_NAME

def swiSignatureExists( swiFile ):
   # VerifySwi program can also check if a SWI is signed, but there are cases
   # where the inputted file might not exist or be a SWI file, which would
   # have different error codes associated. Also it would run the entire
   # program if a signature exists. This is simpler for a quick check.
   import zipfile
   if not zipfile.is_zipfile( swiFile ):
      return False
   with zipfile.ZipFile( swiFile, 'r' ) as swi:
      if signatureFileName( swiFile ) not in swi.namelist():
         return False
      else:
         return True

def swiSignedWithDevCA( swiFile ):
   ''' Returns boolean indicating whether SWI was signed with the devCA '''
   import VerifySwi
   if not swiSignatureExists( swiFile ):
      return False

   swiSig = VerifySwi.getSwiSignatureData( swiFile )
   devCA = VerifySwi.loadRootCert( VerifySwi.DEV_ROOT_CA_FILE_NAME )
   return not VerifySwi.checkSigningCert( swiSig, devCA )

def verifySwiSignature( swiFile, rootCA=None, userHint=False ):
   # Returns ( boolean of whether SWI signature is valid, error message ).
   # If no rootCA is given, uses the CA stored in /etc/swi-signature-rootCa.crt
   import VerifySwi
   retCode, caUsed = VerifySwi.verifySwi( swiFile, rootCA=rootCA )

   if userHint:
      if retCode == VerifySwi.VERIFY_SWI_RESULT.SUCCESS:
         message = VerifySwi.VERIFY_SWI_MESSAGE[ retCode ]
      else:
         message = ( f"{VerifySwi.VERIFY_SWI_MESSAGE[ retCode ]}"
                    f" The authenticity of the image could not be verified."
                    f" Please ensure this is an official, signed image and"
                    f" that the file is not corrupted (verify the checksum)." )
   else:
      message = VerifySwi.VERIFY_SWI_MESSAGE[ retCode ]

   return ( retCode == VerifySwi.VERIFY_SWI_RESULT.SUCCESS,
            message, caUsed )

def verifySwixSignature( swixFile, rootCAs, rootCAIsFile=True ):
   # Returns ( boolean of whether SWIX signature is valid, error message ).
   # Tries for all rootCAs until it finds one that trusts the signing crt.
   import VerifySwi
   for rootCA in rootCAs:
      retCode, caUsed = VerifySwi.verifySwi( swixFile, rootCA,
                                             rootCAIsFile=rootCAIsFile )
      if not ( retCode == VerifySwi.VERIFY_SWI_RESULT.ERROR_CERT_MISMATCH or
               retCode == VerifySwi.VERIFY_SWI_RESULT.ERROR_INVALID_ROOT_CERT ):
         return ( retCode == VerifySwi.VERIFY_SWI_RESULT.SUCCESS,
                  VerifySwi.VERIFY_SWIX_MESSAGE[ retCode ], caUsed )
   # None of the rootCas matched the signing crt (or none were actual root CAs...)
   retCode = VerifySwi.VERIFY_SWI_RESULT.ERROR_CERT_MISMATCH
   return ( False, VerifySwi.VERIFY_SWIX_MESSAGE[ retCode ], None )
