#!/usr/bin/python3
#
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
import os
from importlib import resources

SpecTemplate = "/usr/share/Swix/Lowmem/lowmem.spec.in"

PayloadMetadataTags = "{{PAYLOAD_METADATA_TAGS}}"
PayloadFiles = "{{PAYLOAD_FILES}}"
PayloadInstall = "{{PAYLOAD_INSTALL}}"
PayloadUninstall = "{{PAYLOAD_UNINSTALL}}"
PayloadFileTagsList = [ PayloadMetadataTags, PayloadFiles, PayloadInstall,
                        PayloadUninstall ]

ConfigMountScript = "{{CONFIG_MOUNT_SCRIPT}}"
ConfigPreInstallPayload = "{{CONFIG_PRE_INSTALL_PAYLOAD}}"
ConfigPostInstallPayload = "{{CONFIG_POST_INSTALL_PAYLOAD}}"
ConfigPreUninstallPayload = "{{CONFIG_PRE_UNINSTALL_PAYLOAD}}"
ConfigPostUninstallPayload = "{{CONFIG_POST_UNINSTALL_PAYLOAD}}"
ConfigFileTagsList = [ ConfigMountScript, ConfigPreInstallPayload,
                       ConfigPostInstallPayload, ConfigPreUninstallPayload,
                       ConfigPostUninstallPayload ]

ConfigMountPoint = "{{CONFIG_MOUNT_POINT}}"
ConfigRpmName = "{{CONFIG_RPM_NAME}}"
ConfigRpmVersion = "{{CONFIG_RPM_VERSION}}"
ConfigRpmRelease = "{{CONFIG_RPM_RELEASE}}"
ConfigRpmSummary = "{{CONFIG_RPM_SUMMARY}}"
ConfigRpmLicense = "{{CONFIG_RPM_LICENSE}}"
ConfigRpmVendor = "{{CONFIG_RPM_VENDOR}}"
ConfigRpmUrl = "{{CONFIG_RPM_URL}}"
ConfigRpmDescription = "{{CONFIG_RPM_DESCRIPTION}}"
ConfigTagsList = [ ConfigMountPoint, ConfigRpmName, ConfigRpmVersion,
                   ConfigRpmRelease, ConfigRpmSummary, ConfigRpmLicense,
                   ConfigRpmVendor, ConfigRpmUrl, ConfigRpmDescription ]

def readFileAppendingGhostTags( file ):
   output_content = ''
   for l in file:
      output_content += "%ghost " + l
   return output_content

def loadLowmemSwixMountScript():
   with resources.open_text( "Swix.Lowmem", "mount.py" ) as f:
      return f.read()

def tryLoadRpmSpecFileContents( configYaml, section ):
   if section in configYaml:
      absPath = os.path.abspath( configYaml[ section ] )
      with open( absPath, 'r' ) as f:
         return f.read()
   else:
      return None

def generatePayloadTemplateMap( files, tags, install, uninstall ):
   configMap = {}
   with open( files, 'r' ) as listf, \
        open( tags, 'r' ) as tagsf, \
        open( install, 'r' ) as installf, \
        open( uninstall, 'r' ) as uninstallf:
      # As we are placing symlinks instead of actually installing files
      # directly, mark all files as %ghost
      configMap[ PayloadFiles ] = readFileAppendingGhostTags( listf )
      configMap[ PayloadMetadataTags ] = tagsf.read()
      configMap[ PayloadInstall ] = installf.read()
      configMap[ PayloadUninstall ] = uninstallf.read()
   return configMap

def generateRpmSpecTemplateMap( configYaml ):
   configMap = {
      ConfigRpmName: configYaml[ "name" ],
      ConfigRpmVersion: configYaml[ "version" ],
      ConfigRpmRelease: configYaml[ "release" ],
      ConfigRpmSummary: configYaml[ "summary" ],
      ConfigRpmLicense: configYaml[ "license" ],
      ConfigRpmVendor: configYaml[ "vendor" ],
      ConfigRpmUrl: configYaml[ "url" ],
      ConfigRpmDescription: configYaml[ "description" ],
      ConfigMountPoint: configYaml[ "mountPoint" ],
   }

   with resources.open_text( "Swix.Lowmem", "mount.py" ) as f:
      mountScript = f.read()
   configMap[ ConfigMountScript ] = mountScript

   preInstallPayload = tryLoadRpmSpecFileContents( configYaml,
                                                   "preInstallPayload" )
   if preInstallPayload:
      configMap[ ConfigPreInstallPayload ] = preInstallPayload

   postInstallPayload = tryLoadRpmSpecFileContents( configYaml,
                                                    "postInstallPayload" )
   if postInstallPayload:
      configMap[ ConfigPostInstallPayload ] = postInstallPayload

   preUninstallPayload = tryLoadRpmSpecFileContents( configYaml,
                                                     "preUninstallPayload" )
   if preUninstallPayload:
      configMap[ ConfigPreUninstallPayload ] = preUninstallPayload

   postUninstallPayload = tryLoadRpmSpecFileContents( configYaml,
                                                      "postUninstallPayload" )
   if postUninstallPayload:
      configMap[ ConfigPostUninstallPayload ] = postUninstallPayload

   return configMap

def createSpec( rpmSpecConfig, files, deps, installCmds, uninstallCmds,
                specOutput ):

   # Generate maps that contain actual content to replace placeholders with
   payloadMap = generatePayloadTemplateMap( files, deps, installCmds,
                                            uninstallCmds )
   rpmSpecMap = generateRpmSpecTemplateMap( rpmSpecConfig )

   with open( SpecTemplate, 'r' ) as template, \
        open( specOutput, 'w' ) as output:

      for line in template:
         continueOuter = False

         # Replace placeholders with payload generation output file content
         for tag in PayloadFileTagsList:
            if tag in line:
               output.write( payloadMap[ tag ] )
               continueOuter = True
         if continueOuter:
            continue

         # Replace placeholders with config file entries with file content
         for tag in ConfigFileTagsList:
            if tag in line:
               if tag in rpmSpecMap:
                  output.write( rpmSpecMap[ tag ] )
               continueOuter = True
         if continueOuter:
            continue

         # Replace placeholders with config file entries directly
         newLine = line
         for tag in ConfigTagsList:
            if tag in newLine:
               newLine = newLine.replace( tag, rpmSpecMap[ tag ] )
         output.write( newLine )

if __name__ == '__main__':
   description = """
   Generates a SPEC file for rpmbuild from a lowmem RPM/SWIX spec template.
   Replaces {{TEMPLATE_VARIABLES}} with contents of relevant input files or
   config file entries.
   """

   ap = argparse.ArgumentParser( description=description )
   ap.add_argument( '-c', '--config', help='Extension config YAML filename' )
   ap.add_argument( '-f', '--files',
                    help='Input file with list of files metadata.' )
   ap.add_argument( '-d', '--deps',
                    help='Input file with list of dependency metadata.' )
   ap.add_argument( '-i', '--installCmds',
                    help="Input file with install commands." )
   ap.add_argument( '-u', '--uninstallCmds',
                    help="Input file with uninstall commands." )
   ap.add_argument( '-o', '--output',
                    help='Output spec file.' )
   args = ap.parse_args()

   configArg = args.config
   listFile = args.files
   depsFile = args.deps
   installFile = args.installCmds
   uninstallFile = args.uninstallCmds
   out = args.output

   createSpec( configArg, listFile, depsFile, installFile, uninstallFile, out )
