# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
from enum import Enum
import os
import sys
import SignatureFile
import SignatureRequest
import SwiSignLib
import Tac

class SWIX_SIGN_RESULT( Enum ):
   SUCCESS = 0
   ALREADY_SIGNED_INVALID = 1
   ALREADY_SIGNED_VALID = 2
   SERVER_ERROR = 3
   ERROR_NOT_A_SWIX = 4

class SwixSignException( Exception ):
   ''' Exception that occurs when we fail to sign a SWIX '''
   def __init__( self, code, message ):
      self.code = code
      super().__init__( message )

def signSwix( swixFile, forceSign=False, useDevCA=False, devCaKeyPair=None,
              signUrl=SignatureRequest.defaultSwiSignURL() ):
   ''' Sign a SWIX. Throws SwixSignException if it failed to sign. '''
   if not SwiSignLib.isSwixFile( swixFile ):
      raise SwixSignException( SWIX_SIGN_RESULT.ERROR_NOT_A_SWIX,
                               "Input is not a SWIX." )
   # Check if there's an existing signature
   signed = SwiSignLib.swiSignatureExists( swixFile )
   validSig, _, _ = SwiSignLib.verifySwiSignature( swixFile )
   if signed:
      if not forceSign: # pylint: disable=no-else-raise
         message = 'Warning: SWIX is signed with an invalid signature.'
         code = SWIX_SIGN_RESULT.ALREADY_SIGNED_INVALID
         if validSig:
            message = 'SWIX is already signed with a valid signature.'
            code = SWIX_SIGN_RESULT.ALREADY_SIGNED_VALID
         raise SwixSignException( code, message )
      else:
         # Force sign, remove swix-signature file from SWI
         Tac.run( [ 'zip', '-d', swixFile, SwiSignLib.SWIX_SIG_FILE_NAME ] )

   # Start signing process
   swixSignature = SignatureFile.Signature()
   # Use the basename of the file as the version
   swixData = SignatureFile.prepareDataForServer( swixFile,
                                                  os.path.basename( swixFile ),
                                                  swixSignature )
   try:
      if useDevCA:
         signatureData = SignatureRequest.getDataFromDevCA( swixFile, swixData,
                                                      devCaKeyPair=devCaKeyPair )
      else:
         signatureData = SignatureRequest.getDataFromServer( swixFile, swixData,
                                                      licenseServerUrl=signUrl )
      SignatureFile.generateSigFileFromServer( signatureData, swixFile,
                                               swixSignature )
   except SignatureRequest.SigningServerError as e:
      # Remove null signature
      Tac.run( [ 'zip', '-d', swixFile, SwiSignLib.SWIX_SIG_FILE_NAME ] )
      # pylint: disable-next=raise-missing-from
      raise SwixSignException( SWIX_SIGN_RESULT.SERVER_ERROR, str( e ) )

def sign( swixFile, forceSign=False, useDevCA=False, devCaKeyPair=None,
          signUrl=SignatureRequest.defaultSwiSignURL() ):
   ''' Sign a SWIX, returning a status code (enum) from SWIX_SIGN_RESULT '''
   try:
      signSwix( swixFile, forceSign, useDevCA, devCaKeyPair )
   except SwixSignException as e:
      print( e )
      return e.code
   else:
      return SWIX_SIGN_RESULT.SUCCESS

def signHandler( args ):
   parser = argparse.ArgumentParser( prog="swix sign" )
   parser.add_argument( "swix", metavar="EXTENSION.swix",
                        help="Path of the SWIX to be signed" )
   parser.add_argument( "--force-sign", help="Force signing the SWIX",
                        action="store_true" )
   parser.add_argument( "--dev-ca", help="Use development certificates for signing",
                        action="store_true" )
   parser.add_argument( "--intermediate-cert", type=int,
                        choices=list(
                           range( 1, len( SignatureRequest.SWI_SIGN_URL ) + 1 )
                        ),
                        default=SignatureRequest.DEFAULT_SWI_SIGN_URL_POS + 1,
                        help="Choose the intermediate CA to use for the signing "
                             "request" )

   args = parser.parse_args( args )
   signUrl = SignatureRequest.SWI_SIGN_URL[ args.intermediate_cert - 1 ]
   retCode = sign( args.swix, args.force_sign, args.dev_ca, signUrl=signUrl )
   exit( retCode.value ) # pylint: disable=consider-using-sys-exit

if __name__ == "__main__":
   signHandler( sys.argv[ 1: ] )
