# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import os
import sys
import Tac
from PyClient import PyClient, RpcError
import AgentDirectory

def buildExtraPreinitOptions( customDir=None ):
   # Default location of preinit profile is in either /usr/lib/preinit or
   # /usr/lib64/preinit. Some tests may want to load additional preinit profiles
   # outside of this directory in order to not pollute the default directory with
   # test related profiles. This function will return a list containing the default
   # preinit location and the custom directory "customDir", to be passed to
   # Sysdb/EntityManager
   envVar = os.environ.get( "SYSDB_EXTRA_PREINIT_PATH" )
   if not envVar and not customDir:
      return []
   elif envVar:
      customDir = envVar
   is64bit = sys.maxsize > 2**32
   defaultPreInitDir = '/usr/lib64/preinit/' if is64bit else '/usr/lib/preinit'
   return [ defaultPreInitDir, customDir ] if customDir else [ defaultPreInitDir ]

class SysdbPathHelper( object ):
   '''Provides a convience wrapper to access entities through Sysdb'''
   def __init__( self, sysname ):
      self.sysname = sysname
      self.sysdbRoot_ = None

   def _getSysdbRoot( self ):
      if not self.sysdbRoot_:
         assert AgentDirectory.agentIsRunning( self.sysname, "Sysdb" ), \
               "Sysdb is not running"
         self.sysdbRoot_ = PyClient( self.sysname, "Sysdb" ).agentRoot()
      return self.sysdbRoot_

   def _getEntityFromSysdb( self, path ):
      if AgentDirectory.agentIsRunning( self.sysname, "Sysdb" ):
         try:
            root = self._getSysdbRoot()
            return root.entity[ path ]
         except RpcError:
            return None
      else:
         return None

   def getEntity( self, path ):
      '''
      Method to get an entity residing at a path. Currently the Sysdb agent provides
      all paths in the system.
      '''
      assert isinstance( path, str )
      entity = self._getEntityFromSysdb( path )
      assert entity is not None, "Path %s not provided by Sysdb" % path
      return entity
