#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliInputWrapper as CliInput
import CliExtensions
import errno
import fcntl
import os
import struct
import sys
import termios
import time
import tty
import Tracing
import UtmpDump

th = Tracing.Handle( 'TerminalUtil' )
t0 = th.trace0

# The following is intended to support the "[no|default] terminal length|width"
# command.
#
# We do not have full control of the window size. The user can still resize the
# window even after running the terminal commands, or setting the window size
# manually through commands such as stty.  We don't prevent that.
#
# savedTerminalLength/Width stores the real terminal size. When we run "no terminal
# length", the terminal is reset to this value. Although this will work effectively
# only if the window was not resized, this is okay for most cases. The alternate
# approach of keeping track of window-resize events involves signal-handling which
# isn't desirable.

minTerminalWidth = 10
maxTerminalWidth = 32767
defaultTerminalLength = 24
defaultTerminalWidth = 80

def getTerminalFileno():
   # return the file descriptor of the terminal, or -1 if no terminal is found
   for f in ( sys.stdin, sys.stdout, sys.stderr ):
      try:
         fd = f.fileno()
         if os.isatty( fd ):
            return fd
      except AttributeError:
         # not all files have fileno() (e.g., cStringIO)
         pass
   return -1

def ioctl_GWINSZ():
   for f in ( sys.stdin, sys.stdout, sys.stderr ):
      try:
         fd = f.fileno()
         return struct.unpack( 'HHHH', fcntl.ioctl( fd, termios.TIOCGWINSZ,
                               '12345678' ) )
      except: # pylint: disable-msg=W0702
         pass
   return ( 0, 0, 0, 0 )

def ioctl_SWINLEN( numLines ):
   t0( "ioctl_SWINLEN", numLines )
   assert numLines <= 65535, "unsigned short overflow."
   try:
      win_params = ioctl_GWINSZ()
      if win_params[ 0 ] == numLines:
         return
      win_params_struct = struct.pack( 'HHHH', numLines, win_params[ 1 ],
                                       win_params[ 2 ], win_params[ 3 ] )
      fcntl.ioctl( sys.stdout.fileno(), termios.TIOCSWINSZ, win_params_struct )
   except: # pylint: disable-msg=W0702
      pass

# In addition to invoking ioctl with TIOCSWINSZ to set the terminal
# width, the environment variable COLUMNS is set to the correct value. This
# is required for 'less' to work correctly. Further, since readline
# does not alter its notion of terminal width, we adjust readline's notion
# of terminal width using rl_set_screen_width.
def ioctl_SWINWIDTH( numCols, session ):
   t0( "ioctl_SWINWIDTH", numCols )
   assert numCols <= maxTerminalWidth, "width too large"
   try:
      os.environ[ 'COLUMNS' ] = str( numCols )
      win_params = ioctl_GWINSZ()
      if win_params[ 1 ] == numCols:
         return
      win_params_struct = struct.pack( 'HHHH', win_params[ 0 ], numCols,
                                       win_params[ 2 ], win_params[ 3 ] )
      fcntl.ioctl( sys.stdout.fileno(), termios.TIOCSWINSZ, win_params_struct )
      if session:
         session.cliInput.TerminalWidthIs( numCols )
      else:
         CliInput.TerminalWidthIs( numCols ) # pylint: disable=no-member
   except OSError as e:
      if sys.stdout.isatty():
         # pylint: disable-next=consider-using-f-string
         t0( "IOCTL Error: %s" % ( str( e ) ) )

def isConsole():
   # conN is console, vtyN is through ssh
   return "con" in UtmpDump.getTtyName()

class TerminalContext:

   def __init__( self, session=None ):
      self.originalTerminalLength = -1
      self.configuredTerminalLength = -1
      self.originalTerminalWidth = -1
      self.configuredTerminalWidth = -1
      self.session = session

   def saveTerminalLength( self ):
      length = ioctl_GWINSZ()[ 0 ]
      if length > 0:
         if length == self.configuredTerminalLength:
            # We don't really know if this is the real size or just because of
            # the override. So we keep the previous value.
            return
      elif self.originalTerminalLength < 0:
         # In breadth tests runTestCli() actually sets it to 0 and we don't
         # want to change it to accidentally enable paging. On the product
         # this can only really happen on console which can have leftover state.
         if isConsole():
            length = defaultTerminalLength
      else:
         # do not override
         return
      self.originalTerminalLength = length
      t0( "saved terminal length", length )

   def saveTerminalWidth( self ):
      # Save terminal width to restore to
      width = ioctl_GWINSZ()[ 1 ]
      if width >= minTerminalWidth:
         if width == self.configuredTerminalWidth:
            return
      elif self.originalTerminalWidth <= 0:
         width = defaultTerminalWidth
      else:
         # Invalid width and we've already got a width
         return
      self.originalTerminalWidth = width
      t0( "saved terminal width", width )

   def terminalLengthOverrideIs( self, numLines ):
      t0( "terminalLengthOverrideIs", numLines )
      self.saveTerminalLength()
      self.configuredTerminalLength = numLines
      # 0 means no paging.  We dont call ioctl_SETWINLEN
      # since it would be unhelpful if we then dropped into bash mode
      # and used zile, etc. No screen actually has zero lines after all.
      if numLines == 0:
         return

      # -1 means default page length. We use originalTerminalLength.
      if numLines == -1:
         numLines = self.originalTerminalLength
      ioctl_SWINLEN( numLines )
      # There is no way to set terminal height, yet disable pagination
      # from cli. If you must, go to bash mode, and run stty

   def terminalWidthOverrideIs( self, numCols ):
      t0( "terminalWidthOverrideIs", numCols )
      self.saveTerminalWidth()
      self.configuredTerminalWidth = numCols
      # -1 means automatic page width. We use originalTerminalWidth.
      popColumns = False
      if numCols == -1:
         numCols = self.originalTerminalWidth
         popColumns = True
      elif numCols < minTerminalWidth:
         # should not happen
         return
      ioctl_SWINWIDTH( numCols, self.session )
      if popColumns:
         os.environ.pop( 'COLUMNS', None )

   def pagerDisabled( self ):
      # if window size < 2, it doesn't make sense to do paging
      return ( self.configuredTerminalLength == 0 or
               ioctl_GWINSZ()[ 0 ] < 2 )

class NoTerminalSettings:
   """Temporarily disable terminal settings."""
   def __init__( self, ctx ):
      self.ctx = ctx
      self.tcowner = False
      self.ttyFileno = getTerminalFileno()

   def __enter__( self ):
      try:
         if ( self.ttyFileno >= 0 and
              os.getpid() == os.tcgetpgrp( self.ttyFileno ) ):
            self.tcowner = True
      except OSError:
         # os.tcgetpgrp() throws ENOTTY if we don't have a controlling
         # terminal, so ignore it.
         pass
      # Restore to initial temrinal setting (to run bash commands, etc)
      if self.ctx.configuredTerminalWidth > 0:
         self.ctx.saveTerminalWidth()
         ioctl_SWINWIDTH( self.ctx.originalTerminalWidth, self.ctx.session )
         os.environ.pop( 'COLUMNS', None )
      if self.ctx.configuredTerminalLength > 0:
         self.ctx.saveTerminalLength()
         ioctl_SWINLEN( self.ctx.originalTerminalLength )

   def __exit__( self, _type, value, _traceback ):
      if self.tcowner:
         try:
            os.tcsetpgrp( self.ttyFileno, os.getpid() )
         except OSError:
            # This can fail if the controlling terminal is no longer
            # associated with us
            pass
      # Restore terminal if length/width is configured
      if self.ctx.configuredTerminalWidth > 0:
         ioctl_SWINWIDTH( self.ctx.configuredTerminalWidth, self.ctx.session )
      if self.ctx.configuredTerminalLength > 0:
         ioctl_SWINLEN( self.ctx.configuredTerminalLength )

def enableCtrlZ( enabled ):
   """Enable/Disable the regular behaviour (SIGTSTP) of the C-z character."""

   # Show command spawns a child subprocess and waits for it to complete.
   # A C-z pressed at this point will stop the child, and the parent process
   # will keep waiting forever. So we disable C-z globally while starting the Cli
   # session and restore the original settings before exiting from Cli process.
   # We temporarily enable C-z in Cli's bash command.
   #
   # Ignoring SIGTSTP in the child subprocess does not work. Initially C-z does get
   # blocked, however when we press C-c after C-z, the C-z's mask gets cleared.
   # This causes the queued C-z signal to get delivered to the child process, again
   # resulting in the original problem. I verified this by looking at the signal
   # attributes in /proc/<pid>/status.
   fileno = sys.stdin.fileno()
   if os.isatty( fileno ):
      termAttr = termios.tcgetattr( fileno )
      if enabled:
         susp = b'\x1a'
      else:
         susp = b'\x00'
      termAttr[ tty.CC ][ tty.VSUSP ] = susp
      termios.tcsetattr( fileno, tty.TCSANOW, termAttr )

class CtrlZ:
   """Manages Ctrl-Z settings."""
   def __init__( self, enabled ):
      self.enabled_ = enabled

   def __enter__( self ):
      enableCtrlZ( self.enabled_ )
      return self

   def __exit__( self, _type, value, _traceback ):
      enableCtrlZ( not self.enabled_ )
      return False

def isTerminalReal():
   """ Determine if I'm talking to a real terminal (vs. an
   expect-like script that doesn't bother to set TERM to dumb) by
   sending some VT100 control characters to stdout and waiting for
   a response."""
   stdinFd = sys.stdin.fileno()
   try:
      oldStdinSettings = termios.tcgetattr( stdinFd )
   except termios.error as e:
      if e.args[ 0 ] == errno.ENOTTY:
         return False
      raise
   if not sys.stdout.isatty():
      return False
   response = b""
   try:
      # Set stdin to raw, nonblocking mode.
      tty.setraw( stdinFd )
      f = fcntl.fcntl( stdinFd, fcntl.F_GETFL )
      f |= os.O_NONBLOCK
      fcntl.fcntl( stdinFd, fcntl.F_SETFL, f )

      # Send a "Query Device Status" request.
      sys.stdout.flush()
      os.write( sys.stdout.fileno(), b"\x1B[5n" )

      # Wait for a response.
      timeout = float( os.environ.get(
         "ARISTA_CLI_TERMINAL_HANDSHAKE_TIMEOUT", "0.5" ) )
      startTime = time.time()
      while time.time() - startTime < timeout:
         try:
            response = os.read( stdinFd, 4 )
            break
         except OSError as e:
            if e.errno != errno.EAGAIN:
               raise
   finally:
      # Restore stdin to previous state.
      termios.tcsetattr( stdinFd, termios.TCSADRAIN, oldStdinSettings )
      f &= ~os.O_NONBLOCK
      fcntl.fcntl( stdinFd, fcntl.F_SETFL, f )

   # If response was "Report Device OK", we have a real terminal.
   return response == b"\x1B[0n"

def isTerminalRealSafe():
   """ If any error occurs while determining the terminal status,
   give up on this minor feature."""
   try:
      return isTerminalReal()
   except: # pylint: disable-msg=W0702
      return False

def terminalCtrlSeq():
   term = os.environ.get( "TERM", "" )
   if term.startswith( "xterm" ):
      if isTerminalRealSafe():
         return ( b"\033]0;", b"\007" )  # pylint: disable-msg=W1401
   elif term.startswith( "screen" ):
      if isTerminalRealSafe():
         return ( b"\033k", b"\033\\" )  # pylint: disable-msg=W1401
   return None

def copyWinSize( fromFd, toFd ):
   ws = struct.pack( 'HHHH', 0, 0, 0, 0 )
   terminalSize = fcntl.ioctl( fromFd, termios.TIOCGWINSZ, ws )
   fcntl.ioctl( toFd, termios.TIOCSWINSZ, terminalSize )

def copyTtySpeed( fromFd, toFd ):
   fromAttrs = termios.tcgetattr( fromFd )
   toAttrs = termios.tcgetattr( toFd )[ : ]
   toAttrs[ 4 ] = fromAttrs[ 4 ] # copy the ispeed flag
   toAttrs[ 5 ] = fromAttrs[ 5 ] # copy the ospeed flag
   termios.tcsetattr( toFd, termios.TCSANOW, toAttrs )

def disableOutputProcessing( ttyFd ):
   if not os.isatty( ttyFd ):
      return
   attrs = termios.tcgetattr( ttyFd )
   attrs[ 1 ] = attrs[ 1 ] & ~termios.OPOST
   termios.tcsetattr( ttyFd, termios.TCSANOW, attrs )

def disableIsig( ttyFd ):
   if not os.isatty( ttyFd ):
      return
   attrs = termios.tcgetattr( ttyFd )
   attrs[ 3 ] = attrs[ 3 ] & ~termios.ISIG
   termios.tcsetattr( ttyFd, termios.TCSANOW, attrs )

def enableIsig( ttyFd ):
   if not os.isatty( ttyFd ):
      return
   attrs = termios.tcgetattr( ttyFd )
   attrs[ 3 ] = attrs[ 3 ] | termios.ISIG
   termios.tcsetattr( ttyFd, termios.TCSANOW, attrs )

# Due to the CLI pool server implementation, the CLI may get its real TTY or
# environment variables after the CLI (including plugins) are initialized.
# This hook allows plugins to register a callback when the TTY becomes known.
#
# Note, the Cli works not just in the FastCli environment, so the hook could
# be called one or twice. If Cli is directly invoked, it is called once at
# Cli init time. If FastCli is used, it is called twice: the first time at
# the Cli init time when FastClid spawns it, and the second time when FastCli
# is connected to the Cli process.
#
# For Capi Cli, the extensions are not notified when the client connects.
ttyHook = CliExtensions.CliHook()
