# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import threading
import os

import collections

class ThreadLocalData( collections.abc.MutableMapping ):
   def __init__( self, original ):
      self.original = original
      self.local_ = threading.local()

   def initialize( self ):
      if not hasattr( self.local_, 'data' ):
         self.local_.data = self.original.copy()

   def __setitem__( self, key, value ):
      self.initialize()
      self.local_.data[ key ] = value

   def __getitem__( self, key ):
      self.initialize()
      return self.local_.data[ key ]

   def __delitem__( self, key ):
      self.initialize()
      del self.local_.data[ key ]

   def __len__( self ):
      self.initialize()
      return len( self.local_.data )

   def __iter__( self ):
      self.initialize()
      return iter( self.local_.data )

def ThreadLocalEnv( env ):
   ''' Coax os.environ in to behaving itself sanely in our multi-threaded
      environment.

      The implementation of _Environ in os.py uses the _data member as a
      backing dictionary for the environment. We hijack that with a
      thread-local equivalent. We must also prevent it from calling
      setenv/putenv/unsetenv, which can cause the underlying "char **" backing
      "environ" to be reallocated in one thread, while being iterated in
      another.
      '''

   assert env == os.environ, "ThreadLocalEnv will only act on os.environ"
   # pylint: disable=protected-access
   os.environ._data = ThreadLocalData( env._data )

   def noop( *args, **kwargs ):
      pass

   os.putenv = noop
   os.unsetenv = noop
