# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import hashlib
import re
import struct

from TpmGeneric.Defs import HashAlg, NoSBToggle
from TpmGeneric.Impl import TpmImpl

class Tpm20( TpmImpl ):
   TPM_VERSION_MAJOR_STRING = '2'
   PCR_HASH_ALGORITHM = HashAlg.SHA256
   SB_TOGGLE_NVINDEX = '0x01400001'

   def newPcrHash( self ):
      return hashlib.new( self.PCR_HASH_ALGORITHM.value )

   def pcrRead( self, pcrIndex ):
      cmd = [ 'tpm2_pcrread', '%s:%d' % ( self.PCR_HASH_ALGORITHM.value, pcrIndex ) ]
      ret = self._runBinary( cmd )
      m = re.search( br'\s+%d\s*:\s+0x(\w+)$' % pcrIndex, ret )
      if not m:
         return b''
      return m.group( 1 ).lower()

   def pcrExtend( self, pcrIndex, pcrHash ):
      cmd = [
         'tpm2_pcrextend',
         '%d:%s=%s' % ( pcrIndex, self.PCR_HASH_ALGORITHM.value, pcrHash )
      ]
      self._runBinary( cmd )
      # The pcrextend command on TPM2.0 doesn't return the value of the PCR after
      # execution. Just do an explicit read.
      return self.pcrRead( pcrIndex )

   def readSbToggle( self ):
      try:
         self._runBinary( [ 'tpm2_nvreadpublic', self.SB_TOGGLE_NVINDEX ] )
      except:
         raise NoSBToggle()
      ret = self._runBinary( [ 'tpm2_nvread', self.SB_TOGGLE_NVINDEX ] )
      return struct.unpack( '=L', ret )[ 0 ]
