# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

from .Defs import NoTpmDevice, NoTpmImpl, NoSBToggle, PcrEventRevision
from .Impl.Tpm12 import Tpm12
from .Impl.Tpm20 import Tpm20
from .Log import PcrLog
import Toggles.TpmGenericToggleLib as toggleLib

toggleSysLogEnabled = toggleLib.toggleTpmGenericSysLogOnFailureEnabled()

if toggleSysLogEnabled:
   from .Defs import MEASUREDBOOT_PCR_EXT_FAIL, SECUREBOOT_STATE_LOAD_FAIL
   import Logging

# 1MB
MAX_READ_CHUNK = 1 * 1024 * 1024

TPM_IMPL_SUPPORTED = [ Tpm12, Tpm20 ]

class TpmGeneric:
   def __init__( self, tpmDevice='tpm0' ):
      self.impl_ = None
      self.tpmDevice_ = tpmDevice
      self.pcrLog_ = PcrLog()

   def tpmDevicePath( self ):
      return os.path.join( '/dev', self.tpmDevice_ )

   def impl( self ):
      if self.impl_:
         return self.impl_

      if not os.path.exists( self.tpmDevicePath() ):
         raise NoTpmDevice()

      for tpmImpl in TPM_IMPL_SUPPORTED:
         tpmImplObj = tpmImpl.probe( self.tpmDevice_ )
         if not tpmImplObj:
            continue
         self.impl_ = tpmImplObj
         return self.impl_

      raise NoTpmImpl()

   def pcrHashAlg( self ):
      '''
      Return the hash algorithm that should be used for PCRs
      '''
      return self.impl().PCR_HASH_ALGORITHM

   def pcrRead( self, pcrIndex ):
      return self.impl().pcrRead( pcrIndex )

   def pcrExtend( self, pcrIndex, event, hexHash, log=None ):
      '''
      Extend `pcrIndex` with `hexHash`.

      If the caller passes a non empty list as `log`, the PCR logs will be appended
      and the list will be used to fill the annotation part of the logs, separated
      by ' | '.
      '''
      ret = self.impl().pcrExtend( pcrIndex, hexHash )
      if log:
         self.pcrLog_.recordPcrExtension( pcrIndex, event.value,
                                          PcrEventRevision.get( event.name ),
                                          hexHash, log )
      return ret

   def pcrExtendFromData( self, pcrIndex, event, data, log=None ):
      '''
      Extend `pcrIndex` with the hash of `data` passed in parameter. The function
      will compute the hash of `data` using the algorithm supported on the current
      running hardware.

      If the caller passes a non empty list as `log`, the PCR logs will be appended
      and the list will be used to fill the annotation part of the logs, separated
      by ' | '.
      '''
      try:
         dataHash = self.impl().newPcrHash()
         dataHash.update( data if isinstance( data, bytes ) else data.encode() )
         return self.pcrExtend( pcrIndex, event, dataHash.hexdigest(), log=log )
      except:
         if toggleSysLogEnabled:
            Logging.log( MEASUREDBOOT_PCR_EXT_FAIL, pcrIndex, event )
         raise


   def hashFile( self, filePath ):
      '''
      Hash a file located at `filePath` using the hash algorithm we're expected to
      use for PCRs
      '''
      dataHash = self.impl().newPcrHash()

      with open( filePath, 'rb' ) as f:
         while True:
            readData = f.read( MAX_READ_CHUNK )
            if not readData:
               break
            dataHash.update( readData )

      return dataHash

   def pcrExtendFromFile( self, pcrIndex, event, filePath, log=None ):
      '''
      Extend `pcrIndex` with the hash of a file located at `filePath`. The file will
      be hashed using the algorithm supported on the current running hardware.

      If the caller passes a non empty list as `log`, the PCR logs will be appended
      and the list will be used to fill the annotation part of the logs, separated
      by ' | '.
      '''
      try:
         h = self.hashFile( filePath )
         return self.pcrExtend( pcrIndex, event, h.hexdigest(), log=log )
      except:
         if toggleSysLogEnabled:
            Logging.log( MEASUREDBOOT_PCR_EXT_FAIL, pcrIndex, event )
         raise

   def isToggleBitSet( self, bit ):
      try:
         sbToggle = self.impl().readSbToggle()
         mask = 1 << bit
         result = sbToggle & mask == mask
      except( NoTpmDevice, NoTpmImpl, NoSBToggle ):
         raise
      except:
         if toggleSysLogEnabled:
            Logging.log( SECUREBOOT_STATE_LOAD_FAIL )
         raise
      return result
