# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from enum import IntEnum
import re

SHA1_LEN = 20
SHA256_LEN = 32
TCG_EVENT_FLAG = 1 << 56
PCR_LOG_PATH = '/mnt/flash/tpm-data/pcr-extension.log'
ABOOT_PCR_LOG_REVISION_PATH = '/export/aboot/measuredboot-revisions'

PcrEventTypeCamelCase = {
   0x0: 'unknownEvent',
   0x1: 'abootImage',
   0x2: 'abootBootConfig',
   0x3: 'abootConsoleLogged',
   0x4: 'abootSwi',
   0x5: 'abootSbConfig',
   0x6: 'abootSbCert',
   0x7: 'abootPassword',
   0x100: 'eosKernelParams',
   0x101: 'eosRcEos',
   0x102: 'eosStartupConfig',
   0x103: 'eosExtensionInstall',
   0x104: 'eosExtensionUninstall',
   0x105: 'eosExtensionInstallFailure',
   0x106: 'eosExtensionUninstallFailure',
   TCG_EVENT_FLAG + 0x1: 'evPostCode',
   TCG_EVENT_FLAG + 0x7: 'evSCrtmContents',
   TCG_EVENT_FLAG + 0x80000001: 'evEfiVariableDriverConfig',
   TCG_EVENT_FLAG + 0x80000008: 'evEfiPlatformFirmwareBlob',
}

verIdx = 0

class LogIdxV1( IntEnum ):
   timestamp = 1
   pcr = 2
   hex = 3
   extra = 4

class LogIdxV2( IntEnum ):
   timestamp = 1
   pcr = 2
   event = 3
   hex = 4
   extra = 5

class LogIdxV3( IntEnum ):
   timestamp = 1
   pcr = 2
   event = 3
   hex = 4
   extra = 5
   eventRev = -1

logIdxList = [ 0, LogIdxV1, LogIdxV2, LogIdxV3 ]

def getEmptyNonce( tpmVersion ):
   if tpmVersion == '2.0':
      return '0' * SHA256_LEN * 2
   return '0' * SHA1_LEN * 2

def checkNonce( tpmVersion, nonce ):
   if tpmVersion == '1.2' and len( nonce ) != SHA1_LEN * 2:
      return ( 'Nonce should be hexadecimal '
               'string of length 40 on TPM 1.2' )
   if tpmVersion == '2.0' and len( nonce ) != SHA256_LEN * 2:
      return ( 'Nonce should be hexadecimal '
               'string of length 64 on TPM 2.0' )
   return ''

def pcrLogRead( pcrIndex, path=PCR_LOG_PATH ):
   pcrList = []

   with open( path ) as f:
      pcrLog = f.readlines()
      pcrLog = [ log.rstrip() for log in pcrLog ]

      if not pcrLog:
         raise ValueError( 'No PCR logs found in log file' )

      for log in pcrLog:
         logSplit = log.split( ' | ' )
         try:
            logVer = int( logSplit[ verIdx ] )
         except ValueError as e:
            raise ValueError( 'PCR log version cannot be cast to int' ) from e
         if len( logSplit ) < len( logIdxList[ logVer ] ):
            raise ValueError( 'Missing fields in logs' )
         if pcrIndex is None or pcrIndex == int( logSplit[ LogIdxV1.pcr ] ):
            pcrList.append( logSplit )

      if not pcrList:
         raise ValueError( 'No PCR logs found for requested index' )

      return pcrList

def parseAbootPcrEventRevision( path=ABOOT_PCR_LOG_REVISION_PATH ):
   abootRevDict = {}

   with open( path ) as f:
      revisions = f.readlines()

      if not revisions:
         raise ValueError( 'No revisions found in Aboot PCR event revision file' )

      exp = re.compile( r'REV_(?P<revName>.*)="(?P<revNum>\d+)"' )
      for rev in revisions:
         matches = exp.match( rev )
         if not matches:
            raise ValueError( 'Unknown format in Aboot PCR event revision file' )

         revNum = int( matches.group( 'revNum' ) )
         revName = matches.group( 'revName' )
         abootRevDict[ revName ] = revNum

      return abootRevDict
