# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from __future__ import absolute_import, division, print_function
from PolicyMap import numericalRangeToRangeString
from Toggles.TrafficPolicyToggleLib import (
      toggleTrafficPolicyEnforceGtsmEnabled
)
from ClassificationLib import (
      numericalRangeToSet,
      getTcpFlagAndMasksEst,
      getTcpFlagAndMaskInit,
)
import Tac
import Arnet
import six
from six.moves import map
tacNeighborProtocol = Tac.Type( 'Classification::NeighborProtocol' )
protocolBgp = tacNeighborProtocol.protocolBgp
protocolBgpTtlSec = tacNeighborProtocol.protocolBgpTtlSec
tacMatchL4Protocol = Tac.Type( 'Classification::MatchL4Protocol' )
matchL4Bgp = tacMatchL4Protocol.matchL4Bgp

ActionType = Tac.Type( "PolicyMap::ActionType" )
TcpFlag = Tac.Type( "Classification::TcpFlag" )
TcpFlagAndMask = Tac.Type( "Classification::TcpFlagAndMask" )
fragmentType = Tac.Type( "Classification::FragmentType" )
EthIntfId = Tac.Type( 'Arnet::EthIntfId' )
IntfId = Tac.Type( 'Arnet::IntfId' )
PortChannelIntfId = Tac.Type( 'Arnet::PortChannelIntfId' )
InternalRecircIntfId = Tac.Type( 'Arnet::InternalRecircIntfId' )
DropWithMessageType4 = Tac.Type( 'PolicyMap::DropWithMessageType4' )
DropWithMessageType6 = Tac.Type( 'PolicyMap::DropWithMessageType6' )
DropWithCliMsgType = Tac.Type( 'TrafficPolicy::DropWithCliMsgType' )

dropWithMsgTypeToToken = {
   DropWithMessageType4.icmpNetUnreachable :
      DropWithCliMsgType.icmpNetUnreachable.token,
   DropWithMessageType4.icmpHostUnreachable :
      DropWithCliMsgType.icmpHostUnreachable.token,
   DropWithMessageType4.icmpProtoUnreachable :
      DropWithCliMsgType.icmpProtoUnreachable.token,
   DropWithMessageType4.icmpPortUnreachable :
      DropWithCliMsgType.icmpPortUnreachable.token,
   DropWithMessageType4.icmpNetProhibited :
      DropWithCliMsgType.icmpNetProhibited.token,
   DropWithMessageType4.icmpHostProhibited :
      DropWithCliMsgType.icmpHostProhibited.token,
   DropWithMessageType4.icmpAdminProhibited :
      DropWithCliMsgType.icmpAdminProhibited.token,
   DropWithMessageType4.tcpReset :
      DropWithCliMsgType.tcpReset.token,
   DropWithMessageType6.icmp6NoRoute :
      DropWithCliMsgType.icmp6NoRoute.token,
   DropWithMessageType6.icmp6AdminProhibited :
      DropWithCliMsgType.icmp6AdminProhibited.token,
   DropWithMessageType6.icmp6AddrUnreachable :
      DropWithCliMsgType.icmp6AddrUnreachable.token,
   DropWithMessageType6.icmp6PortUnreachable :
      DropWithCliMsgType.icmp6PortUnreachable.token,
   DropWithMessageType6.tcp6Reset :
      DropWithCliMsgType.tcpReset.token,
}

# This can be imported by structured filter users such as
# DMF who'd like to get list of supported fields
structuredFilterFieldsList = [
      'vlanTag',
      'vlan',
      'source',
      'srcPrefixSet',
      'sourceLpm'
      'srcLpmPrefixSet',
      'destination',
      'dstPrefixSet',
      'dstLpmPrefixSet',
      'dstSelf',
      'protoNeighborsBgp',
      'protoBgp',
      'protocol',
      'fragment',
      'matchIpOptions',
      'matchTracked',
      'ttl',
      'dscp',
      'cos',
      'ecn',
      'length',
      'location',
      'nexthopGroup',
      'dzGreSwitchId',
      'dzGrePortId',
      'dzGrePolicyId',
      'appProfile' ]

def _addrListToStr( addrs ):
   return " ".join( map( str, sorted( addrs ) ) )

def numericalRangeToHexString( numRangeList, hexWidth ):
   numRangeList.sort()
   return ', '.join( [ x.hexStringValue( hexWidth ) for x in numRangeList ] )

def _numericalRangeToCmd( numRange, optionCmd, matchType, hexFormat=False,
                          hexWidth=0 ):
   numRangeList = [ aRange for aRange in numRange ]
   optionStr = numericalRangeToRangeString( numRangeList, matchType )
   if hexFormat:
      optionStr = numericalRangeToHexString( numRangeList, hexWidth )
   if optionStr:
      cmd = '%s %s' % ( optionCmd, optionStr )
      return cmd
   return ''

def _macAddrToCmd( macAddrs ):
   macAddrList = [ Arnet.EthAddr( i.mac ).displayString for i in macAddrs ]
   return " ".join( sorted( macAddrList ) )

def tcpFlagAndMaskStrings( tcpFlagAndMasks ):
   """
   For tcp flags we make use of Classification::TcpFlagAndMask.
   TcpFlagAndMask allows us to express states 0,1,X where X is don't care.
   If mask value for 'est' is 1 that means we care and we look at the tcpFlag value
   to see if it is a 0 or 1. If the mask value is 0 we do not care about the flag
   and treat it as X.
   When outputting our match rule command we only care for flags with a mask of 1,
   and if the tcpFlag is 0 we match on 'not flag' and if it is 1 we match on 'flag'
   """
   flags = [ attr.name for attr in TcpFlag.tacType.attributeQ
             if attr.isIndependentDomainAttr and attr.name != "value" ]
   hasInit = False
   hasEst = False

   flagAndMasksEst = getTcpFlagAndMasksEst()
   if all( flagAndMask in tcpFlagAndMasks for flagAndMask in flagAndMasksEst ):
      for flagAndMask in flagAndMasksEst:
         tcpFlagAndMasks.remove( flagAndMask )
      hasEst = True

   tcpFlagAndMaskInit = getTcpFlagAndMaskInit()
   if tcpFlagAndMaskInit in tcpFlagAndMasks:
      tcpFlagAndMasks.remove( tcpFlagAndMaskInit )
      hasInit = True

   cmds = []
   for tcpFlagAndMask in tcpFlagAndMasks:
      activeAttrs = []
      for attr in flags:
         if getattr( tcpFlagAndMask.tcpFlagMask, attr, False ):
            activeAttrs.append( attr )
      notFlags = []
      yesFlags = []
      for attr in activeAttrs:
         flagToAdd = attr
         if getattr( tcpFlagAndMask.tcpFlag, attr ):
            yesFlags.append( flagToAdd )
         else:
            notFlags.append( flagToAdd )
      assert set( yesFlags ).isdisjoint( set( notFlags ) )
      cmd = " flags"
      allFlags = sorted( yesFlags + notFlags )
      for flag in allFlags:
         cmd += " %s" % flag if flag in yesFlags else " not-%s" % flag
      cmds.append( cmd )
   if hasEst:
      cmd = " flags established"
      cmds.append( cmd )
   if hasInit:
      cmd = " flags initial"
      cmds.append( cmd )
   return cmds

def _protoToCmds( proto, optionCmd, matchType, l4PortSaveAll ):
   cmds = []
   if matchType == 'mac':
      return cmds

   protos = [] # protocols without additional fields
   protoFieldCmd = ""
   protoRange = [ aRange for aRange in proto ]
   protoRange.sort()
   for p in protoRange: # pylint: disable=too-many-nested-blocks
      tcpFlagAndMaskSet = proto[ p ].tcpFlagAndMask
      port = proto[ p ].port
      icmpTypeWithCodes = []
      icmpTypeWithoutCodes = []
      protoFieldWithIcmpCmd = ""
      icmpTypeColl = proto[ p ].icmpType
      for aRange in icmpTypeColl:
         if icmpTypeColl[ aRange ].icmpCode:
            icmpTypeWithCodes.append( aRange )
         else:
            icmpTypeWithoutCodes.append( aRange )
      if tcpFlagAndMaskSet or port or icmpTypeWithCodes or icmpTypeWithoutCodes:
         protoStr = numericalRangeToRangeString( [ p ], matchType )
         # Sort the ORed TCP flag configs by the default TcpFlagAndMask::operator<
         flagCmds = tcpFlagAndMaskStrings( sorted( tcpFlagAndMaskSet ) )
         for flagsCmd in flagCmds:
            protoFieldCmd = "protocol %s%s" % ( protoStr, flagsCmd )
            cmds.append( protoFieldCmd )
         # Sort the ORed l4 port configs by the numerical values of sport.
         # To implement this,
         # 1) Convert numerical ranges of sport to a list of numerical values, e.g.,
         #    protocol tcp source port 10-12,20 -> [10,11,12,20]
         #    protocol tcp source port 10-13 -> [10,11,12,13]
         # 2) Store the mapping of sport list to port id;
         # 3) Sort the list of lists. ->sort [ [ 10,11,12,20 ], [ 10,11,12,13 ] ]
         # 4) According to the sorted list, get the port id from the mapping and
         #    render the configs.
         portToIdMapping = {}
         portFsToIdMapping = {}
         sportAllPortId = 0
         sportFieldSetAllPortId = 0
         portLists = []
         portFsLists = []
         for portField in six.itervalues( port ):
            if portField.sport or portField.dport:
               if portField.sportFieldSet or portField.dportFieldSet:
                  continue
               if not portField.sport:
                  sportAllPortId = portField.id
                  continue
               sportList = list( numericalRangeToSet( portField.sport ) )
               if tuple( sportList ) not in portToIdMapping:
                  portToIdMapping[ tuple( sportList ) ] = []
                  portLists.append( sportList )
               portToIdMapping[ tuple( sportList ) ].append( portField.id )
            if portField.sportFieldSet or portField.dportFieldSet:
               if not portField.sportFieldSet:
                  sportFieldSetAllPortId = portField.id
                  continue
               portFsList = list( portField.sportFieldSet )
               portFsList.sort()
               portFsToIdMapping[ tuple( portFsList ) ] = portField.id
               portFsLists.append( portFsList )
         sortedPortLists = sorted( portLists )
         sortedPortFsLists = sorted( portFsLists )
         for sportList in sortedPortLists:
            portIds = portToIdMapping[ tuple( sportList ) ]
            for portId in portIds:
               portField = port[ portId ]
               protoFieldCmd = "protocol %s" % protoStr
               sport = [ aRange for aRange in portField.sport ]
               dport = [ aRange for aRange in portField.dport ]
               if sport:
                  sports = numericalRangeToRangeString( sport, matchType )
                  protoFieldCmd += " source port %s" % sports
                  if not dport and l4PortSaveAll:
                     protoFieldCmd += " destination port all"
               if dport:
                  dports = numericalRangeToRangeString( dport, matchType )
                  protoFieldCmd += " destination port %s" % dports
               cmds.append( protoFieldCmd )
         if sportAllPortId:
            protoFieldCmd = "protocol %s" % protoStr
            portField = port[ sportAllPortId ]
            dport = [ aRange for aRange in portField.dport ]
            dports = numericalRangeToRangeString( dport, matchType )
            if l4PortSaveAll:
               protoFieldCmd += " source port all destination port %s" % dports
            else:
               protoFieldCmd += " destination port %s" % dports
            cmds.append( protoFieldCmd )
         for portFsList in sortedPortFsLists:
            portId = portFsToIdMapping[ tuple( portFsList ) ]
            portField = port[ portId ]
            protoFieldCmd = "protocol %s" % protoStr
            sportFieldList = list( portField.sportFieldSet )
            dportFieldList = list( portField.dportFieldSet )
            sportFieldList.sort()
            dportFieldList.sort()
            sportFieldSet = " ".join( sportFieldList )
            dportFieldSet = " ".join( dportFieldList )
            if sportFieldSet:
               protoFieldCmd += " source port field-set %s" % sportFieldSet
            if dportFieldSet:
               protoFieldCmd += " destination port field-set %s" % dportFieldSet
            cmds.append( protoFieldCmd )
         if sportFieldSetAllPortId:
            protoFieldCmd = "protocol %s" % protoStr
            portField = port[ sportFieldSetAllPortId ]
            dportFieldSet = " ".join( portField.dportFieldSet )
            protoFieldCmd += " destination port field-set %s" % dportFieldSet
            cmds.append( protoFieldCmd )
         if icmpTypeWithoutCodes:
            icmpTypeWithoutCodes = numericalRangeToRangeString( icmpTypeWithoutCodes,
                                                                matchType )
            protoFieldWithIcmpCmd = "protocol %s type %s code all\n" % \
                                    ( protoStr, icmpTypeWithoutCodes )
         if icmpTypeWithCodes:
            icmpTypeWithCodes.sort()
            for aType in icmpTypeWithCodes:
               icmpCode = [ aRange for aRange in icmpTypeColl[ aType ].icmpCode ]
               icmpType = numericalRangeToRangeString( [ aType ], matchType )
               icmpTypeVal = numericalRangeToSet( [ aType ] )
               icmpCode = numericalRangeToRangeString( icmpCode, matchType,
                                                       list( icmpTypeVal )[ 0 ] )
               protoFieldWithIcmpCmd += "protocol %s type %s code %s\n" % \
                                       ( protoStr, icmpType, icmpCode )
         if protoFieldWithIcmpCmd:
            # do not add the last empty string by split()
            cmds.extend( protoFieldWithIcmpCmd.split( '\n' )[ : -1 ] )
      else:
         protos.append( p )
   if protos:
      protoCmd = "protocol %s" % \
         numericalRangeToRangeString( protos, matchType )
      cmds.append( protoCmd )
   return cmds

def _denyActionToCmd( denyAction ):
   # For regular drop, we may still have bool True passed in
   if not isinstance( denyAction, Tac.Type( "PolicyMap::DropAction" ) ):
      return 'drop'

   # msgType is a union; noMsg indicates regular drop case
   if denyAction.msgType.containsNoMsg():
      return 'drop'

   msgType = ( denyAction.msgType.dropWith4 if denyAction.msgType.containsDropWith4()
               else denyAction.msgType.dropWith6 )

   cmd = 'drop notification '
   if msgType.startswith( 'tcp' ):
      cmd += 'tcp'
   else:
      cmd += 'icmp'
      if '6' in msgType:
         cmd += 'v6'
      cmd += ' code'
   msgToken = dropWithMsgTypeToToken[ msgType ]
   cmd += f' {msgToken}'
   return cmd

def _policeActionToCmd( policeAction ):
   rate = policeAction.rateLimit.stringValue
   burst = policeAction.burstSize.stringValue
   if policeAction.burstSize.burstSize == 0:
      cmd = "police rate %s" % ( rate )
   else:
      cmd = "police rate %s burst-size %s" % ( rate, burst )
   return cmd

def _nexthopGroupToCmds( sf ):
   cmd = ''
   if sf.nexthopGroup:
      for nhgName in sorted( sf.nexthopGroup ):
         cmd += " " + nhgName
   return cmd

def _matchDescToCmds( sf ):
   cmd = ''
   if sf.ruleDesc:
      cmd = f'description {sf.ruleDesc}'
   return cmd

def _appProfileToCmds( sf ):
   if not sf.appProfile:
      return ''
   return " ".join( sorted( sf.appProfile ) )

def _locationMatchToCmds( sf ):
   cmds = []
   for udfName, udfInfo in sorted( sf.udf.items() ):
      locationBaseCmd = 'location ' + udfName + ' value '
      for udfValueAndMask in sorted( udfInfo.udfValueAndMask ):
         locationCmd = locationBaseCmd + hex( udfValueAndMask.udfValue )
         locationCmd += ' mask ' + hex( udfValueAndMask.udfMask )
         cmds.append( locationCmd )
   return cmds

def _destinationSelfToCmd( sf ):
   cmd = ''
   if sf.toCpu and sf.toCpu.matchFlag.selfIpUnicast:
      cmd = "destination prefix self unicast"
   return cmd

def _vlanToCmds( vlans, matchType ):
   cmds = []
   if not vlans:
      return cmds
   vlanCmd = _numericalRangeToCmd( vlans, "vlan", matchType )
   cmds.append( vlanCmd )
   return cmds

def _vlanTagToCmds( vlanTags, matchType ):
   cmds = []
   vlanTagListToCmdMapping = {}
   vlanTagFsListToCmdMapping = {}
   for vlanTag, dot1QMatch in vlanTags.items():
      innerVlanTagCmd = ''
      if dot1QMatch.innerVlanTag:
         innerVlanTagCmd = ' ' + _numericalRangeToCmd( dot1QMatch.innerVlanTag,
                                                 "inner", matchType )
      if vlanTag.vlan:
         vlanTagCmd = _numericalRangeToCmd( vlanTag.vlan, "dot1q vlan", matchType )
         vlanTagCmd += innerVlanTagCmd
         vlanTagList = list( numericalRangeToSet( vlanTag.vlan ) )
         vlanTagListToCmdMapping[ tuple( vlanTagList ) ] = vlanTagCmd
      if vlanTag.vlanFieldSet:
         vlanFsList = list( vlanTag.vlanFieldSet )
         vlanFsList.sort()
         sortedVlanFsStr = " ".join( vlanFsList )
         vlanTagFsCmd = "dot1q vlan field-set %s" % sortedVlanFsStr
         vlanTagFsCmd += innerVlanTagCmd
         vlanTagFsListToCmdMapping[ tuple( vlanFsList ) ] = vlanTagFsCmd
   for vlanTagList in sorted( vlanTagListToCmdMapping ):
      cmds.append( vlanTagListToCmdMapping[ vlanTagList ] )
   for vlanTagFsList in sorted( vlanTagFsListToCmdMapping ):
      cmds.append( vlanTagFsListToCmdMapping[ vlanTagFsList ] )
   return cmds

def _teidToCmds( sf, matchType ):
   cmds = []
   if matchType == 'mac':
      return cmds
   if sf.teid:
      teidCmd = _numericalRangeToCmd( sf.teid, 'teid', matchType )
      cmds.append( teidCmd )
   if sf.teidFieldSet:
      teidFsStr = " ".join( map( str, sorted( sf.teidFieldSet ) ) )
      teidFsCmd = "teid field-set integer %s" % teidFsStr
      cmds.append( teidFsCmd )
   return cmds

def _packetTypeToCmd( packetTypeSet, matchType ):
   cmds = []
   if matchType == 'mac':
      return cmds
   for packetType in packetTypeSet:
      if packetType.packetFlag.vxlanDecap and packetType.packetFlagMask.vxlanDecap:
         cmds.append( 'vxlan decap' )
      if not packetType.packetFlag.vxlanDecap and \
         packetType.packetFlagMask.vxlanDecap:
         cmds.append( 'vxlan decap exclude' )
      if packetType.packetFlag.multicast and packetType.packetFlagMask.multicast:
         cmds.append( 'multicast' )
   return cmds

def _ethTypeToCmds( sf, matchType ):
   cmds = []
   if sf.ethType:
      ethTypeCmd = _numericalRangeToCmd( sf.ethType, 'ethertype', matchType,
                                         True, 4 )
      cmds.append( ethTypeCmd )
   return cmds

def actionsToCmds( actions ):
   actionsCmds = []
   for actionType, action in actions.items():
      if actionType == ActionType.deny:
         cmd = _denyActionToCmd( action )
         actionsCmds.append( cmd )
      elif actionType == ActionType.police:
         cmd = _policeActionToCmd( action )
         actionsCmds.append( cmd )
      elif actionType == 'count':
         cmd = 'count'
         if action.counterName:
            cmd += ' %s' % action.counterName
         actionsCmds.append( cmd )
      elif actionType == 'log':
         actionsCmds.append( 'log' )
      elif actionType == 'actionGoto':
         if action.gotoClassName:
            actionsCmds.append( 'goto match %s' % action.gotoClassName )
         else:
            actionsCmds.append( 'goto next' )
      elif actionType == 'sample':
         actionsCmds.append( 'sample' )
      elif actionType == 'sampleAll':
         actionsCmds.append( 'sample all' )
      elif actionType == 'setDscp':
         cmd = 'set dscp %d' % action.dscp
         actionsCmds.append( cmd )
      elif actionType == 'setTc':
         cmd = 'set traffic class %d' % action.tc
         actionsCmds.append( cmd )
      elif actionType == 'reportFlow':
         cmd = 'report flow'
         actionsCmds.append( cmd )
      elif actionType == ActionType.loadBalance:
         cmd = 'load-balance nexthop-group %s' % action.nexthopGroupName
         actionsCmds.append( cmd )
      elif actionType == ActionType.setVrf:
         cmd = 'set vrf %s' % action.vrfName
         actionsCmds.append( cmd )
      elif actionType == ActionType.setIdentityTag:
         idTag = action.idTag
         cmd = 'set id-tag %d' % idTag.outer
         if idTag.inner:
            cmd += ' inner %d' % idTag.inner
         actionsCmds.append( cmd )
      elif actionType == ActionType.stripHeaderBytes:
         hdrInfo = action.stripHdrBytes.dot1qRemoveVlans
         cmd = 'remove dot1q outer %s' % hdrInfo
         actionsCmds.append( cmd )
      elif actionType == ActionType.setHeaderRemove:
         headerRemove = action.headerRemove
         cmd = 'remove header size %s' % headerRemove.size
         if headerRemove.preserveEth:
            cmd += ' preserve ethernet'
         actionsCmds.append( cmd )
      elif actionType == ActionType.setAggregationGroup:
         if action.aggGroup:
            groupList = list( action.aggGroup )
            cmd = 'redirect aggregation group '
            cmd += ' '.join( sorted( groupList ) )
            actionsCmds.append( cmd )

         ethList = [ intf for intf in action.aggIntf
                     if EthIntfId.isEthIntfId( intf ) ]
         poList = [ intf for intf in action.aggIntf
                    if PortChannelIntfId.isPortChannelIntfId( intf ) ]
         irList = [ intf for intf in action.aggIntf
                    if InternalRecircIntfId.isInternalRecircIntfId( intf ) ]
         assert len( ethList ) + len( poList ) + len( irList ) == \
                len( action.aggIntf )
         def redirectCmdForIntfList( cmds, intfList ):
            if not intfList:
               return
            shortNamePrefix = IntfId( intfList[ 0 ] ).shortName[ :2 ]
            cmdStr = 'redirect interface ' + shortNamePrefix
            cmdStr += \
               ','.join( IntfId( intf ).shortName.replace( shortNamePrefix, '' )
                  for intf in Arnet.sortIntf( intfList ) )
            cmds.append( '{}'.format( cmdStr ) )
         for intfList in [ ethList, poList, irList ]:
            redirectCmdForIntfList( actionsCmds, intfList )
      elif actionType == ActionType.setTtl:
         continue # handled in nexthop/nexthop-group action
      elif actionType == ActionType.setNexthopGroup:
         groupList = list( action.nexthopGroup )
         cmd = 'redirect next-hop group '
         cmd += ' '.join( sorted( groupList ) )
         if ActionType.setTtl in actions:
            ttlAction = actions[ ActionType.setTtl ]
            cmd += ' ttl %d' % ( ttlAction.ttl )
         actionsCmds.append( cmd )
      elif actionType == ActionType.setNexthop:
         nhops = ' '.join( sorted(
            [ nh.stringValue for nh in action.nexthop ] ) )
         cmd = 'redirect next-hop %s%s%s' % (
            'recursive ' if action.recursive else '', nhops,
            ( ' vrf %s' % action.vrfName )
            if action.vrfName != '' else '' )
         if ActionType.setTtl in actions:
            ttlAction = actions[ ActionType.setTtl ]
            cmd += ' ttl %d' % ( ttlAction.ttl )
         actionsCmds.append( cmd )
      elif actionType == ActionType.setTimestampHeader:
         cmd = 'set mac timestamp header'
         actionsCmds.append( cmd )
      elif actionType == ActionType.setMacAddress:
         macAddress = action.macAddress
         dstMac = Arnet.EthAddr( macAddress.destMac ).displayString
         srcMac = Arnet.EthAddr( macAddress.srcMac ).displayString
         cmd = 'set mac address dest %s' % dstMac
         if srcMac != '0000.0000.0000':
            cmd += ' src %s' % srcMac
         actionsCmds.append( cmd )
      elif actionType == ActionType.mirror:
         cmd = 'mirror session %s' % action.mirrorSession
         actionsCmds.append( cmd )
      elif actionType == ActionType.setVrfSecondary:
         actionsCmds.append( 'set vrf secondary %s' % action.vrfName )
      elif actionType == ActionType.useVrfSecondary:
         # `useVrfSecondary` is used for 'set vrf secondary' without a
         # specific VRF name.
         actionsCmds.append( 'set vrf secondary' )
      elif actionType == ActionType.sflow:
         actionsCmds.append( 'sflow' )
      elif actionType == ActionType.actionSet:
         # For action sets we return a tuple of action set name and list of sub
         # actions. For example, ( 'foo', [ 'redirect ...', 'set id-tag ...' ] )
         for actionSetName, actionSet in action.namedActionSet.items():
            actionSetActions = {}
            if actionSet.setAggGroup and (
                  actionSet.setAggGroup.aggGroup or
                  actionSet.setAggGroup.aggIntf ):
               actionSetActions[ actionSet.setAggGroup.actionType ] = \
                     actionSet.setAggGroup
            if actionSet.stripHeaderBytes:
               actionSetActions[ actionSet.stripHeaderBytes.actionType ] = \
                     actionSet.stripHeaderBytes
            if actionSet.setIdentityTag:
               actionSetActions[ actionSet.setIdentityTag.actionType ] = \
                     actionSet.setIdentityTag
            if actionSet.setMacAddress:
               actionSetActions[ actionSet.setMacAddress.actionType ] = \
                     actionSet.setMacAddress
            actionSetActionCmds = actionsToCmds( actionSetActions )
            replicateAction = ( actionSetName.rsplit( '%' )[ -1 ],
                                actionSetActionCmds )
            actionsCmds.append( replicateAction )
      elif actionType == ActionType.verifyFlowReturnFailure:
         cmd = 'verify flow return failure action %s' % action.failureAction
         actionsCmds.append( cmd )
      elif actionType == ActionType.setDecapVrf:
         cmd = 'set vrf decapsulation %s fallback %s forward %s' % \
            ( action.decapVrfName, action.fallbackVrfName, action.postDecapVrfName )
         actionsCmds.append( cmd )
      elif actionType == ActionType.redirectTunnel:
         cmd = 'redirect next-hop interface {}'.format( action.tunnelIntf )
         actionsCmds.append( cmd )
      else:
         raise NotImplementedError( "Unsupported action" )
   return actionsCmds

def structuredFilterToCmds( structuredFilter, policyActions, matchType,
                            l4PortSaveAll=True ):
   '''
   Takes a structured filter and returns a dictionary containing all commands
   used to create the filter.
   '''
   structuredFilterCmds = {}
   teidCmd = _teidToCmds( structuredFilter, matchType )
   if teidCmd:
      structuredFilterCmds[ 'teid' ] = teidCmd

   sourceAddrs = _addrListToStr( structuredFilter.source )
   if sourceAddrs:
      cmd = 'source prefix %s' % sourceAddrs
      structuredFilterCmds[ 'source' ] = cmd

   sourcePrefixSet = _addrListToStr( structuredFilter.srcPrefixSet )
   if sourcePrefixSet:
      cmd = 'source prefix field-set %s' % sourcePrefixSet
      structuredFilterCmds[ 'srcPrefixSet' ] = cmd

   sourceAddrs = _addrListToStr( structuredFilter.sourceLpm )
   if sourceAddrs:
      cmd = 'source prefix longest-prefix %s' % sourceAddrs
      structuredFilterCmds[ 'sourceLpm' ] = cmd

   sourceLpmPrefixSet = _addrListToStr( structuredFilter.srcLpmPrefixSet )
   if sourceLpmPrefixSet:
      cmd = 'source prefix longest-prefix field-set %s' % sourceLpmPrefixSet
      structuredFilterCmds[ 'srcLpmPrefixSet' ] = cmd

   dstAddrs = _addrListToStr( structuredFilter.destination )
   if dstAddrs:
      cmd = 'destination prefix %s' % dstAddrs
      structuredFilterCmds[ 'destination' ] = cmd

   dstPrefixSet = _addrListToStr( structuredFilter.dstPrefixSet )
   if dstPrefixSet:
      cmd = 'destination prefix field-set %s' % dstPrefixSet
      structuredFilterCmds[ 'dstPrefixSet' ] = cmd

   dstAddrs = _addrListToStr( structuredFilter.destinationLpm )
   if dstAddrs:
      cmd = 'destination prefix longest-prefix %s' % dstAddrs
      structuredFilterCmds[ 'destinationLpm' ] = cmd

   dstLpmPrefixSet = _addrListToStr( structuredFilter.dstLpmPrefixSet )
   if dstLpmPrefixSet:
      cmd = 'destination prefix longest-prefix field-set %s' % dstLpmPrefixSet
      structuredFilterCmds[ 'dstLpmPrefixSet' ] = cmd

   srcMacAddr = _macAddrToCmd( structuredFilter.srcMac )
   if srcMacAddr:
      cmd = 'source mac %s' % srcMacAddr
      structuredFilterCmds[ 'srcMacAddr' ] = cmd

   dstMacAddr = _macAddrToCmd( structuredFilter.dstMac )
   if dstMacAddr:
      cmd = 'destination mac %s' % dstMacAddr
      structuredFilterCmds[ 'dstMacAddr' ] = cmd

   ethType = _ethTypeToCmds( structuredFilter, matchType )
   if ethType:
      structuredFilterCmds[ 'ethType' ] = ethType

   srcMacAddrSet = _addrListToStr( structuredFilter.srcMacAddrSet )
   if srcMacAddrSet:
      cmd = 'source mac field-set %s' % srcMacAddrSet
      structuredFilterCmds[ 'srcMacAddrSet' ] = cmd

   dstMacAddrSet = _addrListToStr( structuredFilter.dstMacAddrSet )
   if dstMacAddrSet:
      cmd = 'destination mac field-set %s' % dstMacAddrSet
      structuredFilterCmds[ 'dstMacAddrSet' ] = cmd

   if policyActions:
      actionsCmds = actionsToCmds( policyActions )

      def sortFn( elem ):
         if isinstance( elem, tuple ):
            return f'replicate { elem[ 0 ] }'
         else:
            return elem
      structuredFilterCmds[ 'actions' ] = sorted( actionsCmds, key=sortFn )

   protoCmd = _protoToCmds( structuredFilter.proto, 'protocol', matchType,
                            l4PortSaveAll )
   if protoCmd:
      structuredFilterCmds[ 'protocol' ] = protoCmd

   serviceSet = _addrListToStr( structuredFilter.serviceSet )
   if serviceSet:
      cmd = f'protocol service field-set {serviceSet}'
      structuredFilterCmds[ 'serviceSet' ] = cmd


   ttlCmd = _numericalRangeToCmd( structuredFilter.ttl, 'ttl', matchType )
   if ttlCmd:
      structuredFilterCmds[ 'ttl' ] = ttlCmd

   dscpCmd = _numericalRangeToCmd( structuredFilter.dscp, 'dscp', matchType )
   if dscpCmd:
      structuredFilterCmds[ 'dscp' ] = dscpCmd

   cosCmd = _numericalRangeToCmd( structuredFilter.cos, 'cos', matchType )
   if cosCmd:
      structuredFilterCmds[ 'cos' ] = cosCmd

   ecnCmd = _numericalRangeToCmd( structuredFilter.ecn, 'ecn', matchType )
   if ecnCmd:
      structuredFilterCmds[ 'ecn' ] = ecnCmd

   lengthCmd = _numericalRangeToCmd( structuredFilter.length, 'ip length',
                                     matchType )
   if lengthCmd:
      structuredFilterCmds[ 'length' ] = lengthCmd

   if structuredFilter.fragment:
      if structuredFilter.fragment.fragmentType == fragmentType.matchAll:
         structuredFilterCmds[ 'fragment' ] = 'fragment'
      if structuredFilter.fragment.fragmentType == fragmentType.matchOffset:
         fragOffsetCmd = _numericalRangeToCmd( structuredFilter.fragment.offset,
                                               'fragment offset', matchType )
         if fragOffsetCmd:
            structuredFilterCmds[ 'fragment' ] = fragOffsetCmd

   if structuredFilter.matchIpOptions:
      structuredFilterCmds[ 'matchIpOptions' ] = 'ip options'

   if structuredFilter.matchTracked:
      structuredFilterCmds[ 'matchTracked' ] = 'tracked'

   if structuredFilter.neighborProtocol == protocolBgp:
      structuredFilterCmds[ 'protoNeighborsBgp' ] = 'protocol neighbors bgp'

   if toggleTrafficPolicyEnforceGtsmEnabled():
      if structuredFilter.neighborProtocol == protocolBgpTtlSec:
         structuredFilterCmds[ 'protoNeighborsBgpTtlSec' ] = \
            'protocol neighbors bgp enforce ttl maximum-hops'

   if structuredFilter.matchL4Protocol == matchL4Bgp:
      structuredFilterCmds[ 'protoBgp' ] = 'protocol bgp'

   vlanTagCmd = _vlanTagToCmds( structuredFilter.vlanTag, matchType )
   if vlanTagCmd:
      structuredFilterCmds[ 'vlanTag' ] = vlanTagCmd

   vlanCmd = _vlanToCmds( structuredFilter.vlanId, matchType )
   if vlanCmd:
      structuredFilterCmds[ 'vlan' ] = vlanCmd

   nexthopGroupMatchCmd = _nexthopGroupToCmds( structuredFilter )
   if nexthopGroupMatchCmd:
      nexthopGroupMatchCmd = "next-hop group" + nexthopGroupMatchCmd
      structuredFilterCmds[ 'nexthopGroup' ] = nexthopGroupMatchCmd

   matchDescCmd = _matchDescToCmds( structuredFilter )
   if matchDescCmd:
      structuredFilterCmds[ 'description' ] = matchDescCmd

   appProfileMatchCmd = _appProfileToCmds( structuredFilter )
   if appProfileMatchCmd:
      appProfileMatchCmd = f'application-profile {appProfileMatchCmd}'
      structuredFilterCmds[ 'appProfile' ] = appProfileMatchCmd

   locationMatchCmd = _locationMatchToCmds( structuredFilter )
   if locationMatchCmd:
      structuredFilterCmds[ 'location' ] = locationMatchCmd

   dstSelfCmd = _destinationSelfToCmd( structuredFilter )
   if dstSelfCmd:
      structuredFilterCmds[ 'dstSelf' ] = dstSelfCmd

   dzGreSwitchIdCmd = _numericalRangeToCmd( structuredFilter.dzGreSwitchId,
                                            'switch id', matchType )
   if dzGreSwitchIdCmd:
      structuredFilterCmds[ 'dzGreSwitchId' ] = dzGreSwitchIdCmd

   dzGrePortIdCmd = _numericalRangeToCmd( structuredFilter.dzGrePortId,
                                          'port id', matchType )
   if dzGrePortIdCmd:
      structuredFilterCmds[ 'dzGrePortId' ] = dzGrePortIdCmd

   dzGrePolicyIdCmd = _numericalRangeToCmd( structuredFilter.dzGrePolicyId,
                                            'policy id', matchType )
   if dzGrePolicyIdCmd:
      structuredFilterCmds[ 'dzGrePolicyId' ] = dzGrePolicyIdCmd

   if structuredFilter.packetTypeSet:
      packetTypeCmds = _packetTypeToCmd( structuredFilter.packetTypeSet, matchType )
      if packetTypeCmds:
         structuredFilterCmds[ 'packetTypes' ] = packetTypeCmds

   if structuredFilter.encapField and structuredFilter.encapField.encapType and \
      structuredFilter.innerMatch:
      innerCmds = {}
      if structuredFilter.innerMatch.af:
         cmd = 'encapsulation %s %s' % ( structuredFilter.encapField.encapType,
            'mac' if structuredFilter.innerMatch.af == 'ipunknown' else
                  structuredFilter.innerMatch.af )
         innerCmds[ 'encap' ] = cmd

      teidCmd = _teidToCmds( structuredFilter.innerMatch, matchType )
      if teidCmd:
         innerCmds[ 'teid' ] = teidCmd

      sourceAddrs = _addrListToStr( structuredFilter.innerMatch.source )
      if sourceAddrs:
         cmd = 'source prefix %s' % sourceAddrs
         innerCmds[ 'source' ] = cmd

      sourcePrefixSet = _addrListToStr( structuredFilter.innerMatch.srcPrefixSet )
      if sourcePrefixSet:
         cmd = 'source prefix field-set %s' % sourcePrefixSet
         innerCmds[ 'srcPrefixSet' ] = cmd

      dstAddrs = _addrListToStr( structuredFilter.innerMatch.destination )
      if dstAddrs:
         cmd = 'destination prefix %s' % dstAddrs
         innerCmds[ 'destination' ] = cmd

      dstPrefixSet = _addrListToStr( structuredFilter.innerMatch.dstPrefixSet )
      if dstPrefixSet:
         cmd = 'destination prefix field-set %s' % dstPrefixSet
         innerCmds[ 'dstPrefixSet' ] = cmd

      protoCmd = _protoToCmds( structuredFilter.innerMatch.proto, 'protocol',
                               structuredFilter.innerMatch.af, l4PortSaveAll )
      if protoCmd:
         innerCmds[ 'protocol' ] = protoCmd

      structuredFilterCmds[ 'innerMatch' ] = innerCmds

   return structuredFilterCmds
