# Copyright (c) 2005-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This module encapsulates the file location syntax used by the
# industry-standard CLI.  See AID 108 for a more complete description of URLs.
# UrlPlugins are used to extend this module with support for additional URL
# schemes and filesystems.

import collections
import errno
from functools import total_ordering
import operator
import os
import random
import string
import sys
import tempfile

import CliAaa
import CliCommon
import CliMatcher
import CliParser
import CliParserCommon
import Plugins
import Tac

entityManager_ = None

class Context:
   def __init__( self, entityManager, disableAaa, cliSession=None ):
      self.entityManager = entityManager
      self.disableAaa = disableAaa
      self.cliSession = cliSession

def urlArgsFromMode( mode ):
   return ( mode.entityManager, mode.session.disableAaa_, mode.session )

_syncFlashFilesystems = None
_syncFilesystemRoot = None
_initSessionDirectories = None

fsRootDir_ = None

def _fsRoot():
   return fsRootDir_

def _fsRootIs( root ):
   global fsRootDir_
   fsRootDir_ = root

def _getFsRoot():
   if not _fsRoot():
      _fsRootIs( os.path.abspath( os.environ.get( 'FILESYSTEM_ROOT', '/mnt' ) ) )
      if _syncFilesystemRoot:
         _syncFilesystemRoot( _fsRoot() )
   return _fsRoot()

def clearFsRoot():
   _fsRootIs( None )

def invokeSyncFlashFilesystems():
   _getFsRoot()
   if _syncFlashFilesystems is not None:
      _syncFlashFilesystems()

# 'flash:' is a special case.  There must always be an entry named 'flash:' in
# the _filesystems dict - even if for some strange reason no such directory exists
# under the filesystem root - so that homeDirectory() and localStartupConfig() are
# happy.  If that directory doesn't exist, any attempts to access the filesystem
# will fail with ENOENT or ENOTDIR, which is a reasonable enough error message for
# this unexpected situation.

def setSyncFilesystemRoot( Fn ):
   """Sets a callback when the filesystem root is known."""
   global _syncFilesystemRoot
   _syncFilesystemRoot = Fn

def setSyncFlashFilesystems( Fn ):
   """Sets the syncFlashFilesystems function."""
   global _syncFlashFilesystems
   _syncFlashFilesystems = Fn

def setInitSessionDirectories( fn ):
   global _initSessionDirectories
   _initSessionDirectories = fn

def initSessionDirectories( session ):
   if _initSessionDirectories:
      _initSessionDirectories( session )

def fsRoot():
   return _fsRoot()

def setEntityManager( entityManager ):
   global entityManager_
   for fs in filesystems():
      fs.entityManager = entityManager
   entityManager_ = entityManager

def syncfile( path ):
   if os.path.isdir( path ):
      flags = os.O_DIRECTORY
   else:
      flags = os.O_RDONLY
   fd = os.open( path, flags )
   try:
      os.fsync( fd )
   finally:
      os.close( fd )

UrlCopyHook = collections.namedtuple( 'UrlCopyHook', 'filter path hook' )

@total_ordering
class Url:
   headerPrefix_ = None

   # Allow extra hook to copy to this URL
   # format (if available): ( location, CliHook )
   #
   # location is relative path to te URl's fs.location_.
   copyHook = None

   @classmethod
   def registerCopyHook( cls, callback ):
      # subclass supporting this must initialize copyHook
      # pylint: disable=unsubscriptable-object
      cls.copyHook.hook.addExtension( callback )

   def __init__( self, fs, url ):
      # url is the string that was originally passed to parseUrl.
      self.url = url
      self.fs = fs
      self.context = None

      # Set to True if you want to skip allowd path check for filesystem restriction
      # Note: Use with caution
      self.allowAllPaths = False

   def _cmpHelper( self, other, op ):
      if isinstance( other, Url ):
         lhs = ( self.fs, self.url )
         rhs = ( other.fs, other.url )
         return op( lhs, rhs )
      else:
         return NotImplemented

   def __eq__( self, other ):
      return self._cmpHelper( other, operator.eq )

   def __ne__( self, other ):
      return not self == other

   def __lt__( self, other ):
      return self._cmpHelper( other, operator.lt )

   def __hash__( self ):
      return hash( ( self.fs, self.url ) )

   def historyEntryName( self ):
      '''If we want to track the URL's change history, return a valid name'''
      return None

   def setAllowAllPaths( self, allowAllPath ):
      self.allowAllPaths = allowAllPath

   def localFilename( self, check=True ):
      '''Return the filename of a local file that corresponds to this object.
      If there is no local file, e.g., this is an http url, then return None.
      check: when it's true, additional check is performed to make sure the filename
      can be accessed in a safe manner (e.g., stored in running-config).
      In certain cases when the check isn't desirable, pass check=False.'''
      # This is the base class implementation.  Derived classes may override.
      return None

   def size( self ):
      '''Return the size of the URL, if can be provided relatively quickly
      (e.g., remote URLs don't have it).'''
      return None

   def exists( self ):
      # This is the base class implementation.  Derived classes may override.
      return False

   def hasHeader( self ):
      # This is the base class implementation.  Derived classes may override.
      return self.headerPrefix_ is not None

   def isHeader( self, header ):
      # This is the base class implementation.  Derived classes may override.
      return self.headerPrefix_ and header.startswith( self.headerPrefix_ )

   def getHeader( self ):
      # This is the base class implementation.  Derived Classes may override.
      if self.headerPrefix_: # pylint: disable=no-else-raise
         raise NotImplementedError
      else:
         return None

   def encrypted( self ):
      # Whether the file is encripted. If the URl has localFilename(),
      # return True if the file content cannot be read directly.
      return False

   def encrypt( self, content ):
      return content

   def ignoreTrailingWhitespaceInDiff( self ):
      return False

   def writeHeaderAndRenameFile( self, srcFileName, dstFileName, dstUrl=None ):
      # If there is a custom-header to be prepended, create a tempfile,
      # write the custom-header and then append the contents of src-file.
      # Finally rename the tempfile to the src-file.
      headerStr = dstUrl.getHeader() if dstUrl else None
      if headerStr and os.path.getsize( srcFileName ):
         with open( srcFileName ) as srcFile:
            # Read the first line of the src-file and
            # match it with the expected prefix. Accordingly
            # create the first line of the startup-config.
            line = srcFile.readline()
            if dstUrl.isHeader( line ):
               line = headerStr
            else:
               line = headerStr + line

            tempfilename = self.tempfilename( dstFileName )

            # Write the first line(s) of the startup-config
            # we got from above and then copy the rest of
            # the file.
            try:
               with open( tempfilename, 'w' ) as dstFile:
                  content = line + srcFile.read()
                  dstFile.write( self.encrypt( content ) )
            except Exception: # pylint: disable=broad-except
               exc_info = sys.exc_info()
               try:
                  os.unlink( tempfilename )
               except OSError:
                  pass
               # pylint: disable=raise-missing-from
               raise exc_info[ 1 ].with_traceback( exc_info[ 2 ] )

         os.rename( tempfilename, srcFileName )

      # do an fsync to ensure the file data is synced to disk
      syncfile( srcFileName )
      os.rename( srcFileName, dstFileName )
      # sync the parent dir
      syncfile( os.path.dirname( dstFileName ) )

   def writeLocalFile( self, srcUrl, filename, filterCmd=None ):
      # This function copies the srcUrl content into a local file.
      #
      # To minimize risk when the copy operation is interrupted, it writes to
      # a temporary file, syncs the data to disk and renames it to the
      # destination file. The destination file is not affected until the final
      # rename() operation.
      #
      # Depending on the filesystem and the underlying storage device, rename()
      # may or may not be exactly atomic when the power is lost, but this is the
      # best we can do.
      self.checkOpSupported( self.fs.supportsWrite )
      tempfilename = self.tempfilename( filename )
      try:
         if self.fs.mask is not None:
            oldMask = os.umask( self.fs.mask )
         if filterCmd:
            srcUrl.getWithFilter( tempfilename, filterCmd=filterCmd )
         else:
            srcUrl.get( tempfilename )

         # If available pass the dstUrl as well in the calls below.
         dstUrl = self if self.localFilename() == filename else None
         self.fs.validateFile( tempfilename, durl=dstUrl, context=self.context )
         self.writeHeaderAndRenameFile( tempfilename, filename, dstUrl )

      except Exception: # pylint: disable=broad-except
         # remember the original exception information so we can raise again
         exc_info = sys.exc_info()
         # clean up the tempfile
         try:
            os.unlink( tempfilename )
         except OSError:
            pass
         # pylint: disable=raise-missing-from
         raise exc_info[ 1 ].with_traceback( exc_info[ 2 ] )
      finally:
         if self.fs.mask is not None:
            os.umask( oldMask ) # pylint: disable=used-before-assignment

   def copyfrom( self, srcUrl ):
      '''Read the given source url, and update this url with its contents.'''
      # Strategy #1: the destination is a local file.  Use "srcUrl.get(...)" to
      # overwrite that file with the contents of the source URL.  We check this
      # condition first to make sure that we sync the local file to disk before
      # returning.
      dstFn = self.localFilename() # pylint: disable=assignment-from-none
      if dstFn:
         self.writeLocalFile( srcUrl, dstFn )
         return
      # Strategy #2: the source is a local file.  Use "self.put(...)" to overwrite
      # the destination URL with that file.
      srcFn = srcUrl.localFilename()
      if srcFn and not srcUrl.encrypted():
         self.put( srcFn )
         return
      # Strategy #3: neither the source nor the destination is a local file.  Copy
      # via a temporary file.
      fd, filename = tempfile.mkstemp()
      try:
         os.close( fd )
         srcUrl.get( filename )
         self.put( filename )
      finally:
         try:
            os.unlink( filename )
         except OSError as e:
            # unlink() will fail in cases(incorrect password) where temp file is not
            # created, but actual failure will be masked, hence ignoring ENOENT
            if e.errno == errno.ENOENT:
               pass
            else:
               raise

   def makeLocalFile( self, ignoreENOENT=False ):
      # Make sure we have a local file holding the URL content;
      # if the URL has no local file, create a temporary file. If filename does not
      # exist, still return an empty temp file if ignoreENOENT
      self.checkOpSupported( self.fs.supportsRead )
      localFilename = self.localFilename() # pylint: disable=assignment-from-none
      if localFilename and not self.encrypted():
         # can directly access
         return localFilename
      else:
         fd, filename = tempfile.mkstemp()
         try:
            os.close( fd )
            self.get( filename )
         except Exception as e: # pylint: disable=broad-except
            if not ( isinstance( e, EnvironmentError ) and
                     e.errno == errno.ENOENT and  # pylint: disable=no-member
                     ignoreENOENT ):
               try:
                  os.unlink( filename )
               except OSError:
                  pass
               raise
         return filename

   def cleanupLocalFile( self, filename ):
      localFilename = self.localFilename() # pylint: disable=assignment-from-none
      if filename and not ( localFilename and
                            localFilename == filename ):
         try:
            os.unlink( filename )
         except OSError:
            pass

   def open( self, mode='r' ):
      '''Return a file-like object from which the contents of this URL can
      be read.'''
      filename = None
      try:
         filename = self.makeLocalFile()
         return open( filename, mode=mode )
      finally:
         self.cleanupLocalFile( filename )

   def _notSupported( self, *a, **kw ):
      raise OSError( errno.EOPNOTSUPP, os.strerror( errno.EOPNOTSUPP ) )

   def get( self, dstFn ):
      '''Fetch the contents of this Url and write it into the given local file.'''
      # All derived classes must override this, or your url is unreadable.
      self._notSupported()

   def getWithFilter( self, dstFn, filterCmd ):
      '''Fetch the contents of this Url and pipe it to filterCmd.'''
      self._notSupported()

   def put( self, srcFn, append=False ):
      '''Read the file and overwrite the contents of this Url.'''
      # All derived classes must override this, or your url is unwritable.
      self._notSupported()

   def tempfilename( self, prefix ):
      # Generate a temporary file for prefix by appending a random encoded string.
      # We do not use tempfile.mkstemp because we do not want to create the file
      # here, and os.tempnam() prints a security warning, so we implement our own
      # little function.
      vlist = string.ascii_letters + string.digits
      # pylint: disable-next=consider-using-f-string
      return '{}.{}'.format( prefix, ''.join( random.sample( vlist, 6 ) ) )

   def verifyHash( self, hashName, mode=None, hashInitializer=None ):
      # Derived classes must override this if needed.
      self._notSupported()

   def checkOpSupported( self, supportsFunc ):
      if not supportsFunc():
         self._notSupported()

   def aaaToken( self ):
      return self.__str__()

@total_ordering
class Filesystem:
   def __init__( self, scheme, fsType, permission, hidden=False, mask=None,
                 noPathComponent=False, entityManager=None ):
      self.scheme = scheme
      self.fsType = fsType
      self.permission = permission
      self.hidden = hidden
      self.noPathComponent = noPathComponent
      self.mask = mask
      self.entityManager = entityManager

   def __repr__( self ):
      # pylint: disable-next=consider-using-f-string
      return "Filesystem( %r )" % self.scheme

   def _cmpHelper( self, other, op ):
      if isinstance( other, Filesystem ):
         return op( self.scheme, other.scheme )
      else:
         return NotImplemented

   def __eq__( self, other ):
      return self._cmpHelper( other, operator.eq )

   def __ne__( self, other ):
      return not self == other

   def __lt__( self, other ):
      return self._cmpHelper( other, operator.lt )

   def __hash__( self ):
      return hash( self.scheme )

   def stat( self ):
      return ( 0, 0 )

   def mountInfo( self ):
      return ( None, None, '-' )

   def ignoresCase( self ):
      """Filesystems that ignore case (for example, vfat/ntfs should override
      this to return True."""
      return False

   def realFileSystem( self ):
      """Filesystems that act like a real file system that supports normal
      file operations and can be used in places that require a filename."""
      return False

   def supportsListing( self ):
      """Filesystems that support listing the contents of a directory should
      override this to return True."""
      return self.realFileSystem() and 'r' in self.permission

   def supportsRename( self ):
      """Filesystems that support renaming files should override this to
      return True."""
      return self.realFileSystem() and 'w' in self.permission

   def supportsDelete( self ):
      """Filesystems that support deleting files should override this to
      return True."""
      return self.realFileSystem() and 'w' in self.permission

   def supportsMkdir( self ):
      """Filesystems that support mkdir/rmdir should override this to
      return True."""
      return self.realFileSystem() and 'w' in self.permission

   def supportsWrite( self ):
      """Filesystems that support writing to files should override this to
      return True."""
      return 'w' in self.permission

   def supportsRead( self ):
      """Filesystems that support reading from files should override this to
      return True."""
      return 'r' in self.permission

   def supportsAppend( self ):
      """Filesystems that support appending to files should override this to
      return True."""
      return self.realFileSystem() and 'w' in self.permission

   def filenameToUrl( self, filename ):
      """Filesystems that support converting from a filename to a URL should
      override this to return a string representing the URL of this filename."""
      return None

   def filenameToUrlQuality( self ):
      """In a hierarchical filesystem, there may be multiple matches to a given
      filename -> URL conversion (e.g.,
      file:/mnt/flash/.extensions/foo
      flash:/.extensions/foo
      extensions:/foo
      ).  The higher the returned Quality, the better the match."""
      return 0

   def validateFile( self, filename, durl=None, context=None ):
      """This is the base class implementation. Derived Classes may override
      to validate the file for particular filesystem. Currently, this is called
      with the temporary file before copying the file to the local filesystem.
      The Derived classes should raise EnvironmentError exception if not
      a valid file for the particular filesystem. For example: the
      certificate: and sslkey: file systems will raise exception if its
      not a valid certificate/key file."""
      pass # pylint: disable=unnecessary-pass

_filesystems = {}

def registerFilesystem( fs ):
   assert fs.scheme not in _filesystems
   _filesystems[ fs.scheme ] = fs
   if entityManager_:
      fs.entityManager = entityManager_

def unregisterFilesystem( fs ):
   assert _filesystems.get( fs.scheme ) is fs
   del _filesystems[ fs.scheme ]

def filesystemsUnsynced():
   """Returns a list of all currently registered filesystems."""
   return _filesystems

def filesystems( showHidden=True ):
   """Returns a list of all currently registered filesystems."""
   invokeSyncFlashFilesystems()
   if showHidden:
      return list( _filesystems.values() )
   else:
      return [ v for v in _filesystems.values() if not v.hidden ]

def getFilesystem( scheme ):
   """Returns the filesystem with the specified scheme, or None if no such filesystem
   exists."""
   invokeSyncFlashFilesystems()
   return _filesystems.get( scheme )

currentFilesystemNoLongerExists = object()

def parseUrl( url, context, fsFunc=None, acceptSimpleFile=True ):
   """Constructs a Url object from the given string.  Raises a ValueError if the
   scheme is not recognized, or if fsFunc does not return True when passed the
   filesystem.  Returns the special object currentFilesystemNoLongerExists if no
   scheme was specified and the current filesystem no longer exists. Also raises
   a ValueError if acceptSimpleFile is set to False and the given string does not
   have a ':' somewhere in it."""

   if ':' in url:
      # A scheme was specified.
      scheme = url[ : url.find( ':' ) + 1 ]
      fs = getFilesystem( scheme )
      if not fs:
         # pylint: disable-next=consider-using-f-string
         raise ValueError( "Unknown URL scheme: %r" % scheme )
      if fsFunc and not fsFunc( fs ):
         raise ValueError(
            # pylint: disable-next=consider-using-f-string
            "Filesystem %s does not meet the requirements" % fs )
      rest = url[ url.find( ':' ) + 1 : ]
   else:
      # If no scheme was specified, the URL is on the current filesystem.  In this
      # case, we ignore the fsTypes - we always accept relative URLs, and it is up to
      # any individual file CLI command to reject them with an appropriate error
      # message if they are not acceptable to that command.
      session = context.cliSession
      fs = session.currentDirectory().fs if session else None
      rest = url
      if not acceptSimpleFile:
         raise ValueError( "Want URL with colon" )
      if not ( fs and fs in filesystems() ):
         return currentFilesystemNoLongerExists

   return fs.parseUrl( url, rest, context )

# Filesystems that represent actual locations on the UNIX filesystem
# have an attribute "location_", which specifies the root.  For any
# registered filesystem with a location_ attribute, if the filename
# starts with the location, pick the filesystem with the longest
# prefix.
def filenameToUrl( filename, simpleFile=True, relativeTo=None ):
   filename = os.path.normpath( os.path.abspath( filename ) )
   if relativeTo is not None and filename.startswith( relativeTo ):
      return os.path.relpath( filename, relativeTo )
   bestMatch = None
   for fs in filesystems():
      if fs.filenameToUrl( filename ) is not None and (
            bestMatch is None or
            bestMatch.filenameToUrlQuality() < fs.filenameToUrlQuality() ):
         bestMatch = fs
   if bestMatch:
      return bestMatch.filenameToUrl( filename )
   else:
      assert simpleFile
      return filename

class UrlMatcher( CliMatcher.Matcher ):
   """Class used by CLI plugins to match a URL."""

   def __init__( self, fsFunc, helpdesc, notAllowed=None, acceptSimpleFile=True,
                 allowAllPaths=False, **kargs ):
      """I match a URL if the filesystem matched by the URL passes the check
      specified by fsFunc, which must be a callable that takes a single
      parameter, a Filesystem instance, and return True when passed a
      Filesystem that I should match. If acceptSimpleFile is set to False,
      then any filename without a ':' in it will lead to a ValueError.

      @notAllowed: a list of strings that no match should be a prefix of.
                   These are typically keywords that could appear in place of a URL,
                   e.g. '/recursive' in the 'dir [/recursive] <url>' command.
      @allowAllPaths: a flag to skip FileSystem allowed paths check.
                      By default, the CLI only allows certain paths in the file:/
                      (/tmp, /var/tmp, /var/log). This is to prevent CLI passes
                      file paths to privileged processe(e.g., agents), which then
                      operate on the files as a privileged user instead of the CLI
                      user itself.
                      However if you know that the file is accessed inside
                      CLI user session context(using the login user credential),
                      you can pass 'allowAllPaths=True' to access more paths
                      under file:/ with read access.
                      If you don't know what you are doing, leave it as the default.
                      Note: allowAllPaths is always False for config commands.
      """
      self.fsFunc_ = fsFunc
      self.acceptSimpleFile = acceptSimpleFile
      self.allowAllPaths = allowAllPaths
      self.notAllowed_ = notAllowed if notAllowed is not None else []

      CliMatcher.Matcher.__init__( self, helpdesc=helpdesc, **kargs )

   def __str__( self ):
      return "<url>"

   def _contextFromMode( self, mode ):
      return Context( *urlArgsFromMode( mode ) )

   def match( self, mode, context, token ):
      if any( k.startswith( token ) for k in self.notAllowed_ ):
         # We return noMatch so that the keyword is the unique match for this
         # token, which will allow this token to auto-complete to that keyword.
         return CliCommon.noMatch
      try:
         url = parseUrl( token, self._contextFromMode( mode ),
                         self.fsFunc_, self.acceptSimpleFile )
         if isinstance( url, Url ):
            aaaToken = url.aaaToken()
            url.setAllowAllPaths( self.allowAllPaths )
         else:
            aaaToken = str( url )
         return CliParserCommon.MatchResult( url, aaaToken )
      except ValueError:
         # The path contains a scheme, but either it's not a valid scheme or it's not
         # one of the schemes accepted by this command, so we reject this token.
         # Note that this means that filenames that contain colons can never be
         # entered as relative paths.  This is not a real problem as we don't
         # anticipate many filenames containing colons, and there's an easy
         # workaround (specify the filename as an absolute path, with the scheme).
         return CliCommon.noMatch

   def completions( self, mode, context, token ):
      # Note that we only show filename completions if a valid scheme has been
      # specified, not for relative paths.

      if not ':' in token:
         # The user has not finished typing a scheme yet.  Return a list of
         # completions for the scheme.
         completionsList = []
         for fs in filesystems( showHidden=False ):
            if fs.scheme.startswith( token ) and self.fsFunc_( fs ):
               completionsList.append(
                  CliParser.Completion( name=fs.scheme,
                                        help=self.helpdesc_,
                                        partial=( not fs.noPathComponent ) ) )
      else:
         try:
            url = parseUrl( token, self._contextFromMode( mode ),
                            self.fsFunc_, self.acceptSimpleFile )
            # The user has typed a scheme.  Return a list of filename completions.
            if hasattr( url, 'getCompletions' ):
               # Completion reveals directory content so we need to check AAA.
               # This is a bit unusual but I guess there is no other way. We
               # just send "dir <url>" and see if it's allowed.
               if mode.session_.authorizationEnabled():
                  ok, _ = CliAaa.authorizeCommand( mode,
                                                   mode.session_.privLevel_,
                                                   [ 'dir', url.url ] )
                  if not ok:
                     return []
               completionsList = url.getCompletions()
            else:
               completionsList = [
                  CliParser.Completion( name='A URL beginning with this prefix',
                                        help='', literal=False ) ]
         except ValueError:
            # The user has typed a scheme, but it's either not a valid scheme or it's
            # not one of the schemes accepted by this command.
            completionsList = []

      return completionsList

# The current UrlPlugin approach would need to be extended to support moving
# the flash filesystem code into a plugin because there is no way to register
# a callback to invoke _syncFlashFilesystems() at the appropriate times.  It
# is straightforward to extend the UrlPlugin mechanism to do this if/when we
# develop a UrlPlugin that needs it.
Plugins.loadPlugins( "UrlPlugin" )
