#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This module implements the "checkpoint" URL scheme.
"""

import errno
import os

import GitLib
import Tac
import Url
from UrlPlugin.FlashUrl import LocalFilesystem, LocalUrl

class CheckpointUrl( LocalUrl ):
   def get( self, dstFn ):
      if self.isdir():
         raise OSError( errno.EISDIR, os.strerror( errno.EISDIR ) )

      commitHash = self.pathname[ 1: ]
      contents = GitLib.gitShow( self.context.entityManager.sysname(),
            commitHash, GitLib.CONFIG_FILE_NAME )
      if contents is None:
         raise OSError( errno.ENOENT, os.strerror( errno.ENOENT ) )

      with open( dstFn, 'w' ) as dstFd:
         dstFd.write( contents )

   def listdir( self ):
      if self.pathname != '/':
         return []
      results = []
      for commit in GitLib.getGitCommits( self.context.entityManager.sysname() ):
         results.append( commit[ 'commitHash' ] )
         results.extend( commit[ 'tags' ] ) # add all tags
      return results

   def exists( self ):
      if self.pathname == '/':
         return True

      commitHash = self.pathname[ 1: ]
      commits = GitLib.getGitCommits( self.context.entityManager.sysname(),
            commitHash=commitHash )
      if not commits:
         raise OSError( errno.ENOENT, os.strerror( errno.ENOENT ) )
      return True

   def isdir( self ):
      return self.pathname == '/'

   def islink( self ):
      return False

   def size( self ):
      if self.isdir():
         return 0
      commitHash = self.pathname[ 1: ]
      return GitLib.gitCatFileSize( self.context.entityManager.sysname(),
            commitHash, GitLib.CONFIG_FILE_NAME )

   def date( self ):
      if self.isdir():
         return 0
      commitHash = self.pathname[ 1: ]
      commits = GitLib.getGitCommits( self.context.entityManager.sysname(),
            commitHash=commitHash )
      if not commits:
         raise OSError( errno.ENOENT, os.strerror( errno.ENOENT ) )
      return commits[ 0 ][ 'commitTime' ]

   def permission( self ):
      if self.isdir():
         return ( True, True, True, True )
      return ( False, True, False, False )

   def ignoreTrailingWhitespaceInDiff( self ):
      return True

class CheckpointFileSystem( LocalFilesystem ):
   urlClass_ = CheckpointUrl

   def __init__( self, scheme ):
      LocalFilesystem.__init__( self, scheme, 'system', 'r' )

   def supportsListing( self ):
      return True
         
def Plugin( context=None ):
   Url.registerFilesystem( CheckpointFileSystem( 'checkpoint:' ) )
