#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
import io
import os
import shutil
import time

import Ark
import CliExtensions
import FileUrlDefs
import SecMonUtil
import Url
from UrlPlugin.FlashUrl import FlashUrl, FlashFilesystem
import UtmpDump

# hook to print warning when someone is copying a file directly to startup-config
copySourceWarningHook = CliExtensions.CliHook()

def startupConfigCopyHookFilter( mode, surl, durl ):
   # only invoke copy hooks when we are copying from running-config
   if ( ( surl.fs.scheme == "system:" and
          surl.pathname.startswith( "/running-config" ) ) or
        # CVP does "copy session-config startup-config"
        surl.fs.scheme == "session:" ):
      return True

   # someone is copying non running/session-config to startup-config, warn as it
   # may cause inconsistency.
   for callback in copySourceWarningHook.extensions():
      callback( mode, surl, durl )
   return False

class StartupConfigUrl( FlashUrl ):
   headerPrefix_ = '! Startup-config last modified at '
   copyHook = Url.UrlCopyHook( startupConfigCopyHookFilter,
                               FileUrlDefs.MANAGED_CONFIG_DIR,
                               CliExtensions.CliHook() )

   def historyEntryName( self ):
      return "startup"

   def getHeader( self ):
      # Return string formatted with save time and user
      # that will be inserted in the startup-config.
      info = UtmpDump.getUserInfo()
      saveTime = time.asctime( time.localtime() )
      # pylint: disable-next=consider-using-f-string
      return '{}{} by {}\n'.format( self.headerPrefix_, saveTime, info[ 'user' ] )

   def ignoreTrailingWhitespaceInDiff( self ):
      return True

   def empty( self ):
      if self.copyHook:
         copyDir = os.path.join( self.fs.location_,
                                 self.copyHook.path )
         shutil.rmtree( copyDir, ignore_errors=True )

      FlashUrl.empty( self )

   # BUG136154: Parallel runs of `copy run start` cause internal error. Sync.
   writeLocalFile = Ark.synchronized()( FlashUrl.writeLocalFile )

class SecureMonitorStartupConfigUrl( StartupConfigUrl ):
   headerPrefix_ = '! Secure-monitor startup-config last modified at '
   copyHook = None # no copyHook

   def encrypted( self ):
      return True

   def encrypt( self, content ):
      return SecMonUtil.encryptfile( content )

   def open( self, mode='rt' ):
      if set( mode ) - set( 'rbt' ):
         raise ValueError( 'url may only be opened for reading' )
      text = 'b' not in mode
      self._checkSecureMonitor()
      content = SecMonUtil.readfile( self.localFilename(), text=text )
      return ( io.StringIO if text else io.BytesIO )( content )

   def _checkSecureMonitor( self ):
      if not ( self.context.cliSession and self.context.cliSession.secureMonitor() ):
         if ( self.context.entityManager is None and self.context.disableAaa and
              not self.context.cliSession ):
            # this is LoadConfig
            return
         raise OSError( errno.EPERM, os.strerror( errno.EPERM ) )

   def checkOpSupported( self, supportsFunc ):
      # piggyback on this function
      StartupConfigUrl.checkOpSupported( self, supportsFunc )
      self._checkSecureMonitor()

   def get( self, dstFn ):
      self._checkSecureMonitor()
      content = SecMonUtil.readfile( self.localFilename() )
      with open( dstFn, "w" ) as f:
         f.write( content )

   # Don't allow direct writing - only copy command is allowed
   # which will use writeLocalFile().
   put = StartupConfigUrl._notSupported

def Plugin( context=None ):
   FlashFilesystem.registerClass(
      FileUrlDefs.STARTUP_CONFIG_FILE_NAME,
      StartupConfigUrl )
   FlashFilesystem.registerClass(
      FileUrlDefs.SECURE_MONITOR_STARTUP_CONFIG_FILE_NAME,
      SecureMonitorStartupConfigUrl )
