# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This module contains utility functions for querying the contents
# of the utmp file, /var/run/utmp, which contains information login
# records on the current system.  It provides the same information
# as a 'who' command, along with other login related information.

import TacUtils
import Tracing
import datetime
import os
import re
import pwd
import sys
import time
import argparse
import json

# Set trace flags to TRACE=Cli::UtmpDump/*
t0 = Tracing.trace0        # Basic tracing
t1 = Tracing.trace1        # Details

UTMP_TYPE_UNKNOWN = "0"
UTMP_TYPE_RUN_LEVEL = "1"
UTMP_TYPE_BOOT_TIME = "2"
UTMP_TYPE_NEW_TIME = "3"
UTMP_TYPE_OLD_TIME = "4"
UTMP_TYPE_INIT_PROCESS = "5"
UTMP_TYPE_LOGIN_PROCESS = "6"
UTMP_TYPE_USER_PROCESS = "7"
UTMP_TYPE_DEAD_PROCESS = "8"

def setEnvDefaultUtmpFile( utmpFileName ):
   os.environ[ "DEFAULT_UTMP_FILE" ] = utmpFileName

def getDefaultUtmpFileName( utmpFileName ):
   if not utmpFileName:
      utmpFileName = os.environ.get( "DEFAULT_UTMP_FILE", "/var/run/utmp" )
   return utmpFileName

def utmpDump( utmpFileName=None, asJson=False ):
   """Dump the utmp data to stdout."""
   utmpFileName = getDefaultUtmpFileName( utmpFileName )
   data = getUtmpData( utmpFileName )
   if asJson:
      json.dump( data, sys.stdout )
   else:
      for entry in data:
         try:
            print( "[%s] [%s] [%s] [%s]" # pylint: disable=consider-using-f-string
                  " [%s] [%s] [%s] [%s] [%s]" % (
                  entry[ "type" ], entry[ "pid" ],
                  entry[ "tty" ], entry[ "tty4" ],
                  entry[ "user" ], entry[ "host" ],
                  entry[ "ipAddr" ], entry[ "time" ],
                  entry[ "idle" ] ) )
         except KeyError:
            pass

def parseTime( timeStr ):
   # parse the time string in Utmp (example: 2008-12-02T02:19:25,000000+00:00)
   #
   # It seems utmpdump always returns +00:00 (UTC)
   return datetime.datetime.strptime( timeStr,
                                      "%Y-%m-%dT%H:%M:%S,%f+00:00" )

def ttyLastAccessTime( ttyName ):
   if not ttyName.startswith( "/" ):
      ttyName = "/dev/%s" % ttyName # pylint: disable=consider-using-f-string
   try:
      return os.stat( ttyName ).st_atime
   except OSError:
      return None

def sanitizeTtyName( ttyName ):
   """Do some name mangling to get industry-standard tty names"""
   # Strip off a leading '/dev/', if any
   ttyName = ttyName.replace( '/dev/', '' )
   ttyName = ttyName.replace( 'pts/', 'vty' ).replace( 'ttyS', 'con' )
   return ttyName

def getTtyName( ):
   ttyName = os.getenv( "REALTTY" )
   if not ttyName:
      for fd in ( 0, 1, 2 ):
         try:
            ttyName = os.ttyname( fd )
            break
         except OSError:
            pass
   if ttyName is None:
      return ''
   else:
      return sanitizeTtyName( ttyName )

def getTtyLoginEntry( ttyName, utmpFileName=None ):
   """Get the utmp login entry for the specified ttyName
   If the specified ttyName is found in the utmp file, it
   returns a single utmp dictionary entry.
   See getUtmpData() for details of the dictionary entry.
   Returns None if ttyName not found in utmp data
   """
   assert ttyName
   utmpFileName = getDefaultUtmpFileName( utmpFileName )

   ttyName = sanitizeTtyName( ttyName )
   data = getUtmpData( utmpFileName )
   if data:
      for entry in data:
         try:
            if ( entry[ "type" ] == UTMP_TYPE_USER_PROCESS and
                 entry[ "tty" ] == ttyName ):
               return entry
         except KeyError:
            pass
   return None

def getUtmpData( utmpFileName=None ):
   """Reads the specified utmp file.
   Returns a list of dictionaries, one for each utmp entry.
   All dictionary keys and values are strings
   Values are right padded with spaces and may contain all
   spaces if that utmp field is empty.
   Dictionary keys:
      "type":     See UTMP_TYPE_* above
      "pid":      Process ID as a string
      "tty":      TTY (line) name - device name of tty w/o "/dev/"
      "tty4":     4 char abbreivated TTY (line) name
      "user":     User ID
      "host":     Hostname for remote login,
                  kernel release for Run Level and Boot Time
      "ipAddr":   IP Address
      "time":     Time and date entry was made
      "idle":     Time the terminal has been idle for
   See linux docs on utmp and utmpdemp for more info.
   """
   utmpFileName = getDefaultUtmpFileName( utmpFileName )
   t0( "utmpDump:", utmpFileName )
   entries = [ ]
   regExp = re.compile(
      r"\[(?P<type>"   r"[^\]]*?)\s*\] \[(?P<pid>"  r"[^\]]*?)\s*\] "
      r"\[(?P<tty4>"   r"[^\]]*?)\s*\] \[(?P<user>" r"[^\]]*?)\s*\] "
      r"\[(?P<tty>"    r"[^\]]*?)\s*\] \[(?P<host>" r"[^\]]*?)\s*\] "
      r"\[(?P<ipAddr>" r"[^\]]*?)\s*\] \[(?P<time>" r"[^\]]*?)\s*\]" )

   if not os.path.exists( utmpFileName ):
      return []
   output = TacUtils.run( [ 'utmpdump', utmpFileName ],
                             stdout=TacUtils.CAPTURE, stderr=TacUtils.CAPTURE )
   start = time.time()
   lines = re.split( "\n", output )
   for line in lines:
      m = regExp.match( line )
      if not m:
         # Skip header and any other lines we don't recognize
         continue
      entry = m.groupdict( )

      lastAccess = ttyLastAccessTime( entry[ 'tty' ] )
      if lastAccess:
         entry[ 'idle' ] = start - lastAccess
      else:
         entry[ 'idle' ] = None

      entry[ 'tty' ] = sanitizeTtyName( entry[ 'tty' ] )
      entries.append( entry )

   return entries

def getRemoteHostFromEnv( ):
   # Get remote host IP address from SSH or TELNET environment variable
   sshConn = os.environ.get( 'SSH_CONNECTION' )
   if sshConn:
      return sshConn.split( ' ' )[ 0 ]
   else:
      # try REMOTEHOST (set by telnet)
      remoteHost = os.environ.get( 'REMOTEHOST' )
      if remoteHost:
         return remoteHost
   return None

def getUserInfo():
   # UTMP might be corrupted, so we trust the environment variables more.
   user = os.environ.get( 'LOGNAME' )
   ipAddr = getRemoteHostFromEnv()
   userTty = getTtyName()

   if userTty:
      userTty = userTty.replace( '/dev/', '' )
      if not ( user and ipAddr ):
         # Get the login entry for the current tty
         entry = getTtyLoginEntry( userTty )
         if entry:
            if not user:
               user = entry.get( 'user' )
            if not ipAddr:
               ipAddr = entry.get( 'ipAddr' )

   if not user:
      try:
         pw = pwd.getpwuid( os.getuid() )
         user = pw.pw_name
         os.environ[ 'LOGNAME' ] = user
      except KeyError:
         pass

   return {
      'user': user if user else 'UnknownUser',
      'ipAddr': ipAddr if ipAddr else 'UnknownIpAddr',
      'tty': userTty if userTty else 'UnknownTty'
      }

if __name__ == '__main__':
   parser = argparse.ArgumentParser()
   parser.add_argument( "-j", "--json", action='store_true' )
   parser.add_argument( "file", nargs='?' )
   options = parser.parse_args()
   utmpDump( options.file, options.json )
