#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

AWS = 'AWS'
AZURE = 'Azure'
GCP = 'GCP'
KVM = 'KVM'
ESXI = 'ESXi'
BAREMETAL = 'BareMetal'
CEOSR = 'cEOSR'
ALIBABA = 'Alibaba'
CARAVAN = 'Caravan'
OPENSTACK = 'Openstack'
COUNCILBLUFF = 'Arista Council Bluffs'
INDEPENDENCE = 'Independence'
WILLAMETTE = 'Willamette'
OCTEON10 = 'Marvell Octeon10 DPU'
CEOS_CLOUD_LAB = "cEOSCloudLab"

def caravanProducts_():
   dmiProductNames = [ COUNCILBLUFF, INDEPENDENCE, WILLAMETTE ]
   return ( product.lower() for product in dmiProductNames )

caravanProductList_ = caravanProducts_()

## check if EOS is running inside docker container
def runningInDocker_():
   with open( '/proc/1/cgroup' ) as ifh:
      cgroupData = ifh.read()
      return os.path.exists( '/.dockerenv' ) and \
         any( v in cgroupData for v in [ 'docker', 'lxc', 'kubepods' ] )

isDocker_ = runningInDocker_()
def runningInDocker():
   return isDocker_

def getDmiInfo( propName ):
   propPath = os.path.join( '/', 'sys', 'class', 'dmi', 'id', propName )
   if not os.path.exists( propPath ):
      return None
   try:
      with open( propPath ) as f:
         return str( f.read() ).rstrip()
   except OSError:
      # It's possible to run in namespace dut using ssh in some stest scenerio.
      # Handle gracefully as we may not have permissions to access /sys subdir.
      return None

def getDeviceTreeModel():
   try:
      with open( '/sys/firmware/devicetree/base/model' ) as f:
         return f.read()
   except OSError:
      # We might not be running on an ARM device
      return None

def getSysVendorInfo():
   return getDmiInfo( 'sys_vendor' )

def getProductUuid():
   return getDmiInfo( 'product_uuid' )

def getProductName():
   return getDmiInfo( 'product_name' )

def platformAWS():
   return getPlatform() == AWS

def platformAzure():
   return getPlatform() == AZURE

def platformGCP():
   return getPlatform() == GCP

def platformKVM():
   return getPlatform() == KVM

def platformESXi():
   return getPlatform() == ESXI

def platformBareMetal():
   return getPlatform() == BAREMETAL

def platformCeosR():
   return getPlatform() == CEOSR

def platformAlibaba():
   return getPlatform() == ALIBABA

def platformCaravan():
   return getPlatform() == CARAVAN

def platformOpenstack():
   return getPlatform() == OPENSTACK

def platformCouncilBluffs():
   return getPlatform() == CARAVAN and getProductName() == COUNCILBLUFF

def platformIndependence():
   return getPlatform() == CARAVAN and getProductName() == INDEPENDENCE

def platformWillamette():
   return getPlatform() == CARAVAN and getProductName() == WILLAMETTE

def platformHasNac():
   return platformCouncilBluffs() or platformIndependence()

def platformOcteon10():
   return getPlatform() == OCTEON10

def platformCEOSCloudLab():
   return getPlatform() == CEOS_CLOUD_LAB

# platformCloud() function returns True for cloud platforms and baremetal.
# To check only for cloud platforms use function platformCloudOnly().
def platformCloud():
   p = getPlatformCloudList() + [ BAREMETAL ]
   return getPlatform() in p

def platformCloudOnly():
   return getPlatform() in getPlatformCloudList()

def getPlatformCloudList():
   return [ AWS, AZURE, GCP, ALIBABA ]

# Case insensitive compare sysVendor/productName/Uuid,
# since productUUid in 4.19 kernel on Aws is lowercase, while
# in older kernel is uppercase.
def getPlatform_():
   # A new platform, ceoscloudlab, should be introduced, and
   # when this is done, change the check below to ceoscloudlab.
   if os.environ.get( 'EOS_PLATFORM' ) == 'ceoslab':
      return CEOS_CLOUD_LAB

   if runningInDocker():
      return CEOSR

   sysVendor = getSysVendorInfo()
   productName = getProductName()
   productUuid = getProductUuid()

   if sysVendor is not None:
      sysVendor = sysVendor.lower()

   if productName is not None:
      productName = productName.lower()

   # pylint: disable-next=consider-using-in
   if sysVendor == 'xen' or sysVendor == 'amazon ec2':
      if productUuid is not None and productUuid.lower()[ :3 ] == 'ec2':
         return AWS
   elif sysVendor == 'microsoft corporation':
      return AZURE
   # Nutanix and RedHat Openstack use KVM as its hypervisor.
   elif sysVendor == 'qemu' or productName == 'kvm' or sysVendor == 'nutanix':
      return KVM
   elif sysVendor == 'alibaba cloud':
      return ALIBABA
   elif sysVendor == 'vmware, inc.':
      return ESXI
   elif sysVendor == 'google':
      return GCP
   elif sysVendor == 'advantech':
      return BAREMETAL
   elif productName in caravanProductList_:
      return CARAVAN
   elif productName == 'openstack compute' or sysVendor == 'red hat':
      return OPENSTACK

   dtModel = getDeviceTreeModel()
   if dtModel is not None:
      dtModel = dtModel.rstrip( '\x00' )

   if dtModel == 'Marvell CN106XX board':
      return OCTEON10

   return None

platform_ = getPlatform_()
def getPlatform():
   return os.environ.get( 'VEOS_SIM_ENVIRONMENT', platform_ )
