# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import MultiRangeRule
# pylint: disable-next=unused-import
from MultiRangeRule import setFromCanonicalString as computeVlanRangeSet

def vlanSetToCanonicalString( vlanSet ):
   """This takes a set of VLAN IDs and yields a string of the form
   '1-3,5,7-9,100' that describes those VLAN IDs."""
   return MultiRangeRule.multiRangeToCanonicalString( vlanSet )

def vlanSetToCanonicalStringGen( vlanSet, strLimit=1000000 ):
   """This is a generator that takes a set of VLAN IDs and yields a sequence of
   strings of the form '1-3,5,7-9,100' that together describe those VLAN
   IDs, where each string is no longer than strLimit characters.  strLimit
   defaults to a ridiculously high value so that we default to returning a
   single string."""
   return MultiRangeRule.multiRangeToCanonicalStringGen( vlanSet, strLimit )

#------------------------------------------------------------------------------
# Class for holding and manipulating a list of VLAN ID ranges
#------------------------------------------------------------------------------
class VlanIdRangeList:
   def __init__ ( self, initialList=None, initialString=None ):
      if initialList:
         self.ranges = initialList
      elif initialString is not None:
         self.ranges = self._stringToRangeList( initialString )
      else:
         self.ranges = []

   def _stringToRangeList( self, rangesString ):
      ranges = []
      if not rangesString:
         return ranges
      rangeStrings = rangesString.split( ',' )
      for rangeString in rangeStrings:
         parts = rangeString.split( '-' )
         if len( parts ) == 1:
            bound = int( parts[ 0 ] )
            ranges.append( ( bound, bound ) )
         else:
            ranges.append( ( int( parts[ 0 ] ), int( parts[ 1 ] ) ) )
      return ranges

   def __str__( self ):
      # Using join here is 25-50% faster than appending onto the end of a string
      # for each range.
      return ','.join( str( lo ) if lo == hi else str( lo ) + '-' + str( hi )
                       for lo, hi in self.ranges )

   def add( self, vlanIdRanges ):
      if not vlanIdRanges:
         return
      if not self.ranges:
         self.ranges = vlanIdRanges
         return
      self.ranges.extend( vlanIdRanges )
      self._merge()

   def addVlanIdRangeList( self, rangeList ):
      self.add( rangeList.ranges )

   def _merge( self ):
      self.ranges.sort()
      rlist = self.ranges
      final = []
      prev = rlist[ 0 ]
      for r in range( 1, len( rlist ) ):
         cur = rlist[ r ]
         if prev[ 1 ] + 1 < cur[ 0 ]:
            final.append( prev )
            prev = cur
         elif prev[ 1 ] <= cur[ 1 ]:
            prev = ( prev[ 0 ], cur[ 1 ] )
      final.append( prev )
      self.ranges = final

   def remove( self, vlanIdRanges ):
      if not vlanIdRanges:
         # Nothing specified to remove
         return
      if not self.ranges:
         # There's nothing to remove from
         return
      newRanges = []
      for begin, end in self.ranges:
         # Process each existing range against the ranges to be removed.
         # A given range can be removed or split into multiple, but existing
         # ranges can't merge by removing things.
         for removeBegin, removeEnd in vlanIdRanges:
            if begin < removeBegin:
               # Current range starts before the remove range
               if end < removeBegin: # pylint: disable=no-else-break
                  # Current range is fully before remove range and all
                  # following remove ranges (they're sorted), so save it and
                  # move on to the next range.
                  break
               elif end <= removeEnd:
                  # Current range ends within the range being removed. Shrink it.
                  end = removeBegin - 1
                  break
               else:
                  # The remove range is entirely within the current range.
                  # Punch a hole and continue with the larger remaining range
                  newRanges.append( ( begin, removeBegin - 1 ) )
                  begin = removeEnd + 1
                  continue
            elif begin <= removeEnd:
               # The current range starts somewhere within the remove range
               if end <= removeEnd: # pylint: disable=no-else-break
                  # The current range is fully within the remove range. Drop it.
                  begin = None
                  break
               else:
                  # The current range starts in the remove range and ends after it.
                  begin = removeEnd + 1
                  continue
            else:
               # The current range starts after the remove range
               continue
         if begin is not None:
            newRanges.append( ( begin, end ) )
      self.ranges = newRanges

   def removeVlanRangeList( self, rangeList ):
      self.remove( rangeList.ranges )

