# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the generic YAML loader.

The loader is responsible for loading and parsing a set of YAML files in a path and
performing any neccessary actions on each of the documents defined within. These
actions can be controlled by subclassing the generic loader and overriding the
loadDocument method to hook into infrastructure provided with the subclass
constructor.

The general flow of the Loader system can be thought of as below:

                              +------------------------+
                              |       File System      |
                              +------------------------+
                              |                        |
                              |    +----------------+  |
                              |  +-----------------+|  |
                              |  |                 ||  |
                              |  |    YAML File    ||  |
                              |  |                 |+  |
                              |  +-----------------+   |
                              |                        |
                              +-----------+------------+
                                          | 2. Loader opens all
                     1. Loader called     |    YAML files in the
                        with source       |    source directory &
                        FS Path &         |    loads each document
 +-----------------+    Destination       v                           +----------+
 |                 |                 +----------+                     |          |
 |  Loader Caller  +---------------->|          +-------------------->|   Load   |
 |                 |                 |  Loader  |         4. Loader   |  Result  |
 +-----------------+          +------+          |<------+    outputs  |          |
                              |      +----------+       |    loaded   +----------+
                              | 3. Loader parses each   |    documents
                              |    document into the    |
                              |    final models using   |
                              |    a versioned parser   |
                              v                         |
                     +-----------------+        +-------+-------+
                     |                 |        |               |
                     |  YAML Document  |        |  Output Hook  |
                     |                 |        |               |
                     +--------+--------+        +---------------+
                              |                         ^
                              |       +----------+      |
                              |       |          |      |
                              +------>|  Parser  +------+
                                      |          |
                                      +----------+


Malformed definitions (as reported by the parser or loadDocument hook) will be
skipped by the loader and reported as errors. Any documents that raise the special
SkipDocument exception will be skipped without reporting an error.

The loader is able to handle multiple versions of the same defined document/object
using different versioned parsers. In order to achieve this ALL present and future
YAML files must have a "version" property defined on the first document on the file.

The version property will be of the form:
   version:
      major: <int>
      minor: <int>

This version will then be used determine which parser the loader will use. Major
versions always indicate the need for completely new parsers. Minor versions indicate
that a particular parser can parse definitions with minor versions that are less than
the parser's minor version.
'''

from abc import (
   ABCMeta,
   abstractmethod,
)
import glob
import os
import yaml

from YamlLoaderLibrary.Exceptions import (
   InternalError,
   ParsingError,
   SkipDocument,
)
from YamlLoaderLibrary.Parser import (
   ParserRegistry,
   parseVersion,
)

class LoadResult:
   '''An entity containing a summary of the results of loading the YAML documents
   with the Loader.

   The object contains two fields:
      - successes: A map from a YAML source to the successfully loaded document.
      - skipped: A map from a YAML source to the reason behind it being skipped.
      - failures: A map from a YAML source to the reason behind it not being loaded.
   Where the YAML source is the file path and the document number within the file.
   '''

   def __init__( self ):
      self.successes = {}
      self.skipped = {}
      self.failures = {}

class LoaderBase( metaclass=ABCMeta ):
   '''A generic Loader object that can be instatiated to read in YAML documents and
   parse them with the provided parsers.

   This generic object should be subclassed for the specific usecases required.
   The following methods/attributes are expected to be implemented by a subclass:
      - TraceHandle; required
      - __init__; optional
      - loadDocument; optional
   See the base definitions for details on the expected usage.
   '''

   @property
   @staticmethod
   @abstractmethod
   def TraceHandle():
      '''The trace handle that the YamlLoader should use to log its progress and any
      errors it may encounter.
      '''

   class UniqSafeLoader( yaml.SafeLoader ):
      '''An internal YAML loader that ensures that we don't have duplicate keys in
      the YAML document. This ensures that we can cleanly convert the YAML document
      into python dicts/lists without data getting lost.
      '''
      def construct_mapping( self, node, deep=False ):
         # Before we actually call construct_mapping in the yaml.SafeLoader, we will
         # traverse the node and verify that there are no duplicate properties
         properties = set()
         for nodeKey, _ in node.value:
            key = self.construct_object( nodeKey, deep=deep )
            if key in properties:
               raise yaml.MarkedYAMLError( problem=f'Duplicate key "{key}"',
                                           problem_mark=nodeKey.start_mark )
            properties.add( key )

         # If no duplicate properties are found, we can proceed with normal
         # yaml.SafeLoader's construct_mapping
         return super().construct_mapping( node, deep )

   def __init__( self, parserRegistry ):
      '''The base implementation of the YAML loader. The user should probably
      override this __init__ method to hardcode the parserRegistry to use and to take
      in any custom types for their loader to populate if necessary.

      Args:
         parserRegistry ( ParserRegistry ): The registry that contains the parsers to
                                            use for any loaded YAML files.
      '''
      if not isinstance( parserRegistry, ParserRegistry ):
         raise TypeError( 'Provided parserRegistry must be an instance of '
                          'ParserRegistry, but got {parserRegistry}' )
      self.parserRegistry = parserRegistry

      # Define the named trace handles we use below
      self.TERROR = self.TraceHandle.trace1
      self.TNOTE = self.TraceHandle.trace3
      self.TINFO3 = self.TraceHandle.trace6

   def loadDocument( self, parsedDocument ):
      '''An optional per-document hook that is provided whatever output the matched
      parser produced for the document. This function should support the outputs of
      any of parsers that the loader supports.

      Intended to provide the loader with a place to sanity check the parsed output
      and storing the parsed output somewhere. The parser can also do these jobs on
      its own instead if desired, but some loader designs might prefer to do the
      logic here instead.

      By default this function asserts that there is no provided output from the
      parser. If the parsers for the specific loader instance do provide an output
      then this function will need to be overridden to handle processing it.

      Raises:
         A ParsingError if there were any issues with a given parsed document.
         A SkipDocument if the document needs to be skipped for any reason. This can
         be thought of a gentler error condition where the document is still unusable
         on the system for some reason, but wasn't really in error.
      '''
      if parsedDocument is not None:
         raise InternalError( 'Encountered a returned parsed document '
                              f'{parsedDocument} without a loader hook specified.' )

   def loadFiles( self, loadPath ):
      '''Reads all YAML documents defined in the loadPath and determines the version
      of each. The version information will not be reported in the loaded documents.

      Returns:
         loadedDocuments ( List[ ( string, version, document ) ] ):
            A collection enumerating the source of each document, the version of that
            document, and the document itself.
         loadErrors ( List[ ( string, string ) ] ): A collection of load errors and
                                                    which file they happened on
      '''

      # We assume that if the specified path isn't a YAML file itself, it is a dir
      # that contains YAML files to look at.
      if not loadPath.endswith( '.yaml' ):
         loadPath = os.path.join( loadPath, '*.yaml' )

      yamlPaths = sorted( glob.glob( loadPath ) )
      if yamlPaths:
         yamlPathStr = '\n\t- '.join( yamlPaths )
         self.TINFO3( f'Found the following YAML files:\n\t- {yamlPathStr}' )
      else:
         self.TINFO3( f'Did not find any YAML files for "{loadPath}"' )

      loadErrors = []
      loadedDocuments = []
      for yamlPath in yamlPaths:
         if not os.path.exists( os.path.realpath( yamlPath ) ):
            loadErrors.append( ( yamlPath, f'Broken link: {yamlPath}' ) )
            continue

         with open( yamlPath ) as yamlFile:
            self.TNOTE( f'Loading YAML file: {yamlPath}' )
            try:
               yamlDocuments = list( yaml.load_all( yamlFile,
                                                    LoaderBase.UniqSafeLoader ) )
            except yaml.YAMLError as error:
               loadErrors.append( ( yamlPath, error ) )
               continue

         # NOTE: We do not want to close the door on putting multiple YAML documents
         #       into the same YAML file. We will implement the functionality to
         #       support multiple files but leave the assertion here so we can just
         #       take off this assertion when we want to support multple documents.
         if len( yamlDocuments ) != 1:
            loadErrors.append( ( yamlPath,
                                 'Only one YAML document supported, but encountered '
                                 f'{len( yamlDocuments )} instead.' ) )
            continue

         for idx, document in enumerate( yamlDocuments, 1 ):
            documentSource = f'{yamlPath}:DOC{idx}'

            # TODO: The version should probably be per file not per document, but
            #       this maintains the existing API for now.
            try:
               documentVersion = parseVersion( document )
            except ParsingError as error:
               loadErrors.append( ( documentSource, error ) )
               continue

            parser = self.parserRegistry.find( documentVersion )
            if parser is None:
               loadErrors.append(
                  ( documentSource, f'Unsupported version {documentVersion}' ) )
               continue

            loadedDocuments.append( ( documentSource, parser, document ) )

      return loadedDocuments, loadErrors

   def load( self, loadPath ):
      '''Loads the specified YAML file(s) into the system using the parsers
      registered in the provided parser registry. An additional hook is provided to
      perform further processing on each parsed document for subclasses to define if
      desired.

      The actual "loading" of the YAML files into the system (e.g. creating entities,
      etc.) must be defined in the aforementioned parsers or hook.

      Args:
         loadPath ( string ): The YAML file or the directory that contains the
                              YAML files to load.
      Returns:
         A LoadResult detailing which YAML documents were successfully loaded and
         which failed.
      '''

      self.TINFO3( f'Loading YAML files from: {loadPath}' )

      result = LoadResult()

      loadedDocuments, loadErrors = self.loadFiles( loadPath )

      for documentSource, parser, document in loadedDocuments:
         try:
            self.TINFO3( f'Parsing {documentSource} with parser version '
                         f'{parser.version}' )
            parsedDocument = parser.parse( documentSource, document )
            self.TINFO3( f'Parsed {documentSource}' )
         except SkipDocument as skip:
            # There are some conditions that aren't errors, but also not successes
            # either. We just don't report a success for these cases and continue
            # onto the next document.
            result.skipped[ documentSource ] = skip
            self.TINFO3( f'Skipped loading {documentSource}: {skip}' )
            continue
         except ParsingError as error:
            loadErrors.append( ( documentSource, error ) )
            continue

         # Allow the subclass defintion to take the parsed YAML file and turn it into
         # the actual objects we need in Sysdb
         try:
            self.loadDocument( parsedDocument )
            result.successes[ documentSource ] = document
            self.TINFO3( f'Loaded {documentSource}' )
         except SkipDocument as skip:
            # There are some conditions that aren't errors, but also not successes
            # either. We just don't report a success for these cases and continue
            # onto the next document.
            result.skipped[ documentSource ] = skip
            self.TINFO3( f'Skipped loading {documentSource}: {skip}' )
         except ( ParsingError, InternalError ) as error:
            loadErrors.append( ( documentSource, error ) )

      for documentSource, error in loadErrors:
         result.failures[ documentSource ] = error
         for line in f'Not loading {documentSource}: {error}'.splitlines():
            self.TERROR( line )

      return result
