# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from abc import (
   ABCMeta,
   abstractmethod,
)
import pprint

from YamlLoaderLibrary.Exceptions import (
   InternalError,
   ParsingError,
   SkipDocument,
)

from YamlLoaderLibrary.Versions import (
   Version,
   Registry,
)

def getValidationError( reason, document ):
   '''A convenience function to get a nicely formatted ParserError for a
   validation issue.

   Note that this *returns* the error; it doesn't raise it.
   Its up to the user to actually raise the error.
   '''
   documentFmt = pprint.pformat( document, indent=3 )
   return ParsingError( f'Document validation failed:\n{reason}',
                        yamlDocument=documentFmt )

def parseVersion( yamlDocument, pop=True ) -> Version:
   '''Parses and returns the YAML document version from a YAML document.

   Args:
      yamlDocument ( dict ):  The YAML document to parse the version for.
      pop ( bool ):           If True, the parsed version keys are removed from
                              the YAML document.

   Returns:
      The Version corresponding to the YAML document.

   Raises:
      ParsingError if the version information of the YAML document is malformed.
   '''
   if 'version' not in yamlDocument:
      raise getValidationError( 'No version specified', yamlDocument )

   versionDict = ( yamlDocument.pop( 'version' ) if pop else
                   yamlDocument[ 'version' ] )

   majorVer = versionDict.get( 'major' )
   if majorVer is None:
      raise getValidationError( 'No major version specified', yamlDocument )
   if not ( isinstance( majorVer, int ) and majorVer > 0 ):
      raise getValidationError( 'Major version must be a positive integer',
                                yamlDocument )

   minorVer = versionDict.get( 'minor' )
   if minorVer is None:
      raise getValidationError( 'No minor version specified', yamlDocument )
   if not ( isinstance( minorVer, int ) and minorVer >= 0 ):
      raise getValidationError( 'Minor version must be a non-negative integer',
                                yamlDocument )

   return Version( majorVer, minorVer )

class ParserBase( metaclass=ABCMeta ):
   '''A base class for all YAML loader parsers.

   Parsers must accept YAML documents which match the specified document validator
   and produce a single object representing that document as an output.

   Parsers can raise one of two types of exceptions:
      ParsingError: Whenever there is an error associated with the parsing of the
                    YAML document. These errors are indicative of input errors.
      SkipDocument: Whenever a YAML document should be skipped rather than loaded.
                    This can be though of a gentler error where the document isn't
                    usable but is still a valid document.
      InternalError: Whenever there is an error associated with internal parsing
                     logic. They are associated with bugs in the actual parser
                     implementation.
   '''

   @property
   @staticmethod
   @abstractmethod
   def version() -> Version:
      '''The version of YAML document that the parser can handle.'''

   @abstractmethod
   def parseDocument( self, document ):
      '''Contains the actual logic responsible for generating a model from
      an input YAML document.

      Args:
         document ( YAML dict ): The loaded YAML document to try and parse.

      Returns:
         The model for a given document. This should be whatever type will be handled
         by the Loader corresponding to this type of parser.

      Raises:
         A SkipDocument if the document needs to be skipped for any reason. This can
         be thought of a gentler error condition where the document is still unusable
         on the system for some reason, but wasn't really in error.
         A ParsingError if there were any other issues in parsing a given document.
      '''

   @abstractmethod
   def validateDocument( self, documentSource, document ):
      '''Defines the logic that we want to perform to ensure that our yaml file is
      structured properly for the rest of the parser to handle. This is provided as a
      separate API to allow the parser to use prebuilt validators if desired.

      If the file is improperly formatted, the function should raise a parsing
      exception that describes the error so that we can log it.

      Raises:
         A ParsingError describing any errors in the format of the document.
      '''

   def parse( self, documentSource, document ):
      try:
         self.validateDocument( documentSource, document )
         return self.parseDocument( document )
      except ( InternalError, ParsingError, SkipDocument ):
         raise
      except Exception as e: # pylint: disable=broad-except
         raise InternalError( 'Unhandled error detected while processing '
                              f'{documentSource}:\n{e}' ) from e

      raise InternalError( 'Reached a supposedly unreachable code path.' )

class ParserRegistry( Registry[ ParserBase ] ):
   def register( self, parser ): # pylint: disable=arguments-differ
      '''Overrides to the register function in the VersionedRegsitry to make it
      simpler to register ParserBases. This is due to ParserBases defining an
      abstract version attribute that we use for registration.'''

      if not isinstance( parser, ParserBase ):
         raise TypeError( 'Attempted to register a non-ParserBase object in the '
                          'ParserRegistry: {parser}' )
      super().register( parser.version, parser )
