# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the influence group L1 configuration loader.

The loader is responsible for loading and parsing a set of influence group l1
configuration definitions from their YAML file paths into a
ZeroTouch::L1::InfluenceGroupDefinitionDir which is specified by the caller.
'''

import Tracing

from YamlLoaderLibrary.Loader import LoaderBase
from YamlLoaderLibrary.Parser import ParserRegistry
from YamlLoaderLibrary.Exceptions import (
   ParsingError,
)

from ZeroTouchL1Lib.InfluenceGroup import (
   ParserV1,
)

class ZeroTouchInfluenceGroupLoader( LoaderBase ):
   # Set the trace handle used by the base class when loading files
   TraceHandle = Tracing.Handle( 'ZeroTouch.L1.InfluenceGroup' )

   def __init__( self, influenceGroupDefDir ):
      '''The Loader implementation for the influence groups. Takes in the dir to
      populate with the loaded influence groups.

      Args:
         influenceGroupDefDir( Tac.Dir ): The TAC directory to store influence group
                                          definitions in; we will write all results
                                          into this dir.
      '''
      self.influenceGroupDefDir = influenceGroupDefDir

      # Add the parsers that we support to a registry to pass to the parent
      parserRegistry = ParserRegistry()
      for parser in ( ParserV1.ParserV1, ):
         parserRegistry.register( parser() )
      super().__init__( parserRegistry )

   def loadDocument( self, parsedDocument ):
      '''Converts the provided document describing a set of influence groups into a
      model stored in the provided influenceGroupDefDir.

      Raises:
         ParsingError if the influence group already exists.
      '''
      if parsedDocument.name in self.influenceGroupDefDir.groupDefinition:
         # TODO BUG698933: It would be nice to check if there is a difference
         #                 between the two definitions. In such cases maybe
         #                 emitting an error trace is better.
         raise ParsingError(
            f'Influence group already exists for {parsedDocument.name}' )

      previousIntfAliases = None
      for configId, groupConfig in parsedDocument.groupConfig.items():
         groupIntfAliases = sorted( groupConfig.intfConfig.keys() )
         if not previousIntfAliases:
            previousIntfAliases = groupIntfAliases
         elif previousIntfAliases != groupIntfAliases:
            raise ParsingError( f'Group config {configId} specifies different '
                                'interface aliases than previous groups',
                                previousIntfAliases=previousIntfAliases,
                                currentIntfAliases=groupIntfAliases )

      influenceGroupDef = \
         self.influenceGroupDefDir.newGroupDefinition( parsedDocument.name )
      influenceGroupDef.copyFrom( parsedDocument )
