# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

from CliPlugin.EthIntfCli import speedLinkMode
from EthIntfLib import speedCompatTokenToEnum
from Tac import endOfTime
from TypeFuture import TacLazyType
from YamlLoaderLibrary.Exceptions import ParsingError
from YamlLoaderLibrary.Parser import ParserBase
from YamlLoaderLibrary.Versions import Version

CONFIG_TOKEN = 'configurations'
DESC_TOKEN = 'description'
NAME_TOKEN = 'name'
ROOT_TOKENS = { NAME_TOKEN, CONFIG_TOKEN, DESC_TOKEN }

# TODO: See BUG840052; We should try to handle interface ranges, for now we only
#       accept single names
SINGLE = r'(\d*)/?(\d*)'
PREFIX = r'[A-Z][A-Za-z]*'
INTF_REGEX = fr'^{PREFIX}({SINGLE})$'

INFLUENCE_GROUP_NAME_REGEX = r'^\w+$'

FIRECODE_TOKEN = 'fire-code'
REEDSOLOMON_TOKEN = 'reed-solomon'
FEC_TOKENS = { REEDSOLOMON_TOKEN, FIRECODE_TOKEN }
FEC_CMD_REGEX = ( r'^(?P<disableToken>no)?\s*error-correction encoding'
                  fr'\s*(?P<fecToken>{"|".join( FEC_TOKENS )})?$' )
SPEED_CMD_REGEX = ( r'^speed(?= \S+)(?P<autoToken> auto)?'
                    fr'(?: (?P<speedToken>{"|".join( speedLinkMode )}))?$' )
SHUT_CMD_REGEX = r'^shutdown$'
INTF_CMD_REGEXES = { FEC_CMD_REGEX, SPEED_CMD_REGEX, SHUT_CMD_REGEX, SHUT_CMD_REGEX }

LINK_WAIT_TIME_CMD_REGEX = r'^link wait time (?P<linkTimeToken>\d+)$'
SPEED_GROUP_CMD_REGEX = r'^hardware speed-group serdes (?P<speedCompatToken>\d+g)$'
GROUP_CMD_REGEXES = { LINK_WAIT_TIME_CMD_REGEX, SPEED_GROUP_CMD_REGEX }

EthFecEncodingConfigSet = TacLazyType( 'Interface::EthFecEncodingConfigSet' )
InfluenceGroupDefinition = TacLazyType( 'ZeroTouch::L1::InfluenceGroupDefinition' )
InfluenceGroupConfig = TacLazyType( 'ZeroTouch::L1::InfluenceGroupConfig' )
IntfConfig = TacLazyType( 'ZeroTouch::L1::IntfConfig' )
L1GroupSliceSm = TacLazyType( 'ZeroTouch::L1GroupSliceSm' )

class ParserV1( ParserBase ):
   version = Version( 1, 1 )

   def validateDocument( self, documentSource, document ):
      # We do our validation as part of the parsing, so just pass here
      pass

   def parseAlias( self, intfAlias ):
      if not re.match( INTF_REGEX, intfAlias ):
         raise ParsingError( 'Interface alias doesnt match supported format',
                             invalidIntfAlias=intfAlias,
                             supportedFormat=INTF_REGEX )
      return [ intfAlias ]

   def parseIntfCommand( self, intfConfig, configId, intfAlias, intfCmd ):
      if match := re.match( SPEED_CMD_REGEX, intfCmd ):
         if match.group( 'autoToken' ):
            intfConfig.autoneg = True
         if speed := match.group( 'speedToken' ):
            if speed not in speedLinkMode:
               raise ParsingError( 'Invalid speed command token',
                                   configStep=configId,
                                   invalidToken=speed )
            intfConfig.speed = speedLinkMode[ speed ]
      elif match := re.match( FEC_CMD_REGEX, intfCmd ):
         if match.group( 'disableToken' ):
            intfConfig.fec = EthFecEncodingConfigSet( fecEncodingDisabled=True )
         elif match.group( 'fecToken' ) == FIRECODE_TOKEN:
            intfConfig.fec = EthFecEncodingConfigSet( fecEncodingFireCode=True )
         elif match.group( 'fecToken' ) == REEDSOLOMON_TOKEN:
            intfConfig.fec = EthFecEncodingConfigSet( fecEncodingReedSolomon544=True,
                                                      fecEncodingReedSolomon=True )
      elif match := re.match( SHUT_CMD_REGEX, intfCmd ):
         intfConfig.enabled = False
      else:
         raise ParsingError( 'Interface command doesnt match a supported command',
                             configStep=configId,
                             intfAlias=intfAlias,
                             cmd=intfCmd,
                             supportedGroupCmds=INTF_CMD_REGEXES )

   def parseCommand( self, groupConfig, configId, configCmd ):
      if isinstance( configCmd, str ):
         if match := re.match( LINK_WAIT_TIME_CMD_REGEX, configCmd ):
            time = int( match.group( 'linkTimeToken' ) )
            groupConfig.linkUpWaitTime = time if time > 0 else endOfTime
         elif match := re.match( SPEED_GROUP_CMD_REGEX, configCmd ):
            setting = match.group( 'speedCompatToken' )
            groupConfig.speedGroupConfig = speedCompatTokenToEnum( setting )
         else:
            raise ParsingError( 'Group command doesnt match a supported command',
                                configStep=configId,
                                cmd=configCmd,
                                supportedGroupCmds=GROUP_CMD_REGEXES )

      elif isinstance( configCmd, dict ):
         for intfAlias, intfCmds in configCmd.items():
            if not isinstance( intfCmds, list ):
               raise ParsingError( 'Interface alias values must be a list of '
                                   'interface configurations to apply',
                                   configStep=configId,
                                   intfAlias=intfAlias,
                                   invalidType=type( intfCmds ) )

            intfConfig = IntfConfig()
            for intfCmd in intfCmds:
               self.parseIntfCommand( intfConfig, configId, intfAlias, intfCmd )
            if intfConfig.enabled and not intfConfig.speed:
               raise ParsingError( 'Interface aliases must at least configure a '
                                   'speed or shutdown the interface',
                                   configStep=configId,
                                   intfAlias=intfAlias )
            for intfIdAlias in self.parseAlias( intfAlias ):
               groupConfig.intfConfig[ intfIdAlias ] = intfConfig
      else:
         raise ParsingError( 'Configuration entries should be group command strings '
                             'or a map of interface alias commands',
                             configStep=configId,
                             invalidEntry=configCmd )

   def parseConfigurationStep( self, influenceGroupDef, configId, configurations ):
      if not isinstance( configurations, list ):
         raise ParsingError( 'Influence group configuration steps must be a list '
                             'of either interface alias command maps or group '
                             'configurations to apply',
                             configStep=configId,
                             invalidType=type( configurations ) )

      groupConfig = InfluenceGroupConfig()
      for configCmd in configurations:
         self.parseCommand( groupConfig, configId, configCmd )

      if not groupConfig.intfConfig:
         raise ParsingError( 'Influence group configuration steps must define at '
                             'least 1 interface alias',
                             configStep=configId )
      # We want to default to 30 seconds if no specific time is specified so that
      # we don't skip over the first configuration during iteration.
      if not groupConfig.linkUpWaitTime:
         groupConfig.linkUpWaitTime = 30
      influenceGroupDef.groupConfig[ configId ] = groupConfig

   def parseDocument( self, document ):
      if unsupportedTokens := document.keys() - ROOT_TOKENS:
         raise ParsingError( 'Influence group using unsupported token(s)',
                             unsupportedTokens=unsupportedTokens,
                             supportedTokens=ROOT_TOKENS )

      if NAME_TOKEN not in document:
         raise ParsingError( 'Influence group missing a name' )
      name = document[ NAME_TOKEN ]
      if not isinstance( name, str ):
         raise ParsingError( f'Influence group token "{NAME_TOKEN}"\'s value must '
                             'be a string',
                             invalidType=type( name ) )
      influenceGroupDef = InfluenceGroupDefinition( name )

      description = document.get( DESC_TOKEN, '' )
      if not isinstance( description, str ):
         raise ParsingError( f'Influence group token "{DESC_TOKEN}"\'s value must '
                             'be a string',
                             invalidType=type( description ) )
      influenceGroupDef.description = description

      if CONFIG_TOKEN not in document:
         raise ParsingError( 'Influence group missing a list of configurations' )
      configurationSequence = document[ CONFIG_TOKEN ]
      if not isinstance( configurationSequence, list ):
         raise ParsingError( f'Influence group token "{CONFIG_TOKEN}"\'s value must '
                             'be a list of configuration steps to perform',
                             invalidType=type( configurationSequence ) )
      if not configurationSequence:
         raise ParsingError( 'Influence groups must define at least 1 configuration '
                             'step to perform' )
      for configId, configurations in enumerate( configurationSequence ):
         self.parseConfigurationStep( influenceGroupDef, configId, configurations )

      return influenceGroupDef
