# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the sku L1 configuration loader.

The loader is responsible for loading and parsing a set of sku l1 configuration
definitions from their YAML file paths into a ZeroTouch::L1:SkuDefinitionDir which
is specified by the caller.
'''

import Tracing

from YamlLoaderLibrary.Loader import LoaderBase
from YamlLoaderLibrary.Parser import ParserRegistry
from YamlLoaderLibrary.Exceptions import (
   InternalError,
   ParsingError,
   SkipDocument,
)

from ZeroTouchL1Lib.Sku import (
   ParserV1,
)

class ZeroTouchSkuLoader( LoaderBase ):
   # Set the trace handle used by the base class when loading files
   TraceHandle = Tracing.Handle( 'ZeroTouch.L1.Sku' )

   def __init__( self, skuDefDir, influenceGroupDefDir, skuBaseName=None ):
      '''The Loader implementation for the sku definitions. Takes in both the dir to
      populate with the loaded definitions along with the valid influence groups that
      we can use.

      Note:
         The influence group definitions must be loaded prior to loading the sku
         definitions for the loader to be able to know which groups referenced in the
         sku definitions are actually valid.

      Args:
         skuDefDir( SkuDefintionDir ): The directory to store sku definitions, we
                                       will write all results into this dir.
         influenceGroupDefDir( InfluenceGroupDefinitionDir ):
                                       The directory already loaded with influence
                                       group defintions. This is needed for us to
                                       ensure the sku definitions that we try to load
                                       correctly reference existing influence groups.
         skuBaseName ( string ): The basename of the SKU to load definitions for.
                                 When this field is empty, we assume we are running
                                 on a modular system and allow all sku definitions to
                                 be loaded.
      '''
      self.skuDefDir = skuDefDir
      self.influenceGroupDefDir = influenceGroupDefDir
      self.skuBaseName = skuBaseName

      # Add the parsers that we support to a registry to pass to the parent
      parserRegistry = ParserRegistry()
      for parser in ( ParserV1.ParserV1, ):
         parserRegistry.register( parser() )
      super().__init__( parserRegistry )

   def loadDocument( self, parsedDocument ):
      '''Converts the provided document describing a sku into a model stored in the
      provided skuDefDir. 

      Raises:
         ParsingError if the definition already exists or invalid influence groups
         are referenced.
         SkipDocument if the document doesn't apply to the current skuBaseName.
      '''

      if parsedDocument.name in self.skuDefDir.skuDefinition:
         # TODO BUG698933: It would be nice to check if there is a difference
         #                 between the two definitions. In such cases maybe
         #                 emitting an error trace is better.
         raise ParsingError(
            f'Sku definition already exists for {parsedDocument.name}' )

      if self.skuBaseName and self.skuBaseName not in parsedDocument.applicability:
         raise SkipDocument( 'Sku definition not applicable to current SKU:\n'
                             f'\tSKU: {self.skuBaseName}\n\tApplicability: '
                             f'{set( parsedDocument.applicability )}' )

      skuDefinitionInfluenceGroups = { port.groupType for port in
                                       parsedDocument.portAlias.values() }
      existingInfluenceGroups = set( self.influenceGroupDefDir.groupDefinition )
      invalidInfluenceGroups = skuDefinitionInfluenceGroups - existingInfluenceGroups
      if invalidInfluenceGroups:
         raise ParsingError( 'Invalid influence groups are referenced',
                             invalidInfluenceGroups=invalidInfluenceGroups )

      # We have to do the group validation here in the loader rather than the parser
      # since we need access to the influenceGroupDefDir, but that means we have to
      # invert the logic the parser did to re-group up the port aliases.
      # First compute the type and members of each group:
      #   influenceGroups[ groupId ] = ( groupType, dict( portId: alias ) )
      influenceGroups = {}
      for portId, port in parsedDocument.portAlias.items():
         # Due to inverting the logic from the parser we have to sanity check that we
         # can actually re-assemble things
         if port.groupId not in influenceGroups:
            influenceGroups[ port.groupId ] = ( port.groupType, {} )
         elif influenceGroups[ port.groupId ][ 0 ] != port.groupType:
            raise InternalError(
                  'Inconsistent influence group type found for group',
                  groupId=port.groupId,
                  existingGroupType=influenceGroups[ port.groupId ][ 0 ],
                  conflictingPortId=portId,
                  conflictingGroupType=port.groupType )
         influenceGroups[ port.groupId ][ 1 ][ portId ] = port.alias

      missingAliases = {}
      invalidAliases = {}
      for groupId, ( groupType, groupAliases ) in influenceGroups.items():
         groupDef = self.influenceGroupDefDir.groupDefinition[ groupType ]
         # The influence group loader ensures that all configs share the same
         # interface aliases, so just grab the first one here.
         groupConfig = next( iter( groupDef.groupConfig.values() ) )
         # Alias is a per-lane interface e.g. Port/1, but we specify the aliases on a
         # per-port basis on the SKU, so strip off the lane info here.
         expectedAliases = { alias.rsplit( '/', 1 )[ 0 ] for alias in
                             groupConfig.intfConfig }
         missingAliases[ groupId ] = expectedAliases - set( groupAliases.values() )
         invalidAliases[ groupId ] = { portId: alias for portId, alias in
                                       groupAliases.items()
                                       if alias not in expectedAliases }
      if any( missingAliases.values() ) or any( invalidAliases.values() ):
         raise ParsingError( 'Invalid references to port aliases on '
                             f'{parsedDocument.name}',
                             missingAliases=missingAliases,
                             invalidAliases=invalidAliases )

      skuDefinition = self.skuDefDir.newSkuDefinition( parsedDocument.name )
      skuDefinition.copyFrom( parsedDocument )
