# Copyright (c) 2023, Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

from TypeFuture import TacLazyType
from YamlLoaderLibrary.Exceptions import ParsingError
from YamlLoaderLibrary.Parser import ParserBase
from YamlLoaderLibrary.Versions import Version

APP_TOKEN = 'applicability'
DESC_TOKEN = 'description'
GROUP_TOKEN = 'influence-groups'
NAME_TOKEN = 'name'
MOD_TOKEN = 'modifiers'
ROOT_TOKENS = { APP_TOKEN, DESC_TOKEN, GROUP_TOKEN, NAME_TOKEN, MOD_TOKEN }

PORT_TOKEN = 'ports'
TYPE_TOKEN = 'type'
GROUP_TOKENS = { PORT_TOKEN, TYPE_TOKEN }

INTF_SLOT_REGEX = r'^[A-Z][A-Za-z]*(\d*)$'
LINK_WAIT_OFFSET_TOKEN = 'link-wait-offset'

AliasDesc = TacLazyType( 'ZeroTouch::L1::PortAliasDescriptor' )
SkuDefinition = TacLazyType( 'ZeroTouch::L1::SkuDefinition' )

class ParserV1( ParserBase ):
   version = Version( 1, 2 )

   def validateDocument( self, documentSource, document ):
      # We do our validation as part of the parsing, so just pass here
      pass

   def parseModifier( self, skuDefinition, modifierToken, modifierValue ):
      if modifierToken == LINK_WAIT_OFFSET_TOKEN:
         if not isinstance( modifierValue, int ) or not 0 <= modifierValue <= 255:
            raise ParsingError( f'SKU modifier "{LINK_WAIT_OFFSET_TOKEN}"\'s value '
                                'must be an integer from 0 to 255',
                                invalidValue=modifierValue )
         skuDefinition.linkUpWaitTimeOffset = modifierValue
      else:
         raise ParsingError( 'Unknown modifier token',
                             modifier=modifierToken,
                             modifierValue=modifierValue )

   def parseApplicableSku( self, skuDefinition, skuBaseName ):
      # We can't put any formatting checks on the names since we don't have a great
      # way of defining what is possible for product names across all SKUs in EOS.
      if not isinstance( skuBaseName, str ):
         raise ParsingError( 'SKU definition applicability entries must be '
                             'strings representing the sku base name',
                             invalidType=type( skuBaseName ) )
      if skuBaseName in skuDefinition.applicability:
         raise ParsingError(
               f'"{skuBaseName}" specified multiple times in "{APP_TOKEN}".' )
      skuDefinition.applicability.add( skuBaseName )

   def parseInfluenceGroup( self, skuDefinition, groupId, influenceGroup ):
      if unsupportedTokens := influenceGroup.keys() - GROUP_TOKENS:
         raise ParsingError( 'Influence group reference using unsupported token(s)',
                             groupId=groupId,
                             unsupportedTokens=unsupportedTokens,
                             supportedTokens=GROUP_TOKENS )

      if TYPE_TOKEN not in influenceGroup:
         raise ParsingError( 'Influence group reference missing type to use',
                             groupId=groupId )
      groupType = influenceGroup[ TYPE_TOKEN ]
      if not isinstance( groupType, str ):
         raise ParsingError( f'Influence group reference token "{TYPE_TOKEN}"\'s '
                             'value must be a string',
                             groupId=groupId,
                             invalidType=type( groupType ) )

      if PORT_TOKEN not in influenceGroup:
         raise ParsingError( 'Influence group reference missing ports to use',
                             groupId=groupId )
      ports = influenceGroup[ PORT_TOKEN ]
      if not isinstance( ports, dict ):
         raise ParsingError( f'Influence group reference token "{PORT_TOKEN}"\'s '
                             'value must be a map of interface aliases to ports',
                             groupId=groupId,
                             invalidType=type( ports ) )
      if not ports:
         raise ParsingError( 'Influence group references must define at least one '
                             'interface slot alias to port mapping',
                             groupId=groupId )

      for alias, portId in ports.items():
         if not re.match( INTF_SLOT_REGEX, alias ):
            raise ParsingError( 'Interface slot alias doesnt match supported format',
                                groupId=groupId,
                                invalidSlotAlias=alias,
                                supportedFormat=INTF_SLOT_REGEX )

         if not isinstance( portId, int ) or not 1 <= portId <= 128:
            raise ParsingError( 'Port id references must be integers from 1 to 128',
                                groupId=groupId,
                                intfSlotAlias=alias,
                                invalidValue=portId )
         if conflictingAlias := skuDefinition.portAlias.get( portId ):
            raise ParsingError(
                  f'Port {portId} referenced in multiple influence groups.',
                  existingGroup=conflictingAlias.groupId,
                  newGroup=groupId )
         aliasDesc = AliasDesc( f'Group{groupId}', groupType, alias )
         skuDefinition.portAlias[ portId ] = aliasDesc

   def parseDocument( self, document ):
      if unsupportedTokens := document.keys() - ROOT_TOKENS:
         raise ParsingError( 'SKU definition using unsupported token(s)',
                             unsupportedTokens=unsupportedTokens,
                             supportedTokens=ROOT_TOKENS )

      if NAME_TOKEN not in document:
         raise ParsingError( 'SKU definition missing a name' )
      name = document[ NAME_TOKEN ]
      if not isinstance( name, str ):
         raise ParsingError( f'SKU definition token "{NAME_TOKEN}"\'s value must '
                             'be a string',
                             invalidType=type( name ) )
      skuDefinition = SkuDefinition( name )

      description = document.get( DESC_TOKEN, '' )
      if not isinstance( description, str ):
         raise ParsingError( f'SKU definition token "{DESC_TOKEN}"\'s value must be '
                             'a string',
                             invalidType=type( description ) )
      skuDefinition.description = description

      modifiers = document.get( MOD_TOKEN, {} )
      if not isinstance( modifiers, dict ):
         raise ParsingError( f'SKU definition token "{MOD_TOKEN}"\'s value must be '
                             'a map of modifiers to apply to the SKU',
                             invalidType=type( modifiers ) )
      for modifierToken, modifierValue in modifiers.items():
         self.parseModifier( skuDefinition, modifierToken, modifierValue )

      if APP_TOKEN not in document:
         raise ParsingError( 'SKU definition missing a list of applicable SKUs' )
      applicability = document[ APP_TOKEN ]
      if not isinstance( applicability, list ):
         raise ParsingError( f'SKU definition token "{APP_TOKEN}"\'s value must '
                             'be a list of applicable SKUs for this definition',
                             invalidType=type( applicability ) )
      if not applicability:
         raise ParsingError( 'SKU definition must define some applicable SKUs' )
      for skuBaseName in applicability:
         self.parseApplicableSku( skuDefinition, skuBaseName )

      if GROUP_TOKEN not in document:
         raise ParsingError( 'SKU definition missing a list of influence group '
                             'references that make up the SKU' )
      influenceGroups = document[ GROUP_TOKEN ]
      if not isinstance( influenceGroups, list ):
         raise ParsingError( f'SKU definition token "{GROUP_TOKEN}"\'s value must '
                             'be a list of influence group references',
                             invalidType=type( influenceGroups ) )
      for idx, influenceGroup in enumerate( influenceGroups ):
         self.parseInfluenceGroup( skuDefinition, idx, influenceGroup )

      return skuDefinition
