# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BothTrace
import Tac
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( "ZeroTouch.L1" )
TINFO1 = BothTrace.trace4
TVERBOSE = BothTrace.trace7

EntityMibInitStatus = TacLazyType( 'EntityMib::Hardware::InitStatus' )
EntityMibPowerStatus = TacLazyType( 'EntityMib::Card::PowerStatus' )

class CardWatcher( Tac.Notifiee ):
   notifierTypeName = 'EntityMib::Card'

   def __init__( self, card, manager ):
      self.card = card
      self.manager = manager
      super().__init__( card )

      self.handleCard()

   @property
   def isFixed( self ):
      return self.card.tag == 'FixedSystem'

   @property
   def sliceId( self ):
      if self.isFixed:
         # FixedSystems only use the tag and ignore the label
         return self.card.tag
      elif self.card.hasUplink:
         # Uplink cards have a different name than what the tag implies
         return f'Linecard{self.card.label}'
      return self.card.tag + self.card.label

   @property
   def cardReady( self ):
      if self.card.initStatus != EntityMibInitStatus.ok:
         TINFO1( 'Card in slot', self.card.relPos, 'not initialized' )
         return False

      if self.card.modelName == "Unknown":
         TINFO1( 'Card in slot', self.card.relPos, 'has no model name' )
         return False

      if self.card.hasUplink:
         if self.card.uplinkPowerStatus != EntityMibPowerStatus.poweredOn:
            TINFO1( 'Uplink card in slot', self.card.relPos, 'is not powered on' )
            return False
      # TODO: See BUG871665; FixedSystems don't currently mark themselves as powered
      #       on, so we have to ignore that part of the status here.
      elif not self.isFixed:
         if self.card.powerStatus != EntityMibPowerStatus.poweredOn:
            TINFO1( 'Card in slot', self.card.relPos, 'is not powered on' )
            return False

      # By default the card is ready if nothing is stopping it
      return True

   @Tac.handler( 'powerStatus' )
   def handlePowerStatus( self ):
      TVERBOSE( 'powerStatus changed on card in slot', self.card.relPos, 'to',
                self.card.powerStatus )
      self.handleCard()

   @Tac.handler( 'uplinkPowerStatus' )
   def handleUplinkPowerStatus( self ):
      TVERBOSE( 'uplinkPowerStatus changed on card in slot', self.card.relPos, 'to',
                self.card.uplinkPowerStatus )
      self.handleCard()

   @Tac.handler( 'hasUplink' )
   def handleHasUplink( self ):
      TVERBOSE( 'hasUplink changed on card in slot', self.card.relPos, 'to',
                self.card.hasUplink )
      self.handleCard()

   @Tac.handler( 'modelName' )
   def handleModelName( self ):
      TVERBOSE( 'modelName changed on card in slot', self.card.relPos, 'to',
                self.card.modelName )
      self.handleCard()

   # TODO: See BUG871665; FixedSystems don't currently mark themselves as powered on,
   #       so we have to react to initStatus instead for these systems. We should be
   #       able to remove this reactor once this issue is resolved.
   @Tac.handler( 'initStatus' )
   def handleInitStatus( self ):
      # Only want to use this attribute to track state for FixedSystems
      if not self.isFixed:
         return
      TVERBOSE( 'initStatus changed on card in slot', self.card.relPos, 'to',
                self.card.initStatus )
      self.handleCard()

   def handleCard( self ):
      # Need to try in case the manager has already gone away somehow
      try:
         if self.cardReady:
            TINFO1( 'Card in slot', self.card.relPos, 'ready' )
            self.manager.createL1InfluenceGroupsForCard( self.sliceId )
         else:
            TINFO1( 'Card in slot', self.card.relPos, 'not ready' )
            self.manager.clearL1InfluenceGroupsForCard( self.sliceId )
      except ReferenceError:
         pass # Can't do anything up if the manager is gone

   def __del__( self ):
      # Need to clear in  __del__ since this watcher gets deleted when the card does
      # We can get deleted after the manager, so make sure we handle the reference
      # potentially being invalid
      try:
         self.manager.clearL1InfluenceGroupsForCard( self.sliceId )
      except ReferenceError:
         pass # No need to clean anything up if the manger is gone

class CardSlotWatcher( Tac.Notifiee ):
   notifierTypeName = 'EntityMib::CardSlot'

   def __init__( self, cardSlot, manager ):
      self.cardSlot = cardSlot
      self.manager = manager
      super().__init__( cardSlot )

      self.watcher = None

      self.handleCardSlot()

   @Tac.handler( 'card' )
   def handleCardSlot( self ):
      if self.cardSlot.card:
         TINFO1( 'Card inserted to slot', self.cardSlot.relPos )
         self.watcher = CardWatcher( self.cardSlot.card, self.manager )
      elif self.watcher:
         TINFO1( 'Card removed from slot', self.cardSlot.relPos )
         self.watcher = None

class CardSlotDirWatcher( Tac.Notifiee ):
   # Accept any type to appease the infra; this should be overriden by the subclass
   notifierTypeName = '*'

   def __init__( self, parentStatus, manager ):
      self.parentStatus = parentStatus
      self.manager = manager
      super().__init__( parentStatus )

      self.watchers = {}

      for relPos in parentStatus.cardSlot:
         self.handleCardSlot( relPos )

   @Tac.handler( 'cardSlot' )
   def handleCardSlot( self, relPos ):
      TVERBOSE( "Processing slot with relPos", relPos )
      if cardSlot := self.notifier_.cardSlot.get( relPos ):
         self.watchers[ relPos ] = CardSlotWatcher( cardSlot, self.manager )
      elif relPos in self.watchers:
         del self.watchers[ relPos ]

class FixedSlotWatcher( CardSlotDirWatcher ):
   notifierTypeName = 'EntityMib::FixedSystem'

class ChassisWatcher( CardSlotDirWatcher ):
   notifierTypeName = 'EntityMib::Chassis'

class EntMibWatcher( Tac.Notifiee ):
   notifierTypeName = 'EntityMib::Status'

   def __init__( self, entMib, manager ):
      self.entMib = entMib
      self.manager = manager
      super().__init__( entMib )

      self.cardSlotWatcher = None
      self.fixedWatcher = None

      if self.entMib.fixedSystem:
         self.handleFixedSystem()
      elif self.entMib.chassis:
         self.handleChassis()

   @Tac.handler( 'fixedSystem' )
   def handleFixedSystem( self ):
      TVERBOSE( "Processing fixed system" )
      if not self.entMib.fixedSystem:
         return

      self.fixedWatcher = CardWatcher( self.entMib.fixedSystem, self.manager )
      self.cardSlotWatcher = FixedSlotWatcher( self.entMib.fixedSystem,
                                               self.manager )

   @Tac.handler( 'chassis' )
   def handleChassis( self ):
      TVERBOSE( "Processing chassis" )
      if not self.entMib.chassis:
         return

      self.cardSlotWatcher = ChassisWatcher( self.entMib.chassis, self.manager )
