# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import os
import sys
import sysconfig
import traceback
import weakref

# add Arista specific stack summary before
def makesummary( hook=None ):
   # keep local refs, so when interpreter is shutting down we can output
   local_default = sys.__excepthook__
   local_print = print
   local_traceback = traceback
   local_stderr = sys.stderr
   stdlib = sysconfig.get_path( 'stdlib' ) + '/'
   pylib = sysconfig.get_path( 'purelib' ).removesuffix( 'site-packages' )

   def arsummary( type_, value, traceback_, hook=hook ):
      try:
         local_print(
            'Stack Summary: ',
            ' '.join(
               f'{t[0].removeprefix( stdlib ).removeprefix( pylib )}:{t[2]}'
               for t in local_traceback.extract_tb( traceback_ ) ),
            f'\n{getattr( type_, "__qualname__", type_.__name__ )}: {value}',
            file=local_stderr, sep='' )
      except Exception as e:  # pylint: disable=broad-except
         local_print( 'Exception generating stack summary:', e )
      return ( hook or local_default )( type_, value, traceback_ )
   return arsummary

def setHook( firstTime=False ):
   if ( not sys
        or sys.excepthook is None
        or ( sys.excepthook is sys.__excepthook__ and not firstTime )
        or 'arsummary' in sys.excepthook.__name__
        or 'AR_NOEXCEPTHOOK' in os.environ ):
      return
   else:
      newhook = makesummary( sys.excepthook )
      weakref.finalize( newhook, setHook )
      sys.excepthook = newhook
      del newhook

      if firstTime and 'threading' in sys.modules:
         # threading excepthook creates a closure, so reset it to caputre
         # our version
         threading = __import__( 'threading' )
         # pylint:disable=protected-access
         mt = threading._main_thread
         mt._invoke_excepthook = threading._make_invoke_excepthook()

# set hook only for system python interpreters
if sys.executable.startswith( ( '/usr/bin', '/bin' ) ):
   setHook( True )
