# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, print_function

import argparse
import logging
import sys

from ztn.net import get_neighbours
import ztn.settings
from ztn.transaction import ZtnmTransactionManager

logging.basicConfig()
logger = logging.getLogger('ztn')
logger.setLevel(logging.INFO)


def cmd_discover(args):
    obj = ZtnmTransactionManager(logger, None, None, None)
    if args.type == 'mdns':
        rv = obj.discover_controllers_mdns()
    elif args.type == 'peer':
        rv = obj.discover_controllers_peer()
    else:
        raise ValueError("unhandled discover option %s" % args.type)

    sys.exit(len(rv) == 0)



def cmd_transact(args):
    obj = ZtnmTransactionManager(logger, None, ztn.settings.ZTNM_MANAGEMENT_INTERFACE,
                                 args.platform)

    if args.static:
        # At least one static address was specified.
        obj.add_static_addresses(args.static)
    else:
        # No static addresses specified.
        # As a convenience both mdns and peer
        # are enabled if neither were specified on the command line.
        if not (args.mdns or args.peer):
            args.peer = True
            args.mdns = True

    (result, _, _) = obj.transact(1, mdns=args.mdns, peer=args.peer)

    sys.exit(result is None)



def cmd_peers(args):
    ## Get all local v4 and v6 neighbors
    neighbors = get_neighbours(ztn.settings.ZTNM_MANAGEMENT_INTERFACE, args.wait)
    peers = neighbors['v6'] + neighbors['v4']

    if len(peers) == 0:
        logger.info("No peers.")
    else:
        for address in peers:
            print(address)

    sys.exit(len(peers) == 0)



def main():

    ap = argparse.ArgumentParser('ztn',
                                 description='Ztn manifest transaction tool.')

    sp = ap.add_subparsers()

    # discover subcommand
    ap_discover = sp.add_parser('discover')
    ap_discover.add_argument('type',
                             choices=['mdns', 'peer'],
                             help='Run mdns or peer controller discovery.')
    ap_discover.set_defaults(func=cmd_discover)


    # transact subcommand
    ap_transact = sp.add_parser('transact')
    ap_transact.add_argument('platform',
                             help='The platform name for this transaction.')

    ap_transact.add_argument('--static',
                             nargs='+',
                             action='store')
    ap_transact.add_argument('--mdns', action='store_true')
    ap_transact.add_argument('--peer', action='store_true')
    ap_transact.set_defaults(func=cmd_transact)


    # peers subcommand
    ap_peers = sp.add_parser('peers',
                             help='Show peer addresses')
    ap_peers.add_argument("--wait",
                          help='Time to wait for peer discovery',
                          type=int,
                          default=5)
    ap_peers.set_defaults(func=cmd_peers)

    ops = ap.parse_args()
    ops.func(ops)



if __name__ == '__main__':
    main()
