# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


"""Discover ZTN Controllers through neighbor interrogation."""

from __future__ import absolute_import

import logging

from ztn.controllers import ZtnmController
from ztn.net.http import HttpRequestThread
from ztn.net import get_neighbours
import ztn.settings


class PeerDiscovery:

    def __init__(self, interface_name,
                 neighbor_wait=5,
                 controller_wait=5,
                 log=None):
        self.log = log or logging.getLogger(self.__class__.__name__)
        self.interface_name = interface_name
        self.neighbor_wait = neighbor_wait
        self.controller_wait = controller_wait
        self.controllers = []

    @staticmethod
    def _discover_url(address):
        if ':' in address:
            address = "[" + address + "]"
        return "http://%s/onie-installer-discover" % address

    def run(self):

        ## Get all local v4 and v6 neighbors
        neighbors = get_neighbours(self.interface_name, self.neighbor_wait)

        self.log.info("found %d IPv6 and %d IPv4 neighbor(s)",
                      len(neighbors['v6']), len(neighbors['v4']))

        requests = []
        for address in neighbors['v4'] + neighbors['v6']:
            requests.append(HttpRequestThread(address,
                                              self._discover_url(address),
                                              timeout=ztn.settings.ZTNM_DEFAULT_HTTP_TIMEOUT_SEC,
                                              log=self.log))
        for request in requests:
            request.start()
        for request in requests:
            request.join()

        for request in requests:
            if request.response is not None:
                if request.response.status_code in [302, 404]:
                    # Just assume a possible controller
                    self.controllers.append(ZtnmController(request.address,
                                                           8843, True, 0))
            else:
                pass

        if self.controllers:
            self.log.info("\nfound %d ztn controllers via peer discoverys:\n    %s",
                          len(self.controllers),
                          "\n    ".join([str(ep) for ep in self.controllers]))
        else:
            self.log.error("\nno ztn controllers via peer discovery")


if __name__ == '__main__':
    logging.basicConfig()
    logger = logging.getLogger('discovery.peer')
    logger.setLevel(logging.INFO)

    import argparse
    ap = argparse.ArgumentParser('discovery.peer')
    ap.add_argument("interface", help='The interface on which to run discovery.')
    ops = ap.parse_args()

    PeerDiscovery(ops.interface, log=logger).run()
