# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import

import json
import logging
import threading

from urllib.parse import urlparse, urlunparse

from ztn.controllers import ZtnmController
from ztn.net.http import HttpRequestThread
from ztn.net import Interface
import ztn.settings

class ManifestRequest(threading.Thread):

    TEMPLATE = "%(protocol)s://%(target)s:%(port)d/ztn/switch/%(mac)s/switch_light_manifest"

    def __init__(self, log, interface=ztn.settings.ZTNM_MANAGEMENT_INTERFACE):
        threading.Thread.__init__(self)
        self.log = log

        # We will need this later
        self.ma = Interface(interface)

    def _create_manifest_url(self, address, protocol, port, mac_or_interface):
        if protocol not in ['http', 'https']:
            raise ValueError("invalid protocol '%s'" % protocol)

        if ':' in mac_or_interface:
            mac = mac_or_interface
        else:
            mac = Interface(mac_or_interface).mac

        if ':' in address:
            target = '[%s]' % address
        else:
            target = address

        return self._add_zone_id(self.TEMPLATE % dict(protocol=protocol,
                                    target=target,
                                    port=port,
                                    mac=mac))

    def _add_zone_id(self, url):
        result = urlparse(url)

        # pylint: disable=no-else-return
        if not result.hostname.startswith('fe80:'):
            # Not a link-local address
            return url
        elif '%' in result.hostname:
            # A zone id already exists.
            return url

        # Reconstruct the URL with a zone id added to the netloc
        return urlunparse((result.scheme,
                           "[%s%%%d]:%d" % (result.hostname, self.ma.ifindex,
                                            result.port),
                           result.path,
                           result.params,
                           result.query,
                           result.fragment))


    def _post_process_manifest(self, data):

        version = data.get('ztn-version')
        if not version:
            self.log.error('ztn-version is not present')
            return None

        if version not in ztn.settings.ZTNM_ALLOWED_VERSIONS_LIST:
            self.log.error('ztn-version=%d cannot be handled.' % version)
            return None


        # Any link-local addresses in the resulting URLs will not
        # have a zone_id. For the convenience of the client
        # we will add the correct zone_ids before returning
        # the data.
        for item in data.get('images', []) + data.get('startup-config', []):
            url = item.get('url')
            if url:
                item['url'] = self._add_zone_id(url)
        return data



    # pylint: disable=inconsistent-return-statements
    def get(self, controller, mac_or_interface, platform,
            timeout=ztn.settings.ZTNM_DEFAULT_HTTP_TIMEOUT_SEC):

        if not isinstance(controller, ZtnmController):
            controller = ZtnmController(controller)

        url = self._create_manifest_url(controller.addr,
                                        controller.protocol,
                                        controller.port, mac_or_interface)

        params = dict(platform=platform,
                      ztn_version=ztn.settings.ZTNM_VERSION)

        params.update(ztn.settings.ZTNM_MANIFEST_REQUEST_ADD_PARAMS)

        response = HttpRequestThread(controller.addr,
                                     url,
                                     timeout,
                                     self.log,
                                     params=params).get()

        if response is None:
            return None


        if response.status_code == 302:
            # The controller has returned a redirect for the manifest.
            # It appears that the controller always returns the %ma1
            # zone id as part of the redirect. This won't always be even
            # the correct interface, especially for future inband.
            #
            # I think the "right" way to solve this will be to remove
            # the zone id completely from the response and instead try the
            # address on all local links, substituting the correct zone_ids.
            #
            # There really are more implications here once there
            # are more than one interface so we'll just skip this for now.
            #
            # The requests library can only accept urls with %ifindex, not %name
            # as the zone_id, so we just hack it up here.
            #
            new_url = response.headers['location']
            if '%ma1' in new_url:
                self.log.info("%%ma1 --> %%%d" % self.ma.ifindex)
                new_url = new_url.replace('%ma1', '%' + str(self.ma.ifindex))

            response = HttpRequestThread(None,
                                         new_url,
                                         timeout,
                                         self.log).get()

            if response is None:
                self.log.error("The redirect failed.")
                return None

        if response.status_code == 200:
            try:
                data = json.loads(response.text)
                return self._post_process_manifest(data)
            except Exception as e:
                self.log.error("The content could not be parsed as json.\n  %s\n" % e)
                return None





if __name__ == '__main__':

    import argparse
    logging.basicConfig()
    logger = logging.getLogger('manifest')
    logger.setLevel(logging.INFO)

    ap = argparse.ArgumentParser('manifest')
    ap.add_argument("address",
                    help="Address against which to request the manifest.")

    ap.add_argument("platform",
                    help="platform name used in the request.")

    ap.add_argument("interface",
                    help="Local management interface.",
                    default=ztn.settings.ZTNM_MANAGEMENT_INTERFACE)
    ap.add_argument("--protocol",
                    help="The protocol to use.",
                    choices=ztn.settings.ZTNM_ALLOWED_PROTOCOLS,
                    default='https')
    ap.add_argument("--port",
                    help="The port on which to make the request.",
                    type=int,
                    default=8843)

    ops = ap.parse_args()

    ManifestRequest(logger).get(ZtnmController(ops.address, ops.port, ops.protocol, 0),
                                ops.interface,
                                ops.platform,
                                ztn.settings.ZTNM_DEFAULT_HTTP_TIMEOUT_SEC)
