# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import commands
import socket
import struct

def _show_dpi_flow_info( cli, args, modName ):
   ret = cli.bess.run_module_command( modName,
                                      'getDpiClassificationInfo',
                                      'GetDpiInfoArg', args )
   for entry in ret.info:
      ipA = entry.fk.ip_a.ip_addr
      portA = socket.ntohs( entry.fk.port_a )
      ipB = entry.fk.ip_b.ip_addr
      portB = socket.ntohs( entry.fk.port_b )
      cli.fout.write( "vrfId %d ipA %s portA %d ipB %s portB %d protocol %d "
            "appProfileId %d appId %d appServiceId %d classified %d "
            "dpiWorkerId %d dpiAppId %d dpiCategory %d dpiService %d "
            "dpiClassifierPkts %d dpiTotalPkts %d dpiTotalBytes %d "
            "dpiCtsDir %d dpiPath %s dpiFlags %s\n" %
         ( entry.fk.vrf_id, socket.inet_ntoa( struct.pack( '=L', ipA ) ), portA,
           socket.inet_ntoa( struct.pack( '=L', ipB ) ), portB, entry.fk.protocol,
           entry.app_profile_id, entry.app_id, entry.app_service_id,
           entry.classified, entry.dpi_wid,
           entry.dpi_app_id, entry.dpi_category, entry.dpi_service,
           entry.dpi_classifier_pkts, entry.dpi_total_pkts, entry.dpi_total_bytes,
           entry.dpi_cts_dir, entry.dpi_classification_path,
           entry.dpi_classification_state ) )
   return ret.next_iter

@commands.cmd( 'show dpi-flow-info',
     'Show dpi info for all flows' )
def show_dpi_flow_info( cli ):
   mod = [ m.name for m in cli.bess.list_modules().modules
               if m.mclass == 'Dpi' ]
   if len( mod ) == 0:
      return
   cli.fout.write( "%s\n" % ( mod ) )
   args = {}
   args[ 'max_iterate' ] = 100
   args[ 'iter' ] = 0
   next_iter = 1
   cli.fout.write( 'show dpi-flow-info:\n' )
   cli.fout.write(
     'DPI flag legend: c: flow classified C: classification changed\n' )
   cli.fout.write(
     '                 D: Direction changed E: Expired flow\n' )
   cli.fout.write(
     '                 f: final classification F: final classification changed\n' )
   cli.fout.write(
     '                 o: offloaded flow O: offloaded state changed\n' )
   cli.fout.write(
     '                 N: new flow P: classification path changed\n' )
   cli.fout.write(
     '                 R: Results from cache\n' )

   while next_iter != 0:
      next_iter = _show_dpi_flow_info( cli, args, mod[ 0 ] )
      args[ 'iter' ] = next_iter
   cli.fout.write( '\n' )

@commands.cmd( 'show dpi-mem-info',
     'Show memory utilization for the Dpi modules' )
def show_dpi_flow_info( cli ):
   mod = [ m.name for m in cli.bess.list_modules().modules
               if m.mclass == 'Dpi' ]
   if len( mod ) == 0:
      return
   args = {}
   modName = mod[ 0 ]
   ret = cli.bess.run_module_command( modName,
                                      'getDpiMemUsage',
                                      'EmptyArg', args )

   cli.fout.write( f"Flows active:\t\t{ret.flows_used}\n" )
   cli.fout.write( f"Flow capacity:\t\t{ret.flow_total}\n" )

   cli.fout.write( f"Admission control:\t{ret.admission_enabled}\n" )
   cli.fout.write( f"Admission delay:\t{ret.admission_secs} seconds\n" )

   cli.fout.write( f"Malloc bytes in use: \t"
                   f"{ret.malloc_total_mem//(1024*1024)} MB\n" )

   cli.fout.write( f"32 byte pool stats: \t{ret.pool_32_used} used "
                   f"{ret.pool_32_total} total\n" )
   cli.fout.write( f"64 byte pool stats: \t{ret.pool_64_used} used "
                   f"{ret.pool_64_total} total\n" )
   cli.fout.write( f"128 byte pool stats: \t{ret.pool_128_used} used "
                   f"{ret.pool_128_total} total\n" )
   cli.fout.write( f"256 byte pool stats: \t{ret.pool_256_used} used "
                   f"{ret.pool_256_total} total\n" )
   cli.fout.write( f"512 byte pool stats: \t{ret.pool_512_used} used "
                   f"{ret.pool_512_total} total\n" )
   cli.fout.write( f"768 byte pool stats: \t{ret.pool_768_used} used "
                   f"{ret.pool_768_total} total\n" )
