# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import commands
import socket
import struct
from cli_parser.option_parser import OptionParser

class ShowFragtableEntriesOptionParser( OptionParser ):

   def register_options( self ):
      self.add_option( "-max_iterate", "uint" )
      self.add_option( "-h", "action", action=self.help )

   @staticmethod
   def help():
      help_text = [
         "Usage:",
         " show ip fragment-table entries [OPTIONS]",
         "",
         "Options:",
         " -max_iterate\t\tmax number of entries to print (by default print all)",
         " -h\t\t\thelp"
         ]
      print( "\n".join( help_text ) )

   def get_max_iterate( self ):
      # default value is None, which will make all entries be printed if user does
      # not specify a -max_iterate value
      return self.get_value( "-max_iterate" )

def _show_fragtable_entries_args_preproc( cli, args ):
   cmd_args = {}
   if args.get( 'max_iterate', None ):
      cmd_args[ 'max_iterate' ] = args[ 'max_iterate' ]
   if args.get( 'iter', None ):
      cmd_args[ 'iter' ] = args[ 'iter' ]
   return cmd_args

def _show_fragtable_entries_parse_options( cli, opts ):
   if opts is None:
      opts = []
   args = {}
   try:
      parser = ShowFragtableEntriesOptionParser( opts )
      parser.parse()
      args[ 'max_iterate' ] = parser.get_max_iterate()
   # pylint: disable=broad-except
   except Exception as error:
      cli.CommandError( error )
      return None

   return args

def _show_fragtable_entries( cli, mgmtName, args ):
   cmd_args = _show_fragtable_entries_args_preproc( cli, args )
   ret = cli.bess.run_module_command( mgmtName,
                                      'getFragTableEntries',
                                      'GetFragTableEntriesArg', cmd_args )
   for entry in ret.info:
      sipStr = socket.inet_ntoa( struct.pack( '!L', entry.sip ) )
      dipStr = socket.inet_ntoa( struct.pack( '!L', entry.dip ) )
      ipId = entry.ip_id
      cli.fout.write( f'vrfId {entry.vrf_id} sip {sipStr} sport {entry.sport}'
            f' dip {dipStr} dport {entry.dport} protocol {entry.proto}'
            f' ipId {ipId} numFragsRcvd {entry.num_frags_rcvd}'
            f' leadingFragSeen {entry.leading_frag_seen}'
            f' totalSizeRcvd {entry.total_size_rcvd}'
            f' totalSize {entry.total_size}'
            f' offloadOgate {entry.offload_ogate}\n' )
   return ret.next_iter

@commands.cmd( 'show ip fragment-table entries [FRAGTABLE_CMD_OPTS...]',
     'show ip fragment-table entries: display entries in fragment table' )
def show_fragtable_entries( cli, opts ):
   mgmt = [ m.name for m in cli.bess.list_modules().modules
               if m.mclass == 'Management' ]
   if len( mgmt ) == 0:
      return
   ret = cli.bess.run_module_command( mgmt[ 0 ],
                                      'getFragTableInfo',
                                      'EmptyArg', {} )
   cli.fout.write( f'Fragment table entries: {ret.entry_count} entries\n' )
   if ret.entry_count:
      args = _show_fragtable_entries_parse_options( cli, opts )
      if args is None:
         return
      next_iter = 1
      args[ 'iter' ] = 0
      # In case a large max_iterate value is specified - let's split the
      # request into chunks.
      max_chunk_size = 100

      if args.get( 'max_iterate', None ):
         # Show exactly as many iterates as user specified... unless there aren't
         # that many.
         iters_left = args[ 'max_iterate' ]
         while next_iter != 0 and iters_left > 0:
            args[ 'max_iterate' ] = min( max_chunk_size, iters_left )
            next_iter = _show_fragtable_entries( cli, mgmt[ 0 ], args )
            args[ 'iter' ] = next_iter
            iters_left -= args[ 'max_iterate' ]
      else:
         # When max_iterate is not specified - let's show all entries.
         args[ 'max_iterate' ] = max_chunk_size
         while next_iter != 0:
            next_iter = _show_fragtable_entries( cli, mgmt[ 0 ], args )
            args[ 'iter' ] = next_iter

   cli.fout.write( '\n' )

@commands.cmd( 'show ip fragment-table entries -h',
      'show ip fragment-table entries help' )
def show_fragtable_entries_help( cli ):
   opts = [ '-h' ]
   _show_fragtable_entries_parse_options( cli, opts )

@commands.cmd( 'show ip fragment-table stats',
     'show ip fragment-table stats: display fragment-table stats' )
def show_fragtable_stats( cli ):
   mgmt = [ m.name for m in cli.bess.list_modules().modules
               if m.mclass == 'Management' ]
   if len( mgmt ) == 0:
      return
   ret = cli.bess.run_module_command( mgmt[ 0 ],
                                      'getFragTableInfo',
                                      'EmptyArg', {} )

   max_keys = "unlimited" if ret.max_keys == 0 else str( ret.max_keys )
   cli.fout.write( f'Entry Count: {ret.entry_count}\n' )
   cli.fout.write( f'Num Buckets: {ret.num_buckets}\n' )
   cli.fout.write( f'Max Keys: {max_keys}\n' )
   cli.fout.write( f'Fragment buffer limit: {ret.frag_buffer_limit}\n' )
   cli.fout.write(
         f'Num leading fragments seen: {ret.num_leading_fragments_seen}\n' )
   cli.fout.write( f'Num fragments in buffer: {ret.num_fragments_in_buffer}\n' )
   cli.fout.write( 'Num fragments sent without buffering: '
      f'{ret.num_fragments_sent_without_buffering}\n' )
   cli.fout.write( 'Num fragments sent after buffering: '
      f'{ret.num_fragments_sent_after_buffering}\n' )
   cli.fout.write( f'Num entries added: {ret.num_entries_added}\n' )
   cli.fout.write( f'Num entries deleted: {ret.num_entries_deleted}\n' )
   cli.fout.write( f'Num entries purged: {ret.num_entries_purged}\n' )
   cli.fout.write( f'Num fragments purged: {ret.num_fragments_purged}\n' )
   cli.fout.write( 'Num fragments dropped after purge: '
         f'{ret.num_fragments_dropped_after_purge}\n' )
   cli.fout.write(
         f'Num max fragments exceeded: {ret.num_max_fragments_exceeded}\n' )
   cli.fout.write(
         f'Num key insertion failures: {ret.num_key_insertion_failures}\n' )
   cli.fout.write(
         f'Num L4 info add failures: {ret.num_l4_info_add_failures}\n' )
   cli.fout.write( '\n' )

@commands.var_attrs( '[FRAGTABLE_CMD_OPTS...]' )
def frag_table_var_attrs():
   # Return (var_type(str), var_desc(str), var_candidates([str]))
   return ( 'opts',
         '[OPTIONS]( run show ip fragment-table entries -h for more help )', [] )
