#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import commands
import socket
import struct

def int2ip( intIpAddr ):
   return socket.inet_ntoa( struct.pack( '!L', intIpAddr ) )
def pBool( flag ):
   return "true" if flag else "false"
def pPathState( state ):
   return 'active' if state else 'inactive'

def get_its_sessions( its_module, cli ):
   args = {}
   ret = cli.bess.run_module_command( its_module, 'getSessStats', 'ItsDebugArg',
                                       args )
   if not ret or len( ret.its_stats ) == 0:
      raise Exception( "Could not get any ITS session stats" )
   return [ stats.session for stats in ret.its_stats ]

def its_change_config( its_module, session, cli, updateDict ):
   try:
      args = { 'src_ip' : session.src_ip,
               'dst_ip' : session.dst_ip,
               'udp_dst_port' : session.udp_dst_port,
               'tc' : 0 }
      args.update( updateDict )
      cli.bess.run_module_command( its_module, 'setRxStats',
                                 'ItsRxStatsArg', args )
   except Exception as error:
      # pylint: disable=no-member
      print( "Error :", str( error ) )
      return

def its_change_global_config( cli, updateDict ):
   # If we do not have an ItsCtrl, exit the code silently
   mod = [ m.name for m in cli.bess.list_modules().modules
         if m.mclass == 'ItsCtrl' ]
   if len( mod ) == 0:
      return
   sessions = get_its_sessions( mod[ 0 ], cli )
   for s in sessions:
      its_change_config( mod[ 0 ], s, cli, updateDict )

@commands.cmd( 'its adaptive-win-sizing enable',
               'enable adaptive resizing in ITS' )
def its_enable_adaptive_window_sizing( cli ):
   updateDict = { 'mod_adaptive_sizing' : 1,
               'adaptive_sizing' : 1 }
   its_change_global_config( cli, updateDict )

@commands.cmd( 'its adaptive-win-sizing disable',
               'disable adaptive resizing in ITS' )
def its_disable_adaptive_window_sizing( cli ):
   updateDict = { 'mod_adaptive_sizing' : 1,
               'adaptive_sizing' : 0 }
   its_change_global_config( cli, updateDict )

@commands.cmd( 'its adaptive-win-sizing set-win-size WINDOW_SIZE',
               'setting ITS sequence window size to WINDOW_SIZE' )
def its_set_window_size( cli, win_size ):
   # When we set the sequence window size, we disable the adaptive resizing feature.
   # This is because from the user's perspective, setting a window size to a static
   # value means that the user does not want it to change/adapt.
   updateDict = { 'mod_adaptive_sizing' : 1,
                  'adaptive_sizing' : 0,
                  'mod_seq_window_size' : 1,
                  'seq_window_size' : win_size }
   its_change_global_config( cli, updateDict )

@commands.cmd( 'show its statistics [PATH_INDEX]', 'show ITS path statistics' )
def show_its_statistics( cli, pathId ):
   mod = [ m.name for m in cli.bess.list_modules().modules
           if m.mclass == 'ItsCtrl' ]
   if len( mod ) == 0:
      return
   args = {}
   if pathId:
      args[ 'path_index' ] = pathId
   try:
      ret = cli.bess.run_module_command( mod[ 0 ], 'getSessStats',
                                         'ItsDebugArg', args )
      if not ret:
         return
      for entry in ret.its_stats:
         session_timer_args = {}
         session_timer_args[ 'path_index' ] = entry.session.path_index
         session_timer_args[ 'tc' ] = 0
         ret_timer_stat = cli.bess.run_module_command( mod[ 0 ], 'getSessionTimer',
                                         'ItsPathIndexArg', session_timer_args )

         cli.fout.write( 'Path %d: \n' % entry.session.path_index )
         cli.fout.write( '  src %s, dst %s, dst port %d\n' % \
                         ( int2ip( entry.session.src_ip ),
                           int2ip( entry.session.dst_ip ),
                           entry.session.udp_dst_port ) )
         cli.fout.write( '  IPsec encap ID %d\n' % entry.session.ipsec_encap_id )
         cli.fout.write( '  TX Stats:\n' )
         cli.fout.write( '    path state %s, prev state %s\n' % \
                         ( pPathState( entry.tx_stats.state ),
                           pPathState( entry.tx_stats.prev_state ) ) )
         cli.fout.write( '    active time %d\n' % entry.tx_stats.active_time )
         cli.fout.write( '    send rate %d\n' % entry.tx_stats.send_rate )
         cli.fout.write( '    pkt loss rate %d\n' % entry.tx_stats.pkt_loss_rate )
         cli.fout.write( '    rtt %d, rtt avg %d\n' % ( entry.tx_stats.rtt,
                                                        entry.tx_stats.rtt_avg ) )
         cli.fout.write( '    jitter %d\n' % entry.tx_stats.jitter )
         cli.fout.write( '    recv rate %d\n' % entry.tx_stats.recv_rate )
         cli.fout.write( '    keepalive timer hits %d\n' % \
                         entry.tx_stats.timer_hits )
         if ret_timer_stat:
            cli.fout.write( '    keepalive interval %d ms\n' %
                              ret_timer_stat.intervals.keepalive_timer )
         cli.fout.write( '  RX Stats:\n' )
         cli.fout.write( '    pkt loss rate %d\n' % entry.rx_stats.pkt_loss_rate )
         cli.fout.write( '    recv rate %d\n' % entry.rx_stats.recv_rate )
         cli.fout.write( '    pkts %d, bytes %d\n' % ( entry.rx_stats.pkt_cnt,
                                                       entry.rx_stats.byte_cnt ) )
         cli.fout.write( '    pkts snapshot %d, bytes snapshot %d\n' % \
                         ( entry.rx_stats.pkt_snapshot,
                           entry.rx_stats.byte_snapshot ) )
         cli.fout.write( '    lost pkts %d\n' % entry.rx_stats.lost_pkt_cnt )
         cli.fout.write( '    lost pkts snapshot %d\n' % \
                         entry.rx_stats.lost_pkt_snapshot )
         cli.fout.write( '    too old pkts %d\n' % entry.rx_stats.old_cnt )
         cli.fout.write( '    too old pkts snapshot %d\n' % \
                         entry.rx_stats.old_snapshot )
         cli.fout.write( '    too new pkts %d\n' % entry.rx_stats.new_cnt )
         cli.fout.write( '    feedback timer hits %d\n' % entry.rx_stats.timer_hits )
         if ret_timer_stat:
            cli.fout.write( '    feedback interval %d ms\n' %
                              ret_timer_stat.intervals.feedback_timer )
         cli.fout.write( '    sequence window size %d\n' %
                         entry.rx_stats.seq_window_size )
         cli.fout.write( '    adaptive sizing enabled %s\n' %
                         entry.rx_stats.adaptive_sizing )
      cli.fout.write( 'Helpers:\n  loss rate: (0 to 1) scalled up by 1000000\n' + \
                      '  recv rate: bits per second\n'
                      '  activeTime, rtt, rttAvg and jitter: ns\n\n' )
   except Exception as error:
      # pylint: disable=no-member
      print( error )
      return
   cli.fout.write( '\n' )

@commands.cmd( 'show its counters [PATH_INDEX] [CPULIST]', 'show ITS path counters' )
def show_its_counters( cli, pathId, cpulist ):

   mod = [ m.name for m in cli.bess.list_modules().modules
           if m.mclass == 'ItsCtrl' ]
   if len( mod ) == 0:
      return
   args = {}
   args[ 'cpu_list' ] = 0xffffffff
   if pathId:
      args[ 'path_index' ] = pathId
   if cpulist:
      args[ 'cpu_list' ] = cpulist
   try:
      ret = cli.bess.run_module_command( mod[ 0 ], 'getCounters',
                                         'ItsDebugArg', args )
      if not ret:
         return
      cli.fout.write( 'ITS Global Counters:\n' )
      cli.fout.write( f'  TX drop {ret.tx_drop}\n' )
      cli.fout.write( f'  TX timer err {ret.tx_timer_err}\n' )
      cli.fout.write( f'  RX timer err {ret.rx_timer_err}\n' )
      cli.fout.write( f'  RX drop total {ret.rx_drop_total}\n' )
      cli.fout.write( f'  RX drop No ITS Path cb {ret.rx_drop_no_its_path_cb}\n' )
      cli.fout.write( f'  RX drop Invalid ITS Pkt {ret.rx_drop_inv_its_pkt}\n' )
      cli.fout.write( f'  RX drop No ITS cb {ret.rx_drop_no_its_cb}\n' )
      cli.fout.write( f'  RX drop PMTU cb {ret.rx_drop_pmtu_cb}\n' )
      cli.fout.write( f'  RX drop Short ITS Pkt {ret.rx_drop_short_its_pkt}\n' )
      cli.fout.write( '  RX drop Invalid ITS Data Pkt '
                      f'{ret.rx_drop_invalid_its_data_pkt}\n' )
      for entry in ret.its_sess_counters:
         cli.fout.write( 'Path %d: \n' % entry.session.path_index )
         cli.fout.write( '  path active %d\n' % entry.tx_active )
         cli.fout.write( '  path inactive %d\n' % entry.tx_inactive )
         cli.fout.write( '  keepalives gen err %d\n' % entry.keepalive_gen_err )
         cli.fout.write( '  keepalives sent %d, feedbacks recv %d\n' % \
                         ( entry.keepalive_sent, entry.feedback_recv ) )
         cli.fout.write( '  keepalives recv %d, feedbacks sent %d\n' % \
                         ( entry.keepalive_recv, entry.feedback_sent ) )
         cli.fout.write( '  pkts sent %d, bytes sent %d\n' % \
                         ( entry.pkt_sent, entry.byte_sent ) )
         cli.fout.write( '  pkts recv %d, bytes recv %d\n' % \
                         ( entry.pkt_recv, entry.byte_recv ) )
         cli.fout.write( '  path inactive pkts drop %d\n' % entry.tx_inactive_drop )
         cli.fout.write( '  security err %d\n' % entry.security_err )
         cli.fout.write( '  rx pkt loss %d\n' % entry.rx_packet_loss )
         cli.fout.write( '  window reset because of too old %d\n' % \
                         entry.wnd_reset_all_old )
         cli.fout.write( '  window reset because of half loss %d\n' % \
                         entry.wnd_reset_half_loss )
   except Exception as error:
      # pylint: disable=no-member
      print( error )
      return
   cli.fout.write( '\n' )
