# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import commands

def getSiTransitModName( cli ):
   for m in cli.bess.list_modules().modules:
      if m.mclass == 'SiTransit':
         return m.name
   return None

def show_si_transit_counters( cli, stats ):
   cli.fout.write( "Service Insertion Transit Counters:\n" )
   cli.fout.write( "-----------------------------------\n" )
   cli.fout.write( f"No of groups allocated : {stats.group_alloc_cnt}\n" )
   cli.fout.write( f"No of groups freed : {stats.group_free_cnt}\n" )
   cli.fout.write( f"No of flows allocated : {stats.flow_alloc_cnt}\n" )
   cli.fout.write( f"No of flows freed : {stats.flow_free_cnt}\n" )
   cli.fout.write( "No of groups added to group database : " +
         f"{stats.group_added_to_db_cnt}\n" )
   cli.fout.write( "No of groups deleted from group database : " +
         f"{stats.group_deleted_from_db_cnt}\n" )
   cli.fout.write( "No of group allocation failures : " +
         f"{stats.group_alloc_fail_cnt}\n" )
   cli.fout.write( "No of group addition failures : " +
         f"{stats.group_add_fail_cnt}\n" )
   cli.fout.write( "No of group deletion failures : " +
         f"{stats.group_del_fail_cnt}\n" )
   cli.fout.write( "No of instance additions : " +
         f"{stats.inst_add_to_group_cnt}\n" )
   cli.fout.write( "No of instance addition failures : " +
         f"{stats.inst_add_to_group_fail_cnt}\n" )
   cli.fout.write( "No of instance updates : " +
         f"{stats.inst_upd_to_group_cnt}\n" )
   cli.fout.write( "No of instance update failures : " +
         f"{stats.inst_upd_to_group_fail_cnt}\n" )
   cli.fout.write( "No of instance deletions : " +
         f"{stats.inst_del_from_group_cnt}\n" )
   cli.fout.write( "No of instance deletion failures : " +
         f"{stats.inst_del_from_group_fail_cnt}\n" )
   cli.fout.write( "No of additions to group-instance table : "
         + f"{stats.add_to_group_inst_tbl_cnt}\n" )
   cli.fout.write( "No of deletions from group-instance table : " +
         f"{stats.del_from_group_inst_tbl_cnt}\n" )
   cli.fout.write( "No of flow allocation failures : " +
         f"{stats.flow_alloc_fail_cnt}\n" )
   cli.fout.write( "No of packets detected without flow cache entries : " +
         f"{stats.no_flow_cache_cnt}\n" )
   cli.fout.write( "No of packets dropped due to non availability of instances : " +
         f"{stats.no_usable_instance_cnt}\n" )
   cli.fout.write( "No of packets dropped as the selected instance was not usable : "
         + f"{stats.selected_inst_not_usable_cnt}\n" )
   cli.fout.write( "No of instance reselection failures : " +
         f"{stats.inst_reselect_fail_cnt}\n" )
   cli.fout.write( "No of packets dropped due to VRF mismatch : " +
         f"{stats.connection_if_vrf_mismatch_cnt}\n" )
   cli.fout.write( "No of RCU mark group invalid callbacks : " +
         f"{stats.rcu_mark_group_invalid}\n" )
   cli.fout.write( '\n' )

@commands.cmd( 'show service-insertion transit counters',
     'show service insertion transit counters' )
def show_internet_exit_counters_cmd( cli ):
   modName = getSiTransitModName( cli )
   if not modName:
      return
   stats = cli.bess.run_module_command( modName, 'getCounters', 'EmptyArg', {} )
   show_si_transit_counters( cli, stats )

def show_service_group_policy( cli, group_info ):
   cli.fout.write( f'group_id {group_info.group_id}\n' )
   for instance in group_info.instance_info:
      cli.fout.write( '  index %2d o_gate %4d nh_index %10d vrf_id %10d\n' %
                   ( instance.instance_index,
                      instance.o_gate, instance.nh_index, instance.vrf_id ) )

def show_service_group_helper( cli, cmd, argType, arg, printHeader=True ):
   def _groupIdKey( item ):
      return item.group_id

   modName = getSiTransitModName( cli )
   if not modName:
      return

   try:
      response = cli.bess.run_module_command( modName, cmd, argType, arg )
   except Exception as error:
      # pylint: disable=no-member
      print( error.errmsg )
      return

   if not response.group_info:
      return
   if printHeader:
      cli.fout.write( 'Service Insertion Transit group lookup table:\n' )
   for groupInfo in sorted( response.group_info, key=_groupIdKey ):
      show_service_group_policy( cli, groupInfo )
   cli.fout.write( '\n' )

@commands.cmd( 'show service-insertion transit group SI_GROUP_ID',
      'Show service insertion transit service group information ' )
def show_si_transit_service_group_lookup( cli, group_id ):
   show_service_group_helper( cli, 'getGroupInfo', 'SiTransitServiceGroupArg',
         { 'group_id' : group_id }, printHeader=False )

@commands.cmd( 'show service-insertion transit group all',
      'Show service insertion transit service group information of all groups' )
def show_si_transit_service_group_lookup_all( cli ):
   show_service_group_helper( cli, 'getAllGroupInfo', 'EmptyArg', {} )

@commands.var_attrs( 'SI_GROUP_ID' )
def si_group_id_var_attrs():
   # Return (var_type(str), var_desc(str), var_candidates([str]))
   return ( 'int', '', [] )
