# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from .option_parser import OptionParser

class TcpdumpOptionParser(OptionParser):

   def register_options(self):
      self.add_option("-len", "uint")
      self.add_option("-dir", "string", valid_values=["out", "in"])
      self.add_option("-gate", "uint")
      self.add_option("-bess-filter", "string")
      self.add_option("-type", "string", valid_values=["l2", "l3"])
      self.add_option("-meta", "bool")
      self.add_option("-dpsSport", "uint")
      self.add_option("-dpsDport", "uint")
      self.add_option("-h", "action", action=self.help)

   @staticmethod
   def help():
      help_text = [
            "Usage:",
            " tcpdump MODULE [OPTIONS] [EXPRESSION]",
            "",
            "Module:",
            " It expects a valid bess module name",
            "",
            "Options:",
            " Along with the standard tcpdump options, we support",
            " -len\t\tcapture length (default is 64)",
            " -gate\t\tindex of the gate (default is 0)",
            " -dir\t\tdirection of the gate (choices : out/in, default is out)",
            " -bess-filter\tbackend filter (used by bess)",
            " -type\t\ttype of the packet (choices : l2/l3, default is l2)",
            " -meta\t\tshow bess metadata information along with packet data",
            " -dpsDport\tDest Port for outer udp packet as dps/its packet",
            " -dpsSport\tSource Port for outer udp packet as dps/its packet",
            " -h\t\thelp",
            "",
            "Expression:",
            " Frontend filter (used by tcpdump)"
         ]
      print("\n".join(help_text))

   def get_length(self):
      return self.get_value("-len", 256)

   def get_direction(self):
      return self.get_value("-dir", "out")

   def get_gate(self):
      return self.get_value("-gate", 0)

   def get_bess_filter(self):
      return self.get_value("-bess-filter")

   def get_type(self):
      return self.get_value("-type", "l2")

   def get_frontend(self):
      value = self.get_remaining()
      if value:
         return value
      return ""
   
   def get_meta(self):
      return self.get_value("-meta", False)

   def get_dps_ports(self):
      return ( self.get_value("-dpsSport", 0), self.get_value("-dpsDport", 0) )
