#!/usr/bin/env arista-python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# N.B. If you edit this file you must regenerate and replace
# the associated AsuPatch patch RPM. For more information see
# /src/AsuPatch/packages/README.

from __future__ import absolute_import, division, print_function
import six

import AsuPStore
import Cell
import Tracing
import Tac

t0 = Tracing.trace0

def pciName( pciAddress ):
   return '%04x:%02x:%02x.%1x' % ( pciAddress.domain, pciAddress.bus,
                                   pciAddress.slot, pciAddress.function )

class SandFastPktAsuPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   def __init__( self, fruConfigDir=None, sandStatus=None, bridgingConfig=None,
                 ethIntfStatusDir=None, lagStatus=None ):
      AsuPStore.PStoreEventHandler.__init__( self )
      self.fruConfigDir = fruConfigDir
      self.sandStatus = sandStatus
      self.bridgingConfig = bridgingConfig
      self.ethIntfStatusDir = ethIntfStatusDir
      self.lagStatus = lagStatus

   def getSupportedKeys( self ):
      return [ 'platformType', 'devices', 'driverParams', 'interfaces',
               'lagInterfaces' ]

   def getKeys( self ):
      return [ 'platformType', 'devices', 'driverParams', 'interfaces',
               'lagInterfaces' ]

   def getDevicesInfo( self ):
      deviceInfo = []
      FapPortId = Tac.Type( 'Hardware::Sand::FapPortId' )
      for fruConfig in six.itervalues( self.fruConfigDir ):
         for fapConfig in six.itervalues( fruConfig.fapConfig ):
            deviceInfo.append( {
               'name' : fapConfig.name,
               'pciName' : pciName( fapConfig.pciHam.hamImpl.address ),
               'numFaps' : 1,
               'fapType' : Tac.enumValue(
                  'Hardware::Sand::FapType', fapConfig.fapType ),
               'cpuTm' : Tac.enumValue(
                  'Hardware::Sand::SystemPhyPortId::SppIdConst', 'cpuTm' ),
               'cpuSflow' : FapPortId.cpuSflow,
               'cpuAradSysPort' : Tac.enumValue(
                  'Hardware::Sand::SystemPhyPortId::SppId', 'cpuFap' ),
               'cpuArad' : FapPortId.cpuArad,
               'cpuAradBridge' : 0,
               'cpuFapBridgeNonIp' : 134,
            } )
      return deviceInfo

   def getDriverParams( self ):
      oldestFapType = Tac.enumValue( 'Hardware::Sand::FapType',
                                     self.sandStatus.oldestFapInSystem )
      return {
         'maxLags' : self.sandStatus.maxLags,
         'maxMembersPerLag' : self.sandStatus.maxMembersPerLag,
         'oldestFapType' : oldestFapType
      }

   def getInterfacesInfo( self ):
      interfaces = {}
      systemPhyPortId = self.sandStatus.systemPhyPortId
      portMap = self.sandStatus.portMap
      fapName = self.sandStatus.fapName
      isSubIntfId = Tac.Type( 'Arnet::SubIntfId' ).isSubIntfId
      for intfName, ethPhyIntfStatus in six.iteritems( self.ethIntfStatusDir ):
         if not intfName.startswith( 'Ethernet' ) or \
            isSubIntfId( intfName ) or \
            ethPhyIntfStatus.operStatus != 'intfOperUp':
            continue
         switchIntfConfig = self.bridgingConfig.switchIntfConfig[ intfName ]
         sppId = systemPhyPortId[ intfName ]
         fapId = portMap[ sppId ].fapId
         dot1qTunnel = 1 if switchIntfConfig.switchportMode == 'dot1qTunnel' else 0
         interfaces[ intfName ] = {
            "systemPhyPortId" : systemPhyPortId[ intfName ],
            "nativeVlan" : switchIntfConfig.nativeVlan,
            "trafficClass" : 7,
            "deviceName" : fapName[ fapId ],
            "dot1qTunnel" : dot1qTunnel,
         }
      return interfaces

   def getLagInterfacesInfo( self ):
      lagInterfaces = {}
      for intfName, lagStatus in six.iteritems( self.lagStatus.lagStatus ):
         lagInfo = {
            'lagNumber' : lagStatus.lagDestId.lagNumber,
            'isHwLag' : 1 if lagStatus.lagDestId.isHwLag else 0,
            'members' : {}
         }
         for memberName, memberId in six.iteritems( lagStatus.memberId ):
            lagInfo[ 'members' ][ memberName ] = memberId
         lagInterfaces[ intfName ] = lagInfo
      return lagInterfaces

   def save( self, pStoreIO ):
      pStoreIO.set( 'platformType', 'Sand' )
      pStoreIO.set( 'devices', self.getDevicesInfo() )
      pStoreIO.set( 'driverParams', self.getDriverParams() )
      pStoreIO.set( 'interfaces', self.getInterfacesInfo() )
      pStoreIO.set( 'lagInterfaces', self.getLagInterfacesInfo() )


def Plugin( ctx ):
   featureName = 'SandPlatformFastPktRestore'

   if ctx.opcode() == 'GetSupportedKeys':
      eventHandler = SandFastPktAsuPStoreEventHandler()
   else:
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()

      fruConfigDir = mg.mount( 'hardware/sand/fap/config/fru/fap',
                               'Tac::Dir', 'ri' )
      # Required by pciHam.hamImpl
      pciDeviceStatusDirPath = Cell.path( 'hardware/pciDeviceStatusDir' )
      mg.mount( pciDeviceStatusDirPath, 'Hardware::PciDeviceStatusDir',
                'r' )
      sandStatus = mg.mount( 'hardware/sand/system/status/sand',
                             'Hardware::Sand::Status', 'r' )
      bridgingConfig = mg.mount( 'bridging/config', 'Bridging::Config', 'r' )
      ethIntfStatusDir = mg.mount( 'interface/status/eth/intf',
                                   'Interface::EthIntfStatusDir', 'ri' )
      lagStatus = mg.mount( 'hardware/sand/lag/sandLagStatus',
                            'Hardware::SandLag::Status', 'r' )

      mg.close( blocking=True )
      eventHandler = SandFastPktAsuPStoreEventHandler( fruConfigDir,
                                                       sandStatus,
                                                       bridgingConfig,
                                                       ethIntfStatusDir,
                                                       lagStatus )

   ctx.registerAsuPStoreEventHandler( featureName, eventHandler )
