# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys

# Force reload the latest AsuPatchBase from swi
# pylint: disable-msg=import-error
sys.modules.pop( "AsuPatchBase", None )

# pylint: disable-msg=wrong-import-position
# This has to be after the pop above.
from AsuPatchBase import AsuPatchBase

_tdir = '/tmp/'
_pdir = '/usr/lib/python2.7/site-packages/AsuPStorePlugin/'
_pname = 'IraAsuPStore.py'
_pmod = 'AsuPStorePlugin.IraAsuPStore'
_ppath = _pdir + _pname
_backupPath = _tdir + _pname
_patchedPath = _backupPath + '.patched'

# sed script for patching the plugin.
# Remove lines that set the reload blocking lists:
#
#      if self.routingHardwareConfig_ and \
#         len( self.routingHardwareConfig_.resilientEcmpPrefix ) != 0:
#         blockingList.append( ReloadHitlessBlockingReason(
#                                 reason='resilientECMPv4' ) )
#      if self.routing6HardwareConfig_ and \
#         len( self.routing6HardwareConfig_.resilientEcmpPrefix ) != 0:
#         blockingList.append( ReloadHitlessBlockingReason(
#                                 reason='resilientECMPv6' ) )
#
_script = """/if self.routingHardwareConfig_ and /,/reason='resilientECMPv6' ) )/d
"""

class AsuPatchEnableReEcmp( AsuPatchBase ):
   '''
   Patch IraAsuPStore.py so it does not blocks reload.
   Do not block reload no matter what happens because
   if the patch fails then IraAsuPStore.py will block reload anyway.
   '''

   def __init__( self ):
      AsuPatchBase.__init__( self, 'AsuPatchEnableReEcmp' )

   def check( self ):
      # N.B. check() is not called for postInstall scripts.
      # It's defined here to make pylint happy.
      return 0

   def reboot( self ):
      if not os.path.isfile( _ppath ):
         self.log( _ppath + ' does not exist' )
         return 0
      try:
         self._patchFile()
      except OSError as e:
         self.log( str( e ) )
      return 0

   def cleanup( self ):
      # _backupPath existing means the patch was installed.
      if not os.path.isfile( _backupPath ):
         return
      try:
         # Restore from backup files.
         cmd = 'sudo /bin/mv -f ' + _backupPath + '* ' + _pdir
         os.system( cmd )
         self.log( 'Plugin restored.' )
      except OSError as e:
         self.log( str( e ) )

   def _rmBackupPath( self ):
      # Patch failed. Remove backup for cleanup().
      cmd = '/bin/rm -f ' + _backupPath
      os.system( cmd )

   def _patchFile( self ):
      # Create sed script.
      spath = _tdir + 'AsuPatchScdMixinSed'
      with open( spath, 'w' ) as sFile:
         sFile.write( _script )

      # Backup original py & pyc files for cleanup().
      # If backup exists it might be from a previous reboot() that
      # installed a new file, so keep the old backup.
      if not os.path.isfile( _backupPath ):
         cmd = 'sudo /bin/cp -a ' + _ppath + '* ' + _tdir
         os.system( cmd )

      # Make patched file with sed.
      cmd = '/bin/sed -f ' + spath + ' ' + _ppath + ' >' + _patchedPath
      status = os.system( cmd )
      if status is not 0:
         self.log( 'sed errored (%#x) patching the plugin: %s' % ( status, cmd ) )
         self._rmBackupPath()
         return

      # Check that the file changed.
      cmd = '>/dev/null /bin/diff -q ' + _ppath + ' ' + _patchedPath
      status = os.system( cmd )
      # diff exit status is 0 if inputs are the same, 1 if different, 2 if trouble.
      if status != 0x0100:
         self.log( 'The plugin was not changed (status %#x).' % status )
         self._rmBackupPath()
         return

      # Remove old py and pyc files. Keep backup files for cleanup().
      cmd = 'sudo /bin/rm -f ' + _ppath + '* '
      status = os.system( cmd )
      if status is not 0:
         self.log( 'Failed %#x: %s' % ( status, cmd ) )
         return

      # Copy new py file to site-packages. There is no new pyc.
      cmd = 'sudo /bin/cp ' + _patchedPath + ' ' + _ppath
      status = os.system( cmd )
      if status is not 0:
         self.log( 'Failed %#x: %s' % ( status, cmd ) )
         self.cleanup()
         return
      sys.modules.pop( _pmod, None ) # Flush previous version of the plugin.
      self.log( _pname + ' patch installed.' )

_helper = AsuPatchEnableReEcmp()

def execute( stageVal, *args, **kwargs ):
   return _helper.execute( stageVal, *args, **kwargs )
