# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import sys

import Tac

# Force reload the latest AsuPatchBase from swi
# pylint: disable-msg=import-error
# pylint: disable-msg=wrong-import-position
sys.modules.pop( "AsuPatchBase", None )

from AsuPatchBase import AsuPatchBase

SRC_CMDLINE = '/proc/cmdline'
DST_CMDLINE = '/tmp/asu-patched-proc-cmdline'

class KernelCmdlineFilter( AsuPatchBase ):
   '''
   Filter out kernel command line parameters that existed in older releases but
   that cause undesirable effects on newer EOS releases. This problem arises from
   the fact that ASU re-uses the command line from the source EOS image to build
   the destination's kernel command line.

   The kernel command line in ASU is constructed after the execution of AsuPatch.
   In order to still be able to filter out unwanted parameter, this patch alters
   the content of /proc/cmdline by bind-mounting a modified version with the
   filtered out parameters.
   '''

   def __init__( self ):
      AsuPatchBase.__init__( self, 'KernelCmdline' )

   def check( self ):
      return 0

   def reboot( self ):
      ARGS_TO_IGNORE = [
         re.compile( 'systemd.default_standard_output=tty' ),
      ]

      try:
         with open( SRC_CMDLINE, 'r' ) as f:
            origCmdline = f.read()

         finalCmdline = []
         for line in origCmdline.split():
            ignore = False
            for ignoreRe in ARGS_TO_IGNORE:
               if re.match( ignoreRe, line ):
                  ignore = True
                  break
            if not ignore:
               finalCmdline.append( line )

         with open( DST_CMDLINE, 'w' ) as f:
            f.write( ' '.join( finalCmdline ) )

         Tac.run( [ "mount", "--bind", DST_CMDLINE, SRC_CMDLINE ],
                  stdout=Tac.DISCARD, stderr=Tac.DISCARD, asRoot=True )
      except ( Tac.SystemCommandError, IOError ) as e:
         self.log( str( e ) )
      return 0

   def cleanup( self ):
      return 0

_helper = KernelCmdlineFilter()

def execute( stageVal, *args, **kwargs ):
   return _helper.execute( stageVal, *args, **kwargs )
