# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This PStore Plugin is used to save IntfPolicyManager proposed status, including
interface state (activeness) and interface mode (speed, lane, fec and duplex) during
ASU. These statuses are helpful to recover HW port configuration on boot.

There are two keys used in this Plugin, "intfStateMapping" and "ethIntfModeMapping",
where "intfStateMapping" encaps the intfStateDir and "ethIntfModeMapping" encaps
ethIntfModeDir in resourceManagerStatus. They have the corresponding format:
- "intfStateMapping"
  { sliceId :
       { subSliceId :
            { intfId :
                 { "swState" : swState,
                   "hwState" : hwState, }, }, }, }

- "ethIntfModeMapping"
  { sliceId :
       { subSliceId :
            { intfId :
                 { "speed" : speed,
                   "duplex" : duplex,
                   "laneCount" : laneCount,
                   "fec" : fec, }, }, }, }


This patched ASU PStore plugin contains code designed to work around mGig related
issues when upgrading Camptonville from pre Xian releases to post Xian releases.

See BUG549822 for more details.
"""

from __future__ import absolute_import, division, print_function

import AsuPStore

from TypeFuture import TacLazyType

EthSpeed = TacLazyType( 'Interface::EthSpeed' )

class AlePhyIntfPolicyManagerStatusPStoreEventHandler(
   AsuPStore.PStoreEventHandler ):

   def __init__( self,
                 resourceManagerStatusSliceDir,
                 eimRootDir,
                 entityMib ):
      self.resourceManagerStatusSliceDir_ = resourceManagerStatusSliceDir
      self.eimRootDir = eimRootDir
      self.entityMib = entityMib
      super( AlePhyIntfPolicyManagerStatusPStoreEventHandler, self ).__init__()

   def getSupportedKeys( self ):
      return [ 'intfStateMapping',
               'ethIntfModeMapping',
               'intfStateChangeGenMapping' ]

   def getKeys( self ):
      return [ 'intfStateMapping',
               'ethIntfModeMapping',
               'intfStateChangeGenMapping' ]

   def getIntfStateMapping( self ):
      intfStateMapping = {}
      # e.g. sliceId is "FixedSystem" and subSlices is "[ "Strata-FixedSystem" ]".
      for sliceId, subSlices in \
         self.resourceManagerStatusSliceDir_.items():
         intfStateMapping[ sliceId ] = {}
         for subSliceId, resourceManagerStatus in subSlices.items():
            perSubSliceMapping = {}
            intfStateMapping[ sliceId ][ subSliceId ] = perSubSliceMapping
            for intfId, intfState in \
               resourceManagerStatus.intfStateDir.intfState.items():
               state = intfState.state
               perSubSliceMapping[ intfId ] = \
                  { "swState" : state.swState, "hwState" : state.hwState }
      return intfStateMapping

   def adaptEthIntfMode( self, eim ):
      '''
      The following code resolves BUG549822.

      Effectively, starting from Xian, IPM started acknowledging mGig
      speeds and reported them in it's ResourceManagerStatus. Prior to this, mGig
      support was accomplished by IPM always publishing a fallback mode
      ( 10G/1G depending on PHY ) and relying on EthPhySm to discard
      ResourceManagerStatus's EIM and instead use the mGig value coming in from
      PhyAutonegStatus. Since we no longer do this override starting from Xian, we
      expect the pstore'd ResourceManagerStatus to contain the actual mGig speed.

      This adaption is relatively trivial as the inputs to IPM ( Autoneg Lib's EIM )
      already handle mGig correctly. In fact, since these BASE-T interfaces generally
      have no resource constraints, we should just get get away with overriding
      ResourceManagerStatus EIM with Autoneg Lib's EIM.
      '''

      # We only need this adaption on fixed systems
      if not self.entityMib.fixedSystem:
         return eim

      # Presently, this adaption only affects Camptonville.
      affectedSKUs = [ 'DCS-7050TX3-48C8' ]
      if self.entityMib.fixedSystem.modelName not in affectedSKUs:
         return eim

      # Try to retrieve the AutonegLib's EIM. If this fails for some reason then
      # we should just return the passed in EIM instead of crashing as a flapping
      # interface is considerably better than failing ASU.
      eimSlotDir = self.eimRootDir[ 'FixedSystem' ]
      if not eimSlotDir:
         return eim

      eimDir = eimSlotDir[ 'Strata-FixedSystem' ]
      if not eimDir:
         return eim

      autonegLibEim = eimDir.ethIntfMode[ eim.intfId ]
      if not autonegLibEim:
         return eim

      # Just to be on the safe side, we only want to adapt the EIM if it is mGig
      mGigSpeeds = [ EthSpeed.speed2p5Gbps, EthSpeed.speed5Gbps ]
      if autonegLibEim.speed not in mGigSpeeds:
         return eim

      return autonegLibEim

   def getEthIntfModeMapping( self ):
      ethIntfModeMapping = {}
      # e.g. sliceId is "FixedSystem" and subSlices is "[ "Strata-FixedSystem" ]".
      for sliceId, subSlices in \
         self.resourceManagerStatusSliceDir_.items():
         ethIntfModeMapping[ sliceId ] = {}
         for subSliceId, resourceManagerStatus in subSlices.items():
            perSubSliceMapping = {}
            ethIntfModeMapping[ sliceId ][ subSliceId ] = perSubSliceMapping
            for intfId, ethIntfMode in \
               resourceManagerStatus.ethIntfModeDir.ethIntfMode.items():
               ethIntfMode = self.adaptEthIntfMode( ethIntfMode )
               perSubSliceMapping[ intfId ] = \
                  { "speed" : ethIntfMode.speed, "duplex" : ethIntfMode.duplex,
                    "laneCount" : ethIntfMode.laneCount, "fec" : ethIntfMode.fec }
      return ethIntfModeMapping

   def getIntfStateChangeGenMapping( self ):
      intfStateChangeGenMapping = {}
      for sliceId, subSlices in \
         self.resourceManagerStatusSliceDir_.items():
         intfStateChangeGenMapping[ sliceId ] = {}
         for subSliceId, resourceManagerStatus in subSlices.items():
            intfStateChangeGenMapping[ sliceId ][ subSliceId ] = {}
            intfStateChangeGenMapping[ sliceId ][ subSliceId ][ "id" ] = \
               resourceManagerStatus.intfStateChangeGen.id
            intfStateChangeGenMapping[ sliceId ][ subSliceId ][ "valid" ] = \
               resourceManagerStatus.intfStateChangeGen.valid
      return intfStateChangeGenMapping

   def save( self, pStoreIO ):
      intfStateMapping = self.getIntfStateMapping()
      pStoreIO.set( 'intfStateMapping', intfStateMapping )
      ethIntfModeMapping = self.getEthIntfModeMapping()
      pStoreIO.set( 'ethIntfModeMapping', ethIntfModeMapping )
      intfStateChangeGenMapping = self.getIntfStateChangeGenMapping()
      pStoreIO.set( 'intfStateChangeGenMapping', intfStateChangeGenMapping )

def Plugin( ctx ):
   featureName = 'AlePhyIntfPolicyManagerStatus'

   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler(
         featureName,
         AlePhyIntfPolicyManagerStatusPStoreEventHandler( None, None, None ) )
      return
   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()
   resourceManagerStatusSliceDir = mg.mount(
      "interface/resources/status/slice", "Tac::Dir", "ri" )

   eimRootDir = mg.mount( 'interface/archer/status/eth/phy/mode/slice',
                          'Tac::Dir',
                          'ri' )
   entityMib = mg.mount( 'hardware/entmib', 'EntityMib::Status', 'ri' )

   mg.close( blocking=True )
   ctx.registerAsuPStoreEventHandler(
      featureName,
      AlePhyIntfPolicyManagerStatusPStoreEventHandler(
         resourceManagerStatusSliceDir,
         eimRootDir,
         entityMib ) )
