#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
The Strata Logical Port PStore Plugin is used to recover the
logical port mapping during ASU2 for systems which do not have a fixed
physical to logical port mapping
"""
# N.B. If you edit this file you must regenerate and replace
# the associated AsuPatch patch RPM. For more information see
# /src/AsuPatch/packages/README.

# Format: {SliceId: { ModId: { IntfId : logicalPort }}}

import AsuPStore
import Cell
import Tracing
import Tac

t0 = Tracing.trace0

class StrataLogicalPortPStoreEventHandler( AsuPStore.PStoreEventHandler ):

   def __init__( self, modStatusParentDir ):
      self.modStatusParentDir_ = modStatusParentDir
      super( StrataLogicalPortPStoreEventHandler, self ).__init__()

   def getSupportedKeys( self ):
      return [ 'logicalPortMapping' ]

   def getKeys( self ):
      return [ 'logicalPortMapping' ]

   def getLogicalPortMapping( self ):
      """
      Returns sliceIds each with a mapping of interface Ids to logical ports
      """
      sliceIds = {}
      for sliceId in self.modStatusParentDir_.keys():
         fapIds = {}
         sliceIds[ sliceId ] = fapIds
         modStatusDir = self.modStatusParentDir_[ sliceId ]
         for modStatus in modStatusDir.moduleStatus.itervalues():
            intfs = {}
            # Older releases don't have fapId attribute. And fapId was
            # used interchangeably with modid. If fapId attribute is missing
            # use modId. We won't have a mismatch because everything pre-tundra
            # made that assumption.
            if hasattr( modStatus, 'fapId' ):
               ix = modStatus.fapId
            elif hasattr( modStatus, 'modId' ):
               ix = modStatus.modId
            fapIds[ ix ] = intfs
            for intf in modStatus.strataIntfToPortId.keys():
               intfs[ intf ] = modStatus.strataIntfToPortId[ intf ].port
      return sliceIds

   def save( self, pStoreIO ):
      logicalPortMapping = self.getLogicalPortMapping()
      pStoreIO.set( 'logicalPortMapping', logicalPortMapping )

def Plugin( ctx ):
   featureName = 'StrataLogicalPort'

   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler( featureName,
                                      StrataLogicalPortPStoreEventHandler( None ) )
      return
   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()
   modStatusPath = Cell.path( 'hardware/strata/centralStatus/slice' )
   modStatusDir = mg.mount( modStatusPath, "Tac::Dir", "ri" )
   mg.close( blocking=True )
   ctx.registerAsuPStoreEventHandler(
      featureName, StrataLogicalPortPStoreEventHandler( modStatusDir ) )
