#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import getopt
import json
import os
import sys
import time

import Ark
import Logging
import DropCounterLogMsgs

def usage():
   print( """
Generates Syslog messages for the repeated Drop Counters violators
list which was collected by the DropCounterMonitor.py script

Usage:
    DropCounterLog.py [ options ]

Options:
    -s  Print the list of violations and thier timestamps to stdout instead of
        creating SysLog messages
""" )

window = 900  # 15 minute
count = 3
currentCounters = {}   # { "name":value, ... }
violations = []        # [ ["name", timestamp, val, delta], ... ]
logEvents = {}         # { "name":timestamp, ... ]
repeatedViolators = {} # { 'violators':["name":count, ...] }
repeatedViolators[ 'violators' ] = {}

countersPath = "/tmp/DropCounters"
violationsPath = "/tmp/DropCountersViolations"
repeatedViolatorsPath = "/tmp/DropCountersRepeatedViolators"
logEventsPath = "/tmp/DropCountersLogEvents"

show = False
doSyslog = False
timestamp = time.time()

def parseOptions():
   global show, doSyslog

   try:
      opts, _ = getopt.getopt( sys.argv[ 1 : ], "hsl", [ "show", "syslog" ] )
   except getopt.GetoptError as err:
      # print help information and exit:
      print( str( err ) )
      usage()
      sys.exit( 2 )
   for opt, _ in opts:
      if opt in ( "-h", "--help" ):
         usage()
         sys.exit()
      elif opt in ( "-s", "--show" ):
         show = True
      elif opt in ( "-l", "--syslog" ):
         doSyslog = True
      else:
         usage()
         sys.exit( 3 )

def trimLogEvents():
   # Remove violations that are outside the time sliding window
   for counter, eventTimestamp in list( logEvents.items() ):
      if timestamp - eventTimestamp > window:
         del logEvents[ counter ]

def saveLogEvents():
   # Save the logEventList to flash
   with open( logEventsPath, "w" ) as logEventsFp:
      json.dump( logEvents, logEventsFp )
      logEventsFp.write( "\n" )

def jsonLoad( filepath ):
   if os.path.isfile( filepath ):
      with open( filepath ) as fp:
         try:
            return json.load( fp )
         except ValueError:
            pass
   return None

def readValues():
   # Read the violations and repeatedViolators dicts from flash
   global violations, repeatedViolators, logEvents, window, count, currentCounters

   currentCounters = jsonLoad( countersPath )
   if not currentCounters:
      currentCounters = {}

   violations = jsonLoad( violationsPath )
   if not violations:
      violations = []
   repeatedViolators = jsonLoad( repeatedViolatorsPath )
   if not repeatedViolators:
      repeatedViolators = {}
      repeatedViolators[ 'violators' ] = {}
   logEvents = jsonLoad( logEventsPath )
   if not logEvents:
      logEvents = {}

   window = repeatedViolators.get( "window", 900 )
   count = repeatedViolators.get( "count", 3 )

def main():
   parseOptions()
   readValues()
   trimLogEvents()
   for violation in violations:
      counter = violation[ 0 ]
      if counter in repeatedViolators[ 'violators' ]:
         violationTimestamp = violation[ 1 ]
         counterVal = violation[ 2 ]
         counterDelta = violation[ 3 ]
         violationCount = repeatedViolators[ 'violators' ][ counter ]
         if violationCount < count:
            continue
         if show:
            print( time.ctime( violationTimestamp ), counter, counterVal,
                   counterDelta, violationCount )
         elif counter not in logEvents:
            if doSyslog:
               parts = counter.split( '.' )
               if len( parts ) == 2:
                  fap = parts[ 0 ]
                  counterName = parts[ 1 ]
               else:
                  counterName = counter
                  fap = '-'
               Ark.configureLogManager( "EventMgr" )
               counterVal = currentCounters.get( counter )
               Logging.log( DropCounterLogMsgs.HARDWARE_DROP_COUNTER_ALERT,
                            counterName, str( counterVal ), fap )

               logEvents[ counter ] = timestamp
   if not show:
      saveLogEvents()

if __name__ == "__main__":
   main()
