#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=superfluous-parens
# pylint: disable=consider-using-f-string
import Tac
import SmbusUtil
import Pci
import ScdRegisters
import os
import struct
import sys
import time
import FpgaUtil

from optparse import OptionParser # pylint: disable=deprecated-module
from binascii import unhexlify

verboseEnabled = False
factory = SmbusUtil.Factory()
accelId = 0
busId = 3
deviceId = 0x3a 
addrSize = 4
portId = 0
bCrc = True
bEndianSwap = False
eepSize = 4
eepPortSize = 4
offsetCtrl = 0x260
plxEepromCmdWrite = 2
plxEepromCmdRead = 3
plxEepromCmdReadStatus = 5
plxEepromCmdWriteEnable = 6

chips = [ '8725', '8749', '8713', '8725-epk-fc', '8725-epk-lc' ]
eepromFiles = {}
eepromFiles['8749'] = '/usr/bin/PlxPex8749.eeprom.bin'
eepromFiles['8725'] = '/usr/bin/PlxPex8725.eeprom.bin'
eepromFiles['8713'] = '/usr/bin/PlxPex8713.eeprom.bin'
eepromFiles['8725-epk-lc'] = '/usr/bin/PlxPex8725-epk-lc-3a.eeprom.bin'
eepromFiles['8725-epk-fc'] = '/usr/bin/PlxPex8725-epk-fc-38.eeprom.bin'

smbusAddress = {}
smbusAddress['8749'] = { 'accelId' : 0 , 'busId' : 3, 'deviceId' : 0x38 }
smbusAddress['8725'] = { 'accelId' : 0 , 'busId' : 3, 'deviceId' : 0x3a }
smbusAddress['8725-epk-fc'] = { 'accelId' : 0 , 'busId' : 3, 'deviceId' : 0x38 }
smbusAddress['8725-epk-lc'] = { 'accelId' : 0 , 'busId' : 3, 'deviceId' : 0x3a }

SMBUS0_CONTROLSTATUS = 0x8020


textBuffer = ""
errorBuffer = ""

def plxPrint( text ):
   global textBuffer
   print( text )
   textBuffer += text + "\n"

def plxPrintError( text ):
   global errorBuffer
   plxPrint( "ERROR: " + text )
   errorBuffer += text + "\n"

def fatalQuit():
   FpgaUtil.printToConsole(
      "EEPROM program failure:\n" + textBuffer +
      "EEPROM ERROR(s): " + errorBuffer )
   sys.exit( 1 )

def verbose( text ):
   if verboseEnabled == True: # pylint: disable=singleton-comparison
      plxPrint( text )

def plxEepromWaitIdle():
   # Get EEPROM control register
   regCmd = plxRegRead( offsetCtrl )

   # Clear command field [15:13]
   regCmd &= ~(7 << 13)

   # Clear the EepRdy bit [24]
   regCmd &= ~(1 << 24)

   # Clear the EepWrStatus bits[30:28]
   regCmd &= ~(7 << 28)

   # Check if there is initial CRC error reported ([19])
   if (regCmd & (1 << 19)):
      # In event of CRC error on power up, the error must be
      # cleared by writing '1'. In addition, a dummy EEPROM read
      # command might be needed to "unblock" the EEPROM controller
      # (mainly over I2C), along with a small delay to let it
      # complete. Without this dummy command, the 1st EEPROM read
      # command will fail.
      plxRegWrite( offsetCtrl, regCmd | (plxEepromCmdRead << 13) )
      regCmd &= ~(1 << 19)
      time.sleep( 50 )

   # Prepare EEPROM write command
   regCmd |= (plxEepromCmdReadStatus << 13)

   # Set timeout
   timeout = 10000

   # Query EEPROM status until it's ready
   while timeout > 0:
      # Send command to get EEPROM status in bits [31:24]
      plxEepromSendCommand( regCmd )

      # Get EEPROM control register
      regValue = plxRegRead( offsetCtrl )

      # Check EEPROM read (bit 24) & write ([30:28]) status bits
      if ( ((regValue & (1 << 24)) == 0) and
           ((regValue & (7 << 28)) == 0) ):
         return True
      
      # Decrement timeout
      timeout = timeout - 1

   return False

def plxEepromSendCommand( command ):
   verbose( 'plxEepromSendCommand called with command: %x' % command )
   regValue = 0
   # Send EEPROM command
   plxRegWrite( offsetCtrl, command )
   # Setup timeout counter
   timeout = 100000
   # Wait for command to complete
   while True:
      # Get EEPROM control register
      regValue = plxRegRead( offsetCtrl )
      # EEPROM command is complete if status [19:18] is 0 or 2 (with CRC error)
      if ((regValue & (1 << 18)) == 0):
         return True
      # Decrement timeout
      timeout = timeout - 1
      if timeout == 0:
         break
   plxPrint ( 'plxEepromSendCommand timeout: %d, regValue 0x%x' % (
      timeout, regValue ) )
   return False

def doReadByteString( addr, size ):
   result = helper.readByteList( addr, size, size )
   verbose( 'read addr 0x%x size %d returns: %s' % ( addr, size, result ) )
   regValue = ( ( ( result[0] ) << 24 ) | 
                ( ( result[1] ) << 16 ) | 
                ( ( result[2] ) << 8  ) | 
                ( ( result[3] ) << 0 ) )
   verbose( 'regValue:0x%x' % regValue )
   return regValue


def _getSillyPlxAddressFormat(port, addr, read=True):
   if read:
      byte0 = 0x04
   else:
      byte0 = 0x03
   byte1 = ( port >> 1 ) & 0x7
   byte2 = ( ( ( addr >> 10 ) & 0x3) | ( 0xf << 2 ) | 
             ( ( port & 0x1 ) << 7 )  )
   byte3 = ( addr >> 2) & 0xff
    
   return ( ( byte0 << 24 ) | ( byte1 << 16 ) | ( byte2 << 8 ) | byte3 )

def dataStrToCharList( s ): # pylint: disable=inconsistent-return-statements
   length = len( s )
   if (length==0) or ((length % 2) != 0):
      plxPrint ( "Data string of wrong length" )
      return
   start = 0
   if s[1] in b'xX':
      start = 2
   try:
      return unhexlify( s[ start: ] )
   except:  # pylint: disable=bare-except
      plxPrint ( "Wrong data string" )
      return


def plxRegRead( offset ):
   verbose( 'plxRegRead called with offset:0x%x' % offset )
   addr = _getSillyPlxAddressFormat(portId, offset, True)
   verbose( f'plxRegRead called with offset:0x{offset:x} addr 0x{addr:x}' )
   return doReadByteString( addr, 4 )

def plxRegWrite( offset, data ):
   verbose( f"plxRegWrite called with offset 0x{offset:x} data 0x{data:x} " )
   dataStr = b"0x%08x" % data
   addr = _getSillyPlxAddressFormat(portId, offset, False)
   dataList = dataStrToCharList( dataStr )
   if len(dataList) != 4:
      plxPrintError ( "data must be 4 bytes long exactly" )
      plxPrintError ( 'dataList: %s' % dataList )
      plxPrintError ( 'dataStr: %s' % dataStr )
      plxPrintError ( 'data: %s' % data )
      plxPrintError ( 'offset: %s' % offset )
      return

   helper.write( addr, dataList, len( dataList ) )

def setDataWidth( offset=0 ):
   for w in 2, 3:
      if offset < ( 1 << ( 8 * w ) ) :
         width = w
         break
   verbose( "Offset %x requires width %x" %
            ( offset, width ) )

   # See code in
   # //src/PlxPex8600/2012.sequoia-prasanna-new/PlxSdk/Linux/Driver/Source.Plx8000
   # File Eep_8000.c, Fuunction Plx8000_EepromSetAddressWidth
   ctrl = plxRegRead( offsetCtrl )
   # Clear command field [15:13] to avoid EEPROM cycle
   ctrl &= ~( 7 << 13 )
   # Set address width override enable
   ctrl |= ( 1 << 21 )
   # Enable override
   plxRegWrite( offsetCtrl, ctrl )
   # Verify width is overridable
   ctrl = plxRegRead( offsetCtrl )
   assert ctrl & ( 1 << 21 )
   # Clear command field [15:13] to avoid EEPROM cycle
   ctrl &= ~( 7 << 13 )
   # Clear wdith field [23:22]
   ctrl &= ~( 3 << 22 )
   # Set address width [23:22]
   ctrl |= ( width & 0x3 ) << 22
   # Set new address width
   plxRegWrite( offsetCtrl, ctrl )
   # Read back and check
   ctrl = plxRegRead( offsetCtrl )
   assert ( ( ctrl >> 22 ) & 0x3 ) == width
   plxPrint( "Data width reset to %d" % width )

def prepareToAccess( offset ):
   ctrl = plxRegRead( offsetCtrl )
   EepWidth = ( ctrl >> 22 ) & 0x3
   if offset >= ( 1 << ( EepWidth * 8 ) ):
      setDataWidth( offset )
   # Get new EEPROM control register
   regValue = plxRegRead( offsetCtrl )
   # Determine byte addressing ([23:22])
   EepWidth = ( regValue >> 22 ) & 0x3
   return regValue, EepWidth

def eepromReadByOffset32( offset ): # pylint: disable=inconsistent-return-statements
   verbose( 'eepromReadByOffset32 called with offset:0x%x' % offset )
   EepWidth = 0
   regValue = 0
   RegUpper = 0
   # Verify access is supported
   if offsetCtrl == 0x0:
      return
   # Wait until EEPROM is ready
   if plxEepromWaitIdle() == False: # pylint: disable=singleton-comparison
      plxPrintError ( "plxEepromWaitIdle failed" )
      return
   regValue, EepWidth = prepareToAccess( offset )
   if EepWidth == 0:
      EepWidth = 1
   # For 3-byte addressing, set upper byte
   if EepWidth == 3:
      RegUpper = plxRegRead( offsetCtrl + 0xC )    # Offset x26Ch
      #Set 3rd address byte (26Ch[7:0])
      RegUpper &= ~(0xFF << 0)
      RegUpper |= (offset >> 16) & 0xFF
      plxRegWrite( offsetCtrl + 0xC, RegUpper )  # Offset x26Ch
   # Convert offset to an index
   offset = ( offset // 4 )
   # Clear command field [15:13]
   regValue &= ~(7 << 13)
   # Clear offset field [20,12:0]
   regValue &= ~((1 << 20) | (0x1FFF << 0))
   # Prepare EEPROM read command
   regValue |= ( ( ( offset & 0x1FFF ) << 0 ) |           #  Bits [12:0] of offset
                 ( ( ( offset >> 13 ) & 1 ) << 20 ) |     # Bit 13 of offset
                 ( plxEepromCmdRead  << 13 ) )         # EEPROM command
   # Send EEPROM command
   plxEepromSendCommand( regValue )
   # Return EEPROM data
   pValue = plxRegRead( offsetCtrl + 0x4 )   # Offset x264h
   return pValue


def eepromReadByOffset16( offset ):
   value32 = eepromReadByOffset32( offset )
   verbose( 'eepromReadByOffset16 called with offset 0x{:x} value32 0x{:x}'.format(
      offset, value32 ) )
   if ( ( offset >> 1 ) & 1 ): 
      return ( ( value32 >> 16 )  & 0xffff )
   else: 
      return ( value32 & 0xffff )

def eepromPrint():
   # Start with EEPROM header size
   # Get EEPROM register count
   value = eepromReadByOffset16( 0x02 )
   # Add register byte 
   eepSize2 = 4
   eepSize2 = eepSize2 + value
   for offset in range( 4, eepSize2, 6 ):
      addr = eepromReadByOffset16( offset ) * 4
      value2 = eepromReadByOffset16( offset + 2 )
      value1 = eepromReadByOffset16( offset + 4 )
      plxPrint ( f'0x{addr:05x} 0x{value1:04x}{value2:04x}' )

def eepromValid():
   value = eepromReadByOffset16( 0 )
   if value != 0x5a:
      return "Corrupted"
   value = eepromReadByOffset16( 0x02 )
   if value < 4:
      return "Empty"
   return "Valid"

def eepromFormat():
   plxPrint ( "writing 0x0000005a at %x" % ( offsetCtrl + 4 ) )
   plxRegWrite( offsetCtrl + 4, 0x0000005a )
   plxPrint ( "writing 0x00a0c000 at %x" % ( offsetCtrl ) )
   plxRegWrite( offsetCtrl, 0x00a0c000 )
   plxPrint ( "writing 0x00a04000 at %x" % ( offsetCtrl ) )
   plxRegWrite( offsetCtrl, 0x00a04000 )

def endianSwap32( value ):
   return ( ((((value) >>  0) & 0xff) << 24) | 
            ((((value) >>  8) & 0xff) << 16) | 
            ((((value) >> 16) & 0xff) <<  8) | 
            ((((value) >> 24) & 0xff) <<  0) )


def endianSwap16( value ):
   return ( ((((value) >>  0) & 0xff) << 8) | 
            ((((value) >>  8) & 0xff) << 0) )

def eepromWriteByOffset32( offset, value ):
   # Wait until EEPROM is ready
   if plxEepromWaitIdle() == False: # pylint: disable=singleton-comparison
      plxPrintError ( "plxEepromWaitIdle failed" )
      return 
   regValue, EepWidth = prepareToAccess( offset )
   if EepWidth == 0:
      EepWidth = 1
   # For 3-byte addressing, set upper byte
   if EepWidth == 3:
      RegUpper = plxRegRead( offsetCtrl + 0xC )    # Offset x26Ch
      #Set 3rd address byte (26Ch[7:0])
      RegUpper &= ~(0xFF << 0)
      RegUpper |= (offset >> 16) & 0xFF
      plxRegWrite( offsetCtrl + 0xC, RegUpper )  # Offset x26Ch

   # Convert offset to an index
   offset = ( offset // 4 )
   # Clear command field [15:13]
   regValue &= ~(7 << 13)
   # Clear offset field [20,12:0]
   regValue &= ~((1 << 20) | (0x1FFF << 0))
   # Send EEPROM write enable command
   plxEepromSendCommand( regValue | (plxEepromCmdWriteEnable << 13) )
   # Prepare EEPROM data
   plxRegWrite( offsetCtrl+0x4, value )
   # Prepare EEPROM write command
   regValue |= ( ( ( offset & 0x1FFF )      <<  0 ) |     # Bits [12:0] of offset
                 ( ( ( offset >> 13 ) & 1 ) << 20 ) |     # Bit 13 of offset
                 ( plxEepromCmdWrite << 13 ) )         # EEPROM command


   # Send EEPROM write command
   plxEepromSendCommand( regValue )


def eepromWriteByOffset16( offset, value ):
   # Get current 32-bit value
   value_32 = eepromReadByOffset32( offset & ~0x3 )

   # Insert new 16-bit value in correct location
   if offset & 0x3:
      value_32 = (value << 16) | (value_32 & 0xFFFF)
   else:
      value_32 = (value) | (value_32 & 0xFFFF0000)

   # Update EEPROM
   eepromWriteByOffset32( (offset & ~0x3), value_32 )


def eepromComputeNextCrc( nextEepromValue, currentCrc ):
   xorValue = 0
   nextCrc = currentCrc
   CONST_CRC_XOR_VALUE = 0xDB710641

   # Step through each bit of the CRC
   for itr in range( 0, 32, 1 ):
      # Shift the CRC, XOR'ing in the constant as required
      xorValue = ((nextCrc ^ (nextEepromValue << itr)) & (1 << 31))
      if xorValue:
         xorValue = CONST_CRC_XOR_VALUE
      else:
         xorValue = 0
      verbose( "nxtEeVal 0x{:x} crnt 0x{:x} new 0x{:x}".format(
            nextEepromValue, nextCrc,  
            ( ( ( nextCrc << 1 ) ^ xorValue ) & 0xFFFFFFFF ) ) )
      nextCrc = ( ( (nextCrc << 1) ^ xorValue ) & 0xFFFFFFFF )
   return nextCrc


def eepromCrcUpdate( bUpdateEeprom = False ):
   # Set starting offset for CRC calculations
   OffsetCalcStart = 0

   # Determine the CRC EEPROM offset
   value_32 = eepromReadByOffset32( 0 )
   # Verify valid size
   if ( value_32 >> 16 ) == 0xFFFF:
      plxPrintError ( "EEPROM byte count invalid" )
      return False

   # CRC location after EEPROM (header + reg addr/data values)
   OffsetCrc = 4 + (value_32 >> 16)
   plxPrint ( "OffsetCrc: %s" % OffsetCrc )

   # CRC calculation starts at byte 2
   OffsetCalcStart = 2

   # Initialize CRC
   Crc = 0xFFFFFFFF

   # Calculate CRC by reading all values in EEPROM
   for offset in range( OffsetCalcStart, OffsetCrc, 4 ):
      # Read next EEPROM value
      if offset & 0x3:
         # EEPROM offsets are not DWord aligned, must build 16-bits at a time
         value_16 = eepromReadByOffset16( offset )
         value_32 = value_16
         # If final data not aligned on DWord, pad with 0's
         if ( ( offset + 2 ) < OffsetCrc ) :
            value_16 = eepromReadByOffset16( offset + 2 )
         else:
            value_16 = 0
         value_32 |= (value_16) << 16
      else:
         value_32 = eepromReadByOffset32( offset )
      verbose( f"offset 0x{offset:x} value_32 0x{value_32:x} Crc 0x{Crc:x} " )
      Crc = eepromComputeNextCrc( value_32, Crc )
      verbose( f"offset 0x{offset:x} value_32 0x{value_32:x} newCrc 0x{Crc:x} " )

   plxPrint ( f"Calculated CRC = {Crc:08X} (offset={OffsetCrc:X})" )

   if bUpdateEeprom:
      plxPrint ( "Write new CRC to EEPROM" )
      if ( ( OffsetCrc & 0x3 ) != 0 ):
         plxPrint ( "writing at offset 0x{:x}: 0x{:x}".format( OffsetCrc,
                                                         ( Crc & 0xFFFF ) ) )
         plxPrint ( "writing at offset 0x{:x}: 0x{:x}".format( OffsetCrc + 2,
                                                     ( ( Crc >> 16 ) & 0xFFFF ) ) )
         eepromWriteByOffset16( OffsetCrc, ( Crc & 0xFFFF )  )
         eepromWriteByOffset16( OffsetCrc + 2, ( (Crc >> 16) & 0xFFFF ) )
      else:
         plxPrint ( "writing at offset:0x{:x} 0x{:x}".format( OffsetCrc,
                                                        ( Crc & 0xFFFFFFFF ) ) )
         eepromWriteByOffset32( OffsetCrc, ( Crc & 0xFFFFFFFF ) )
   else:
      plxPrint ( "Skipping CRC update in EEPROM" )

   return True

def writeAndVerify( offset, value, useWrite16, timeout=3 ):
   def tryToWtite():
      if useWrite16:
         eepromWriteByOffset16( offset, value )
         verify = eepromReadByOffset16( offset )
      else:
         eepromWriteByOffset32( offset, value )
         verify = eepromReadByOffset32( offset )
      if verify == value:
         plxPrint ( "offset:%02X  wrote:%08lX  read:%08lX epWidth: %d, use16=%s" % (
            offset, value, verify, plxRegRead( offsetCtrl ) >> 22 & 3, useWrite16 ) )
         return True
      else:
         if verify == None: # pylint: disable=singleton-comparison
            verify = "None"
         else:
            verify = "%08x" % verify

         plxPrintError (
            "offset:%02X  wrote:%08lX  read:%s epWidth: %d, use16=%s" % (
            offset, value, verify, plxRegRead( offsetCtrl ) >> 22 & 3, useWrite16 ) )
         return False
   try:
      Tac.waitFor( tryToWtite, timeout=timeout,
                   description="Waiting %d secs to write %08lX to offset %08lx" % (
                      timeout, value, offset ) )
   except Tac.Timeout:
      return False
   return True

def eepromWrite():
   scdResource = Pci.MmapResource( ScdRegisters.scdPciResourceFile(), readOnly=True )

   try:
      f = open(eepromFileName, "rb") # pylint: disable=consider-using-with
   except OSError:
      plxPrintError ( "Unable to open file: %s" % eepromFileName )
      return False
   FileSize = os.stat( eepromFileName ).st_size
   plxPrint ( "FileSize: %s" % FileSize )

   #Move back to start of file
   f.seek(0, os.SEEK_SET)

   plxPrint ( "Ok (%d B)" % FileSize )

   if FileSize < eepSize: 
      plxPrintError ( "File is less than PLX minimum size (%d B)" % eepSize )
      return False

   plxPrint ( "Program EEPROM............" )
   word1 = f.read(4)
   assert word1[ 0:1 ] == b'Z', "Invalid magic exp Z, got %r" % word1[ 0 ]
   if not writeAndVerify( 0, 0x5A000000, False ):
      return False

   # Write buffer into EEPROM
   for offset in range( 4, FileSize, 4 ):
      useWrite16 = False
      # Periodicall update status
      if offset & 0xF == 0:
         plxPrint ( "%02d%%\b\b\b" % ( offset * 100 // FileSize ) )
      # Get next value to write
      word = f.read(4)
      try:
         valueStr = "%04x" % ( struct.unpack( 'I', word ) )
      except struct.error:
         useWrite16 = True
         valueStr = "%02x" % ( struct.unpack( 'H', word ) )

      value = int( valueStr, 16 ) 
      verbose( 'value:0x%x' % value )
      verbose( 'valueStr:%s' % valueStr )

      if bEndianSwap:
         # eepPortSize = 4 in 87xx
         if useWrite16:
            value = endianSwap16(value)
         else:
            value = endianSwap32(value)

      verbose( 'value after swap:0x%x' % value )

      def scdRead( addr ):
         return scdResource.read32( ScdRegisters.scdPciOffset() + addr )

      def fifoIsEmpty():
         # pylint: disable-next=cell-var-from-loop
         return ( ( scdRead( SMBUS0_CONTROLSTATUS ) >> 16) & 0x3ff ) == 0

      try:
         Tac.waitFor( fifoIsEmpty, timeout=2 *60,
                      description="Waiting SCD fifo to be empty" )
      except Tac.Timeout:
         return False

      if not writeAndVerify( offset, value, useWrite16 ):
         return False

   valueStr = "%04x" % ( struct.unpack('I', word1 ) )
   value = int( valueStr, 16 ) 
   if not writeAndVerify( 0, value, False, timeout=15 ):
      return False
   if bCrc:
      plxPrint ( "Calculate & update CRC..." )
      if not eepromCrcUpdate( bUpdateEeprom=True ):
         return False

   plxPrint ( " -- Complete --" )
   return True

def readBinaryEepromFile( binFile, outFile ):
   try:
      f = open(binFile, "rb") # pylint: disable=consider-using-with
   except OSError:
      plxPrint ( "Unable to open file: %s" % binFile )
      return
   FileSize = os.stat( binaryFile ).st_size
   #print "FileSize:", FileSize

   #Move back to start of file
   f.seek(0, os.SEEK_SET)

   if FileSize < eepSize: 
      plxPrintError ( "File is less than PLX minimum size (%d B)" % eepSize )
      return

   word = f.read(4)
   if word[ 0:1 ] != b'Z':
      plxPrintError ( "Invalid magic, expected Z, got %r" % word[ 0 ] )
      Tac.pdb()
      return

   stdout = sys.stdout
   if outFile is not None:
      sys.stdout = open( outFile, 'w' ) # pylint: disable=consider-using-with

   # Write buffer into EEPROM
   for _ in range( 4, FileSize, 6 ):
      word = f.read(2)
      addrStr = "%02x" % ( struct.unpack( 'H', word ) )
      addr = ( int( addrStr, 16 ) * 4 )

      word = f.read(4)
      valueStr = "%04x" % ( struct.unpack( 'I', word ) )
      value = int( valueStr, 16 ) 
      plxPrint( f'0x{addr:05x} 0x{value:08x}' )

   if outFile is not None:
      sys.stdout = stdout

def writeBinaryEepromFile( binFile, inTextFile ):
   try:
      f = open( inTextFile ) # pylint: disable=consider-using-with
   except OSError:
      plxPrint ( "Unable to open file: %s" % inTextFile )
      return

   try:
      outFile = open( binFile, "wb" ) # pylint: disable=consider-using-with
   except OSError:
      plxPrint( "Unable to open file: %s" % binFile )
      return

   #Move back to start of file
   outFile.seek(4, os.SEEK_SET)

   bytesWritten = 0
   for line in f.readlines():
      # Ignore comments
      if ( line.startswith( '#' ) ):
         continue
      words = line.split()
      addr = int( words[0], 16 )
      value = int( words[1], 16 )
      #print 'addr:0x%x' % addr
      #print 'value:0x%x' % value
      outFile.write( unhexlify( '%04x' % ( endianSwap16( addr // 4 ) ) ) )
      outFile.write( unhexlify( '%08x' % endianSwap32( value ) ) )
      bytesWritten += 6

   outFile.seek(0, os.SEEK_SET)
   outFile.write( unhexlify( '%04x' % endianSwap16( 0x5A ) ) )
   outFile.write( unhexlify( '%04x' % endianSwap16( bytesWritten ) ) )
   outFile.close()

   

usage = """
PlxEeprom.py [options] {read|write} [chip]
    chip could be '8749' or '8725'
    Default EEPROM binary files are installed 
      when you install PlxPex8600-util.i686.rpm
      they're in /usr/bin/PlxPex8725.eeprom.bin,
      /usr/bin/PlxPex8749.eeprom.bin

--smbusAddrSpacing
    By default, smbus accelerators are assumed to have base addresses that
    are 0x100 apart.  To change the spacing between accelerator address
    spaces, specify this argument.

  e.g.,
     PlxEeprom.py write 8725 
     PlxEeprom.py write 8749 
     PlxEeprom.py write 8725-epk-lc
     PlxEeprom.py write 8725-epk-fc
     PlxEeprom.py program 8725 /scd/0/3/0x38 /usr/share/PlxPex8725.eeprom.bin
     PlxEeprom.py program 8713 /scd/5/1/0x7 /mnt/flash/PlxPex8713.eeprom.bin
     PlxEeprom.py read 8725 /scd/0/3/0x38 
     PlxEeprom.py valid 8725 /scd/0/3/0x38
     PlxEeprom.py format 8725 /scd/0/3/0x38

     PlxEeprom.py -c readbin -b /mnt/flash/PlxPex8713.eeprom.bin # stdout
     PlxEeprom.py -c readbin -b /mnt/flash/PlxPex8713.eeprom.bin -t /tmp/text_file
     PlxEeprom.py -c writebin -b /tmp/output_bin_file -t /tmp/input_text_file

"""

parser = OptionParser(usage=usage)
parser.add_option( "-v", "--verbose", action="store_true",
                   help="enable verbose output" )

parser.add_option( "-b", "--binary", action="store",
                   help="binary file name" )

parser.add_option( "-t", "--text", action="store",
                   help="text file name" )

parser.add_option( "-c", "--command", action="store",
                   help="command readbin | writebin " )

parser.add_option( "--smbusAddrSpacing", action="store",
                    default='0x100', dest="smbusAddrSpacing",
                    help="Specify accelerator address offsets." )

( options, args ) = parser.parse_args()

verboseEnabled = options.verbose
binaryFile = options.binary 
textFile = options.text 
fileCommand = options.command 

if fileCommand == 'readbin':
   readBinaryEepromFile( binaryFile, textFile )
   sys.exit( 0 )   

if fileCommand == 'writebin':
   writeBinaryEepromFile( binaryFile, textFile )
   sys.exit( 0 )   

if len( args ) < 2:
   parser.error( "Too few arguments" )

op = args[0]
#print 'op:', op
if op not in ( "read", "write", "program", "valid", "format" ):
   parser.error( "Unknown argument1:" + args[0] )

chip = args[1]
#print 'chip:', chip
if chip not in chips:
   parser.error( "Unknown argument2:" + args[1] )

eepromFileName = eepromFiles[chip]

if op == "write":
   plxPrint( "No longer supported, use program" )
# pylint: disable-next=consider-using-in
elif op == "program" or op == "read" or op == "valid" or op == "format":
   if len( args ) < 3:
      parser.error( "Too few arguments" )
   smbusAddr = args[2]
   if op == "program":
      eepromFileName = args[3]
   devId = None
   try:
      # pylint: disable-next=redeclared-assigned-name
      ( i, i, accelId, busId, devId ) = smbusAddr.split( '/' )
   except ValueError:
      plxPrint ( "argument3: must be of the form /scd/0/3/0x38" )

   if devId is not None:
      if '0x' in devId:
         deviceId = int( devId, 16 )
      else:
         deviceId = int( devId, 10 )
      smbusAddress[chip] = { 'accelId' : int( accelId ), 
                             'busId' : int( busId ), 
                             'deviceId' : deviceId }

      helper = factory.device( smbusAddress[chip]['accelId'], 
                               smbusAddress[chip]['busId'], 
                               smbusAddress[chip]['deviceId'], 
                               addrSize, 'delay0',
                               smbusAddrSpacing=options.smbusAddrSpacing )
      if op == "valid":
         validTest = eepromValid()
         if validTest == "Valid":
            plxPrint( "Valid" )
         elif validTest == "Empty":
            plxPrint( "Empty" )
         else:
            plxPrint ( "Corrupted" )
      elif op == "format":
         eepromFormat()
      elif op == "program":
         if eepromWrite() is False:
            fatalQuit()
      else:
         eepromPrint()

