#!/bin/bash

err_notify() {
   # List of existing variables and their value.
   # 'set' lists the functions as well therefore we need to put the sub-shell in
   # posix mode with 'set -o posix' in order to get 'set' to print only variables.
   local variables="$(set -o posix; set)"
   local exitcode="$1" # Failed command's exit code.
   local lineno="$2"   # Failed command's line number.
   local filepath="$(realpath "$BASH_SOURCE")" # Absolute path of current script.
   local linecontent="$(sed -n "$lineno"p "$filepath")" # Failed command's line content.
   local detailedlogpath="/var/log/sysinit-error.log"
   local log="The following command terminated with a non zero exit status: \
${filepath}:${lineno}: $linecontent - exit code: $exitcode \
- Detailed shell environment at that time can be found in $detailedlogpath"

   logger -p local4.info -t sysinit "$log"

   # logger doesn't handle the large number of lines this early in the boot
   # process. Therefore we create a separate file with the shell environment
   # at the time of the error.
   cat >> "$detailedlogpath" << EOF
The following command terminated with a non zero exit status:
   ${filepath}:${lineno}: $linecontent
   exit code: $exitcode
Shell environment at that time:
$variables


EOF
}

# Use trap ERR to log any command that ends with a non 0 exit status. See BUG190356.
# There will be a log of the failure in syslog. The shell environment at the time
# of the error will be recorded in /var/log/sysinit-error.log.
trap 'err_notify $? $LINENO' ERR

# If the kernel cmdline option TEST_SYSINIT_ERR_TRAP=1 is present, execute an invalid
# command to trigger the ERR trap. This is used for testing by EosInit/SysinitErrTrapTest.
if grep -q 'TEST_SYSINIT_ERR_TRAP=1' /proc/cmdline; then
   # Variable used to check that the shell env is recorded.
   testVar="testValue"
   ls --invalidarg
fi

# Stuff from rc.sysinit we need to put somewhere

# Replace various configuration files with our own
for file in `find /etc/ -name \*.Eos -type f`; do
    cp -pf $file ${file%.Eos}
done

function set_flash_acl() {
   local path=$1

   setfacl -Rm mask:rwx "$path" > /dev/null 2>&1 || :
   setfacl -Rm g:eosadmin:rwx "$path" > /dev/null 2>&1 || :
   setfacl -Rdm g:eosadmin:rwx "$path" > /dev/null 2>&1 || :
   setfacl -Rdm o:rx "$path" > /dev/null 2>&1 || :
}

function check_flash_acl() {
   local getfaclOutput
   local getfaclDOutput
   local path=$1

   getfaclOutput=$( getfacl "$path" 2>/dev/null )
   getfaclDOutput=$( getfacl -d "$path" 2>/dev/null )

   # Only reinstall the ACL if it's missing
   if ! echo "$getfaclDOutput" | grep -qs 'group:eosadmin:rwx' || \
      ! echo "$getfaclOutput" | grep -qs '^group:eosadmin:rwx' || \
      ! echo "$getfaclOutput" | grep -qs '^mask::rwx' || \
      ! echo "$getfaclDOutput" | grep -qs '^other::r-x'; then
      # Install an ACL on the flash to guarantee that there is no difference for
      # the user between ext4 and vfat. It is necessary to do it in EOS because
      # even if it's persistent, Aboot doesn't have ACL support and can't
      # reinstall the ACL after a fullrecover has been done
      set_flash_acl "$path"
   fi
}

check_flash_acl /mnt/flash
check_flash_acl /mnt/drive
check_flash_acl /mnt/drive2
# /mnt/flash/{aboot,tpm-data} can be created by Aboot in some circumstances.
# If Aboot version lacks ACL support, it might be set in root but not
# in subdirectory/files. If /mnt/flash/aboot, /mnt/flash/aboot/update, or
# /mnt/flash/tpm-data is missing, we create the dir(s)
mkdir -p /mnt/flash/aboot/update
set_flash_acl /mnt/flash/aboot
mkdir -p /mnt/flash/tpm-data
set_flash_acl /mnt/flash/tpm-data

function set_timezone() {
    # Without actually parsing startup-config there are pathological cases
    # of multi-line commands that can fool the grep below.  In pratice
    # those pathological cases are not expected to occur and if they do
    # the SuperServer Timezone plugin will set the timezone to what is
    # configured in startup-config, resulting in a small cosmetic blip
    # at startup.
    tz_path=/usr/share/zoneinfo/posix
    startup_timezone=$(egrep 'clock +timezone +' /mnt/flash/startup-config \
    		   2> /dev/null | head -1 | cut -d' ' -f 3)
    # Ensure we have a valid timezone value
    if ! test -f $tz_path/$startup_timezone ; then
        startup_timezone=UTC
    fi
    # Set the system timezone
    rm -f /etc/localtime
    ln -s $tz_path/$startup_timezone /etc/localtime
}

# First approximation of the proper timezone
set_timezone
