#!/usr/bin/env python3
# Copyright (c) 2006-2010, 2011, 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=import-outside-toplevel
# pylint: disable=redefined-builtin
# pylint: disable=consider-using-f-string

import AaaDefs
from AaaInfoFile import UserInfoSynchronizer
from AaaPluginLib import TR_ERROR, TR_WARN, TR_AUTHEN, TR_AUTHZ, TR_ACCT
from AaaPluginLib import TR_SESSION, TR_INFO, TR_DEBUG, realTtyNameRe
import AaaHomeLinks
import Agent
import BothTrace
from BothTrace import traceX as bt
from BothTrace import Var as bv
import Cell
import Logging
import Plugins
import Tac
from Tracing import traceX
import UtmpDump
from PasswordPolicyLib import ApplyPolicyReactor

import collections
import grp
from hashlib import sha1
import operator
import os
import shutil
import sys
import threading
import queue

class NullMutex:
   def __init__( self ):
      pass

   def __enter__( self ):
      pass

   def __exit__( self, _type, value, _traceback ):
      pass

agent = None
acctQueue = None

_authenSessionExpiryTime = 300
_acctQueueMaxSize = 1024
AcctQueueElement = collections.namedtuple( 'AcctQueueElement',
                                           'user mlname acctMethod sendFunc' )
authenSourcePassword = Tac.Type( "Aaa::AuthenSource" ).authenSourcePassword

# Ignore those users (mainly for telnet since in other cases system users
# are not handled by us).
systemUsers = ( 'root', )

LOCKOUT_MESSAGE = ( "Account temporarily locked from remote access due to "
                    "too many consecutive failed login attempts" )

AAA_INVALID_AUTHN_METHODLIST = Logging.LogHandle(
      "AAA_INVALID_AUTHN_METHODLIST",
      severity=Logging.logError,
      fmt="Invalid authentication method list configured for "
          "service '%s': method '%s' is unknown",
      explanation="An attempt to authenticate a user for the service "
                  "failed because the method list configured for the "
                  "service contains a method name that does not match "
                  "any available authentication plugins.",
      recommendedAction="Fix the method list configuration." )

AAA_INVALID_AUTHZ_METHODLIST = Logging.LogHandle(
      "AAA_INVALID_AUTHZ_METHODLIST",
      severity=Logging.logError,
      fmt="Invalid authorization method list configured for "
          "action '%s': method '%s' is unknown",
      explanation="An attempt to authorize an action failed because "
                  "the method list configured for the action contains "
                  "a method name that does not match any available "
                  "authorization plugins.",
      recommendedAction="Fix the method list configuration." )

AAA_INVALID_ACCT_METHODLIST = Logging.LogHandle(
      "AAA_INVALID_ACCT_METHODLIST",
      severity=Logging.logError,
      fmt="Invalid accounting method list configured for "
          "action '%s': method '%s' is unknown",
      explanation="An attempt to account for an action failed because "
                  "the method list configured for the action contains "
                  "a method name that does not match any available "
                  "accounting plugins.",
      recommendedAction="Fix the method list configuration." )

AAA_AUTHN_PLUGIN_NOT_READY = Logging.LogHandle(
      "AAA_AUTHN_PLUGIN_NOT_READY",
      severity=Logging.logWarning,
      fmt="Authentication method '%s' is not ready",
      explanation="Authenticating a user with this authentication "
                  "method failed because the method is not ready.  "
                  "This can occur if the method has not been fully configured.",
      recommendedAction="Check to see if the authentication method has "
                        "been configured." )

AAA_AUTHZ_PLUGIN_NOT_READY = Logging.LogHandle(
      "AAA_AUTHZ_PLUGIN_NOT_READY",
      severity=Logging.logWarning,
      fmt="Authorization method '%s' is not ready",
      explanation="Authorizing an action using this authentication "
                  "method failed because the method is not ready.  "
                  "This can occur if the method has not been fully configured.",
      recommendedAction="Check to see if the authorization method has "
                        "been configured." )

AAA_ACCT_PLUGIN_NOT_READY = Logging.LogHandle(
      "AAA_ACCT_PLUGIN_NOT_READY",
      severity=Logging.logWarning,
      fmt="Accounting method '%s' is not ready",
      explanation="Accounting an action using this accounting "
                  "method failed because the method is not ready. "
                  "This can occur if the method has not been fully configured.",
      recommendedAction="Check to see if the accounting method has "
                        "been configured." )

AAA_HOMEDIR_SETUP_ERROR = Logging.LogHandle(
      "AAA_HOMEDIR_SETUP_ERROR",
      severity=Logging.logWarning,
      fmt="A problem occurred setting up the home directory for user %s",
      explanation="AAA creates a home directory at the time of a "
                  "user's first login and may populate it with some "
                  "settings files.  A problem occurred during this "
                  "process, and the user's home directory may not have "
                  "been created or may be partially populated.",
      recommendedAction="Log out and log back in.  Contact your "
                        "support representative if error persists." )

AAA_INVALID_AUTHENTYPE = Logging.LogHandle(
      "AAA_INVALID_AUTHENTYPE",
      severity=Logging.logError,
      fmt="Invalid authentication type '%s' requested",
      explanation="AAA received an authentication request containing "
                  "an unknown authentication type.",
      recommendedAction="Contact your support representative if error persists." )

AAA_INCOMPLETE_LOGINS_LIMIT = Logging.LogHandle(
      "AAA_INCOMPLETE_LOGINS_LIMIT",
      severity=Logging.logError,
      fmt="Too many incomplete login attempts",
      explanation="AAA has received too many incomplete login attempts "
                  "within the last %d minutes.  Login via console is "
                  "available, but login via ssh or telnet may fail "
                  "until some of the incomplete login sessions "
                  "expire." % ( _authenSessionExpiryTime // 60 ),
      recommendedAction="Investigate whether the device is being attacked.  "
                        "Attempt login via the console." )

AAA_USER_SESSION_LIMIT = Logging.LogHandle(
      "AAA_USER_SESSION_LIMIT",
      severity=Logging.logError,
      fmt="User %s has reached maximum session limit (%d)",
      explanation="The user has reached its configured session limit under "
                  "'management accounts'.",
      recommendedAction="Investigate whether the user has logged in too many times. "
                        "If the user is an automation acount, check if the "
                        "automation system is misbehaving." )

AAA_LOGIN = Logging.LogHandle(
      "AAA_LOGIN",
      severity=Logging.logNotice,
      fmt="user %s logged in [from: %s] [service: %s]",
      explanation="A user has logged in successfully",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

AAA_LOGOUT = Logging.LogHandle(
      "AAA_LOGOUT",
      severity=Logging.logNotice,
      fmt="user %s logged out [from: %s] [service: %s]",
      explanation="A user has logged out or the session has terminated",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

AAA_LOGIN_FAILED = Logging.LogHandle(
      "AAA_LOGIN_FAILED",
      severity=Logging.logWarning,
      fmt="user %s failed to login [from: %s] [service: %s] [reason: %s]",
      explanation="A user has failed to login to the switch",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

AAA_AUTHN_FALLBACK = Logging.LogHandle(
      "AAA_AUTHN_FALLBACK",
      severity=Logging.logWarning,
      fmt="Authentication method '%s' is currently unavailable; "
          "falling back to next method for service '%s' for user %s.",
      explanation="The authentication method failed to provide an "
                  "answer and is considered unavailable.  If the "
                  "method list for this service contains a fallback method, "
                  "this request will be retried via that method.",
      recommendedAction="Check the availability and reachability of "
                        "the authentication server(s)." )

AAA_AUTHZ_FALLBACK = Logging.LogHandle(
      "AAA_AUTHZ_FALLBACK",
      severity=Logging.logWarning,
      fmt="Authorization method '%s' is currently unavailable; "
          "falling back to next method for action '%s' for user %s.",
      explanation="The authorization method failed to provide an "
                  "answer and is considered unavailable.  If the "
                  "method list for this service contains a fallback method, "
                  "this request will be retried via that method.",
      recommendedAction="Check the availability and reachability of "
                        "the authorization server(s)." )

AAA_ACCT_FALLBACK = Logging.LogHandle(
      "AAA_ACCT_FALLBACK",
      severity=Logging.logWarning,
      fmt="Accounting method '%s' is currently unavailable; "
          "falling back to next method for action '%s' for user %s.",
      explanation="The accounting method failed to provide an "
                  "answer and is considered unavailable. If the "
                  "method list for this service contains a fallback method, "
                  "this request will be retried via that method.",
      recommendedAction="Check the availability and reachability of "
                        "the accounting server(s)." )

AAA_ACCT_QUEUE_FULL = Logging.LogHandle(
      "AAA_ACCT_QUEUE_FULL",
      severity=Logging.logWarning,
      fmt="Accounting queue is full; the oldest %d messages will be dropped.",
      explanation="There were too many messages in the accounting queue, "
                  "and the switch discarded some pending messages to "
                  "avoid consuming too many resources. It normally means "
                  "that the server was slow at taking the accounting messages.",
      recommendedAction="Check the availability and reachability of "
                        "the accounting server(s)." )

AAA_ACCT_MSG_DROP = Logging.LogHandle(
      "AAA_ACCT_MSG_DROP",
      severity=Logging.logWarning,
      fmt="No available server; accounting messages will be dropped.",
      explanation="None of the accounting servers are responsive, and the "
                  "switch will drop messages until the error is rectified.",
      recommendedAction="Check the availability and reachability of "
                        "the accounting server(s)." )

AAA_ACCT_MSG_RESUME = Logging.LogHandle(
      "AAA_ACCT_MSG_RESUME",
      severity=Logging.logWarning,
      fmt="Resumed successful sending of accounting messages.",
      explanation="Accounting message successfully sent as the switch is "
                  "able to contact the accounting server(s).",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

AAA_HOMEDIR_PERSISTENT_FILE_ERROR = Logging.LogHandle(
      "AAA_HOMEDIR_PERSISTENT_FILE_ERROR",
      severity=Logging.logWarning,
      fmt=( "A problem occurred setting up persistent file links in the home "
            "directory for user %s (errors: %d)" ),
      explanation=( "AAA creates links in user's home directory pointing to "
                    "persistent files under flash:/home/<user> when a user "
                    "logs in. A problem occurred during this process, and the "
                    "user's home directory may not have created all links to "
                    "the persistent files." ),
      recommendedAction="Please log out and log back in.  Contact your "
                        "support representative if error persists." )

class ResponseMgr:
   """This class maintains username/password responses for fallback use.
   The basic idea is that we keep track of the latest promptUser and promptPassword
   responses during a session. If we fallback to another method, then we enable
   the previously saved responses so they can be automatically given to the new
   method if it asks.
   """
   def __init__( self, promptTypes ):
      self.types = promptTypes
      self.autoFeed = { }
      self.expected = { }
      self.response = { }
      for t in promptTypes:
         self.autoFeed[ t ] = False
         self.expected[ t ] = False
         self.response[ t ] = None

   # called at fallback time
   def enableAutoFeed( self ):
      for t in self.types:
         self.expected[ t ] = False
         if self.response[ t ] is not None:
            self.autoFeed[ t ] = True

   # pylint: disable-next=pointless-string-statement
   """Process a message returned by authenticate(),
   and possibly return an auto response"""
   # pylint: disable-next=inconsistent-return-statements
   def processAuthenMessages( self, messages ):
      if len( messages ) != 1:
         return
      for t in self.types:
         if messages[ 0 ].style == t:
            # convert to tacc recognizable styles
            messages[ 0 ].style = self.types[ t ]
            if self.autoFeed[ t ]:
               bt( TR_AUTHEN, bv( t ), "auto-feed response",
                   bv( '*' if t == 'promptPassword' else self.response[ t ] ) )
               # disable autofeed - we only do it once per method
               self.autoFeed[ t ] = self.expected[ t ] = False
               return ( self.response[ t ], )
            else:
               # otherwise, we need to remember the next response
               self.expected[ t ] = True
               return

   # process a new response from pam
   def processAuthenResponses( self, *responses ):
      if len( responses ) != 1:
         return
      for t in self.types:
         if self.expected[ t ]:
            self.response[ t ] = responses[ 0 ]
         self.expected[ t ] = False

class AuthenSession:
   poolStatusIdle = 0  # session is waiting for more input, not in use
   poolStatusBusy = 1  # session is in the middle of continueAuthenticate()
   poolStatusAbort = 2 # session is aborted or expired while in the middle of
                       # continueAuthenticate()

   def __init__( self, id, type, service, tty, remoteHost, remoteUser, user,
                 privLevel, sessionId, localAddr=None, localPort=None,
                 remotePort=None, sshPrincipal=None, authenSource='' ):
      self.id = id
      self.methodName = None
      self.authenticator = None
      self.nextMethodIndex = 0
      self.type = type
      self.service = service or ""
      self.tty = tty or ""
      self.remoteHost = remoteHost or ""
      self.remoteUser = remoteUser or ""
      self.privLevel = privLevel
      self.authToken = ""
      self.state = None
      self.user = user
      self.response = ResponseMgr( { 'promptUser': 'promptEchoOn',
                                     'promptPassword': 'promptEchoOff' } )
      self.localAddr = localAddr
      self.localPort = localPort
      self.remotePort = remotePort
      self.sshPrincipal = sshPrincipal
      self.authenSource = authenSource
      # this sessionId is different from the 'id' member
      # this is the one passed in startAuthenticate, openSession, etc
      # the idea is that we use unique IDs for each authentication session
      # (login, enable), but use the same sessionId for the same CLI session.
      self.sessionId = sessionId
      self.created = Tac.now()
      # The following field is used to avoid race conditions between
      # an authenSession under continueAuthenticate() and another thread
      # trying to abort/remove it. It is accessed with certain mutex held
      # (Aaa.mutex).
      # There is already a 'state' member so use a name more distinguishable
      # than 'status' etc.
      self.poolstatus = self.poolStatusIdle
      # record which server authenticated the user
      self.authenServer = None

   def setAuthenticator( self, authenticator ):
      self.authenticator = authenticator

   def setMethod( self, method ):
      self.methodName = method

   def logFallbackMessage( self ):
      if self.authenticator.logFallback:
         Logging.log( AAA_AUTHN_FALLBACK, self.methodName, self.service, self.user )

   def loginFailureReason( self, state ):
      '''Generate the LOGIN_FAILED reason string.'''
      reasons = []
      for m in ( state.message0, state.message1, 
                 state.message2, state.message3 ):
         if m.style == 'error' and m.text:
            reasons.append( m.text )
      if reasons:
         return 'Authentication failed - ' + ','.join( reasons )

      # no error, let's make up one
      if state.status == 'fail':
         return 'Authentication failed'
      elif state.status == 'unavailable':
         return 'Authentication unavailable'
      else:
         return 'Unknown authentication failure'

   def updateState( self, state ):
      assert self.id == state.id
      self.state = state
      # Sometimes the plugin returns "" for user in case of a failure,
      # which would break fallback when we need to provide a username.
      # Instead of auditing all plugins not to do this, we instead do
      # not update our user member if it's not a valid username.
      if state.user:
         self.user = state.user
      if state.authToken:
         self.authToken = state.authToken

   def expired( self ):
      """I consider myself expired if I am older than
      _authenSessionExpiryTime.  No supported authentication method
      should take this long to complete"""
      age = Tac.now() - self.created
      return age > _authenSessionExpiryTime

class SessionIdPool:
   """The motivation for the SessionIdPool is the possible race where Aaa
   allocates a session id and increments Aaa::Config::nextSessionId and hands
   the session id to a requester, but Aaa dies before sending the entity log
   message to Sysdb containing the change to nextSessionId.  By allocating many
   session ids in each Sysdb interaction and by refilling the pool long before
   necessary we make it very unlikely for this race to occur in practice."""
   def __init__( self, aaaStatus, mutex, maxSize=1000 ):
      self.aaaStatus = aaaStatus
      self.mutex = mutex
      self.maxSize = maxSize
      self.start = 0
      self.end = 0

   def _growPool( self ):
      # self.mutex must be held
      currSize = self.end - self.start
      if currSize < self.maxSize:
         growBy = self.maxSize - currSize
         if self.start == 0:
            # Special case for first time
            assert self.end == 0
            self.start = self.aaaStatus.nextSessionId
            self.end = self.start + ( growBy - 1 )
            self.aaaStatus.nextSessionId = ( self.end + 1 )
         elif ( self.end + 1 ) == self.aaaStatus.nextSessionId:
            self.aaaStatus.nextSessionId += growBy
            self.end += growBy
         else:
            # What happened?  Multiple writers to nextSessionId?
            sys.stderr.write( "Unexpected error: another writer modified "
                              "nextSessionId.\n" )
            sys.exit( 1 )

   def allocate( self ):
      """Allocates a new SessionId and returns it."""
      with self.mutex:
         if self.end - self.start < self.maxSize // 2:
            self._growPool()
         id = self.start
         self.start += 1
         return id

class AaaConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Aaa::Config"

   def __init__( self, entity, agent ): # pylint: disable=redefined-outer-name
      self.agent_ = agent
      Tac.Notifiee.__init__( self, entity )

   @Tac.handler( 'hostgroup' )
   def handleHostGroup( self, name=None ):
      traceX( TR_INFO, "hostgroup", name, "changed" )
      self.agent_.invalidateAllSessionPools( name )

class AaaCounterConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Aaa::CounterConfig"

   def __init__( self, entity, agent ): # pylint: disable=redefined-outer-name
      self.agent_ = agent
      Tac.Notifiee.__init__( self, entity )

   @Tac.handler( 'clearCounterRequestTime' )
   def handleClearCounterRequestTime( self ):
      self.agent_.clearCounters()
      self.agent_.status.counters.lastClearTime = Tac.now()

class AaaUserLockoutConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Aaa::UserLockoutConfig"

   def __init__( self, entity, agent ): # pylint: disable=redefined-outer-name
      self.agent_ = agent
      Tac.Notifiee.__init__( self, entity )

   @Tac.handler( 'clearLockoutRequest' )
   def handleClearLockoutRequestTime( self, user ):
      # We only care about users being added to this collection
      if user not in self.notifier_.clearLockoutRequest:
         return
      requestTime = self.notifier_.clearLockoutRequest[ user ]
      # delete entry if lastFailTime is less than request time 
      # (in case of stale requests)
      status = self.agent_.status.userLockoutStatus
      userLockoutStatus = status.get( user )
      if userLockoutStatus and userLockoutStatus.lastFailedLogin <= requestTime:
         self.agent_.unlockAccount( user )

   @Tac.handler( 'clearAllLockoutRequest' )
   def handleClearAllLockoutRequest( self ):
      requestTime = self.notifier_.clearAllLockoutRequest
      # Delete all entries older than the request time
      for user, status in self.agent_.status.userLockoutStatus.items():
         if status.lastFailedLogin <= requestTime:
            self.agent_.unlockAccount( user )

class HostgroupConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Aaa::HostGroup"

   def __init__( self, entity, agent ): # pylint: disable=redefined-outer-name
      self.agent_ = agent
      Tac.Notifiee.__init__( self, entity )
   
   @Tac.handler( 'member' )
   def handleMember( self, index=None ):
      traceX( TR_INFO, "hostgroup", self.notifier_.name, "changed" )
      self.agent_.invalidateAllSessionPools( self.notifier_.name )

def makePwEntry( username, shell, authenMethod=None ):
   # Fills out some entries in PasswdEntry. Leaves out 
   # userId and groupId to be filled in by Aaa.py later.
   ent = Tac.Value( "AaaApi::PasswdEntry" )
   ent.userName = username
   if authenMethod:
      ent.realName = f"Eos-{username} ({authenMethod})"
   else:
      ent.realName = "Unknown User (%s)" % ( username )
   ent.homeDir = "/home/%s" % username
   ent.shell = shell
   return ent

AAA_SYSACCT_WAIT = int( os.environ.get( 'AAA_SYSACCT_WAIT', '300' ) )

class Aaa( Agent.Agent ):
   # The maximum number of authentication sessions allowed to be in-progress
   # simultaneously.  Aaa must keep some state for each session, and so this
   # limit is necessary to avoid a denial-of-service attack by attempting many
   # simultaneous logins.  If this limit is exceeded any new authentication
   # sessions will fail until some of the existing sessions expire.
   maxAuthenSessions = 200
   # The maximum number of pending login sessions -- opened after successful
   # authentication but for which openSession has not been called -- that can
   # exist simultaneously.
   maxPendingSessions = 20
   # Wait time for a session to be in pending state (which also includes when
   # the session is established but a shell has not started, so it's a bit of
   # a misnomer).
   maxPendingTimeout = 300

   def __init__( self, entityManager ):
      traceX( TR_INFO, "Starting Aaa agent" )
      self.mutex = threading.RLock()
      # pending authentication sessions indexed by id
      self.lockoutMutex = threading.RLock() # login attempts indexed by username
      self.authenSessions = {}
      self.authenSessionsLastPurgeTime = 0
      self.config = None
      self.counterConfig = None
      self.userLockoutConfig = None
      self.status = None
      self.mgmtAcctConfig = None
      self.mgmtSecConfig = None
      self.localUserConfig = None
      self.cliConfig = None
      self.aaaConfigReactor_ = None
      self.aaaCounterConfigReactor_ = None
      self.aaaUserLockoutConfigReactor_ = None
      self.hostgroupReactor_ = None
      self.denyLastReactor_ = None
      self.applyPolicyReactor_ = None
      self.acctDispatcherThread_ = None
      self.plugins = {}
      self.pluginDir = None
      self.sessionIdPool = None
      self.acctCallBackTimer = None
      self.userInfoSynchronizer = None
      self.initialized = False
      self.aaaStreamState = {}
      self.groupId = None
      self.consoleConfig = None
      self.matchListConfig = None

      try:
         self.groupId = grp.getgrnam( "eosadmin" ).gr_gid
      except KeyError:
         self.groupId = None

      qtfile = "Aaa%s.qt" % ( "-%d" if "QUICKTRACEDIR" not in os.environ else "" )
      BothTrace.initialize( qtfile, "64,32,128,32,32,32,32,1,1,32" )

      # make sure we use the right PyClient lib for pam/nss

      # pkgdeps: rpm AaaPam-lib
      from ctypes import CDLL
      LIBPAMAPI = CDLL( "libAaaPamApi.so" )
      LIBPAMAPI.aaaRpcLoadTacPyclient( sys.version_info.major )

      Agent.Agent.__init__( self, entityManager )

   def doInit( self, entityManager ):
      mountGroup = entityManager.mountGroup()
      self.config = mountGroup.mount( 'security/aaa/config', 'Aaa::Config', 'r' )
      self.counterConfig = mountGroup.mount( 'security/aaa/counterConfig',
                                             'Aaa::CounterConfig', 'r' )
      self.userLockoutConfig = mountGroup.mount( 'security/aaa/userLockoutConfig',
                                                 'Aaa::UserLockoutConfig', 'r' )
      self.status = mountGroup.mount( Cell.path( 'security/aaa/status' ), 
                                      'Aaa::Status', 'wf' )
      self.mgmtAcctConfig = mountGroup.mount( 'mgmt/acct/config',
                                              'Mgmt::Account::Config', 'r' )
      self.cliConfig = mountGroup.mount( 'cli/input/aaa',
                                         'Cli::AaaCliConfig', 'w' )
      self.mgmtSecConfig = mountGroup.mount( 'mgmt/security/config',
                                             'Mgmt::Security::Config', 'r' )
      self.localUserConfig = mountGroup.mount( "security/aaa/local/config",
                                               "LocalUser::Config", "r" )
      self.aaaStreamState[ "login" ] = mountGroup.mount(
            Cell.path( 'security/aaa/stream/credentialz/console' ),
            'AaaStream::ConsoleState',
            'wf' )
      self.aaaStreamState[ "sshd" ] = mountGroup.mount(
            Cell.path( 'security/aaa/stream/credentialz/sshServer' ),
            'AaaStream::SshServerState',
            'wf' )

      self.consoleConfig = mountGroup.mount( "mgmt/console/config",
                                             "Mgmt::Console::Config", "r" )

      self.matchListConfig = mountGroup.mount( "matchlist/config/cli",
                                               "MatchList::Config", "r" )

      def _finish():
         # XXX should we just use handleMaster callback ?
         if not entityManager.locallyReadOnly():
            self.cliConfig.aaaProvider = "Aaa"
         # Create reactor to various entities that might affect
         # plugin session pools.
         self.aaaConfigReactor_ = AaaConfigReactor( self.config, self )
         self.aaaCounterConfigReactor_ = \
             AaaCounterConfigReactor( self.counterConfig, self )
         self.aaaUserLockoutConfigReactor_ = \
             AaaUserLockoutConfigReactor( self.userLockoutConfig, self )
         self.hostgroupReactor_ = Tac.collectionChangeReactor(
            self.config.hostgroup, HostgroupConfigReactor,
            reactorArgs=( self, ) )
         self.applyPolicyReactor_ = ApplyPolicyReactor( self.mgmtAcctConfig, self )
         global agent, acctQueue
         agent = self
         self.status.counters.numPendingAcctRequests = 0
         acctQueue = AcctQueue( maxsize = _acctQueueMaxSize )
         self.acctDispatcherThread_ = AcctDispatcher()
         self.acctDispatcherThread_.daemon = True
         self.acctDispatcherThread_.start()
         Ctx = collections.namedtuple( 'Ctx', 'entityManager aaaAgent' )
         mountGroup2 = entityManager.mountGroup()
         self.pluginDir = Plugins.loadPlugins( "AaaPlugin", context=\
                   Ctx( entityManager=entityManager, aaaAgent=self ) )
         for p in self.pluginDir.plugins():
            traceX( TR_INFO, "Loaded AaaPlugin for method", p.name )
            self.plugins[ p.name ] = p
         self.sessionIdPool = SessionIdPool( self.status, self.mutex )
         self.userInfoSynchronizer = UserInfoSynchronizer( self )
         mountGroup2.close( _finishPluginsMount )

      mountGroup.close( _finish )

      def _finishPluginsMount():
         if entityManager.redundancyStatus().mode == 'active':
            if not self.status.reloadStatusAcctGenerated:
               self.status.reloadStatusAcctGenerated = True
               def sendLog():
                  self.sendSystemAcct( 'system restart', 'start' )
               self.acctCallBackTimer = Tac.ClockNotifiee( handler=sendLog )
               self.acctCallBackTimer.timeMin = Tac.now() + AAA_SYSACCT_WAIT
            else:
               self.sendSystemAcct( 'Aaa restart', 'start' )
         else:
            self.status.reloadStatusAcctGenerated = True

         self.initialized = True
         traceX( TR_INFO, "Aaa agent initialized" )

   def warm( self ):
      # We are warm only when everything is initialized
      return self.initialized and not self.userInfoSynchronizer.syncPending()

   def clearCounters( self ):
      self.status.counters.authnSuccess = 0
      self.status.counters.authnFail = 0
      self.status.counters.authnUnavailable = 0
      self.status.counters.authzAllowed = 0
      self.status.counters.authzDenied = 0
      self.status.counters.authzUnavailable = 0
      self.status.counters.acctSuccess = 0
      self.status.counters.acctError = 0

   def incAuthnCounter( self, status, service=None ):
      if status == 'success':
         self.status.counters.authnSuccess += 1
         if service in self.aaaStreamState:
            self.aaaStreamState[ service ].counters.accessAccepts += 1
            self.aaaStreamState[ service ].counters.lastAccessAccept = Tac.utcNow()
      elif status == 'fail':
         self.status.counters.authnFail += 1
         if service in self.aaaStreamState:
            self.aaaStreamState[ service ].counters.accessRejects += 1
            self.aaaStreamState[ service ].counters.lastAccessReject = Tac.utcNow()
      # pylint: disable-next=consider-using-in
      elif status == 'unknown' or status == 'unavailable':
         self.status.counters.authnUnavailable += 1

   def incAuthzCounter( self, status ):
      if status == 'allowed':
         self.status.counters.authzAllowed += 1
      elif status == 'denied':
         self.status.counters.authzDenied += 1
      elif status == 'authzUnavailable':
         self.status.counters.authzUnavailable += 1
      elif status == 'aborted':
         self.status.counters.authzAborted += 1

   def invalidateAllSessionPools( self, hostgroup ):
      bt( TR_WARN, "invalidate all plugins' session pools for hostgroup",
          bv( hostgroup ) )
      for p in self.plugins.values():
         p.invalidateSessionPool( hostgroup )

   def allocateAuthenSessionId( self ):
      return self.sessionIdPool.allocate()

   def applyPluginResult( self, result, authenState ):
      """Applies the result from a call to a plugin's Authenticator instance's
      authenticate method to an AaaApi::AuthenState instance."""
      status = result.get( 'status' )
      try:
         authenState.status = status
      except: # pylint: disable=bare-except
         bt( TR_ERROR, "invalid status", bv( status ), "returned by plugin" )
         authenState.status = 'fail'

      messages = result.get( 'messages', [] )

      for idx in range( 0, 4 ):
         if idx < len( messages ):
            m = messages[ idx ]
         else:
            m = Tac.Value( "AaaApi::AuthenMessage", style='invalid' )
         attr = "message%d" %( idx )
         setattr( authenState, attr, m )

      authenState.user = result.get( 'user', "" )
      authenState.authToken = result.get( 'authToken', "" )
      if authenState.user in systemUsers:
         # force system users to return 'unknown users'
         authenState.status = 'unknown'

   def generateUid( self, username ):
      hashstr = username
      while True:
         # Get a consistent hash for a given string everytime
         uid = 0x7ffffff & int( sha1( hashstr.encode() ).hexdigest(), 16 )
         # We only allocate and handle authentication for uid >= 1500
         # to avoid colliding with system users as well as users created
         # by useradd (which by default starts with 1000).
         if ( uid in self.status.userid ) or ( uid < 1500 ):
            bt( TR_ERROR, "Hash collision or uid < 1500 for uid ", bv( uid ),
                "so rehashing" )
            hashstr = "%s-%d" %( hashstr, uid )
            continue
         break
      return uid

   def isLocalAuthenMethod( self, authenMethod ):
      # There are two cases:
      # 1. the authen method is local
      # 2. the authen method is set from createSession() without
      #    going through authentication (e.g., sshkey). To detect this
      #    we see if there is a plugin that handles this method; if not,
      #    treat it as local.
      if authenMethod == 'local':
         return True
      plugin = self.pluginForMethod( authenMethod )
      return plugin is None

   def setUpNewUser( self, user, authenMethod='' ):
      # Requires holding agent mutex
      uid = self.generateUid( user )
      bt( TR_SESSION, "new user", bv( user ), "uid", bv( uid ) )
      acct = self.status.account.newMember( user, uid, authenMethod )
      self.status.userid.addMember( acct )
      self.userInfoSynchronizer.scheduleSync()
      self.ensureHomeDirExists( user )

   def cleanUpUser( self, user ):
      # Requires holding agent mutex
      #
      # Right now, we only clean up local user status.
      acct = self.status.account.get( user )
      if acct and self.isLocalAuthenMethod( acct.authenMethod ):
         bt( TR_SESSION, "clean up user", bv( user ) )
         del self.status.userid[ acct.id ]
         del self.status.account[ acct.userName ]
         # existing sessions need to be fixed
         sessionsToDelete = []
         for session in self.status.session.values():
            if session.account == acct:
               if session.state == 'established':
                  bt( TR_SESSION, "removing account for session", bv( session.id ) )
                  session.account = None
               else:
                  bt( TR_SESSION, "removing session", bv( session.id ),
                      "for deleted user", user )
                  sessionsToDelete.append( session.id )
         for sessionId in sessionsToDelete:
            del self.status.session[ sessionId ]
         self.userInfoSynchronizer.scheduleSync()

   def handleLogin( self, authenSession ):
      user = authenSession.user
      bt( TR_INFO, "handleLogin for user", bv( user ), "via",
          bv( authenSession.methodName ) )
      accounts = self.status.account
      with self.mutex:
         acct = accounts.get( user )
         if acct:
            if acct.authenMethod != authenSession.methodName:
               acct.authenMethod = authenSession.methodName
               # We don't have a reactor to individual members in status.accounts,
               # so we just do it manually here.
               self.userInfoSynchronizer.scheduleSync()
         else:
            self.setUpNewUser( user, authenSession.methodName )

      # BUG637168: wait for user info file to be generated
      Tac.waitFor( lambda: not self.userInfoSynchronizer.syncPending(),
                   warnAfter=None, description="user info to be flushed",
                   sleep = not Tac.activityManager.inExecTime.isZero )

      self.updateLinks( user )

   def ensureHomeDirExists( self, userName ):
      assert userName
      userId, groupId = self.getUserIdGroupId( userName )
      homeDir = f"/home/{userName}"

      if not os.path.exists( homeDir ):
         e = self.createHomeDir( homeDir, userId, groupId )
         if e != 0:
            Logging.log( AAA_HOMEDIR_SETUP_ERROR, userName )
            bt( TR_ERROR, "Error while setting up account for ", bv ( userName ),
                  ". errcount=", bv( e ) )
      else:
         # home dir exists, but it might have incorrect ownership
         # since uid of this user is re-generated on each restart.
         e = AaaHomeLinks.chownUsersFiles( homeDir, userId, groupId )
         if e != 0:
            Logging.log( AAA_HOMEDIR_SETUP_ERROR, userName )
            bt( TR_ERROR, "Error(s) changing file ownership for ",
                  bv ( userName ), ". errcount=", bv( e ) )

   def updateLinks( self, userName ):
      # We need to call createHomeLinks at every login. Its possible that
      # user had files on flash before the reboot. Post reboot,
      # Aaa just created /home so we need to create symlinks

      fsRoot = os.environ.get( 'FILESYSTEM_ROOT', '/mnt' )
      flashDir = os.path.join( fsRoot, 'flash', 'home', userName )

      if not os.path.isdir( flashDir ):
         return

      userId, groupId = self.getUserIdGroupId( userName )

      e = AaaHomeLinks.createHomeLinks( userName, userId, groupId )
      if e != 0:
         Logging.log( AAA_HOMEDIR_PERSISTENT_FILE_ERROR, userName, e )
         bt( TR_ERROR, "Error setting up symlinks for: ", bv( userName ),
               ". errcount=", bv( e ) )

   def createHomeDir( self, dirpath, uid, gid ):
      bt( TR_AUTHEN, "Creating home", bv( dirpath ), "for uid", bv( uid ),
          "gid", bv( gid ) )
      errors = 0
      try:
         os.makedirs( dirpath, mode=0o750 )
      except OSError as e:
         bt( TR_ERROR, "Failed to create home:", bv( dirpath ), ":",
             bv( e.strerror ) )
         errors += 1
      try:
         os.chown( dirpath, uid, gid )
      except OSError as e:
         bt( TR_ERROR, "Failed to chown home:", bv( dirpath ), ":",
             bv( e.strerror ) )
         errors += 1
      errors += self.copySkel( dirpath, uid, gid )
      return errors

   def copySkel( self, homedir, uid, gid ):
      traceX( TR_AUTHEN, "Copying files in /etc/skel to home dir", homedir )
      errors = 0
      srcTop = "/etc/skel/"
      prefixLen = len( srcTop )
      for skeldir, subdirs, files in os.walk( srcTop ):
         relativeDir = skeldir[ prefixLen : ]
         dstdir = os.path.join( homedir, relativeDir )
         for d in subdirs:
            src = os.path.join( skeldir, d )
            dst = os.path.join( dstdir, d )
            traceX( TR_AUTHEN, "Creating", dst )
            try:
               os.makedirs( dst, mode=0o750 )
               shutil.copystat( src, dst )
            except OSError as e:
               bt( TR_ERROR, "Failed to create dir", bv( dst ), ":",
                   bv( e.strerror ) )
               errors += 1
         for f in files:
            if f[ -4 : ] == ".Eos":
               traceX( TR_INFO, "Skipping", f )
               continue
            src = os.path.join( skeldir, f )
            dst = os.path.join( dstdir, f )
            traceX( TR_INFO, "Copying", src, "to", dst )
            try:
               shutil.copyfile( src, dst )
               shutil.copymode( src, dst )
               os.chown( dst, uid, gid )
            except OSError as e:
               bt( TR_ERROR, "Failed to copy", bv( src ), "to", bv( dst ),
                   ":", bv( e.strerror ) )
               errors += 1
      return errors

   def pluginForMethod( self, method ):
      plugin = self.plugins.get( method )
      if plugin:
         traceX( TR_INFO, "plugin found by name for method", method )
         return plugin
      # Perhaps one of the plugins has an authenMethod regex that will
      # match method.
      for p in self.plugins.values():
         if p.handlesAuthenMethod( method ):
            traceX( TR_INFO, "plugin found to handle method", method )
            return p
      return None

   def maybePurgeExpiredAuthenSessions( self ):
      # self.mutex must be held
      now = Tac.now()
      if ( len( self.authenSessions ) < self.maxAuthenSessions and
           ( now < self.authenSessionsLastPurgeTime + _authenSessionExpiryTime ) ):
         # purge every once a while, or when we reach the limit
         return

      self.authenSessionsLastPurgeTime = now
      expired = set()
      for k, v in self.authenSessions.items():
         if v.expired():
            if v.poolstatus == v.poolStatusIdle:
               bt( TR_WARN, "purging expired session with id", bv( k ) )
               expired.add( k )
            else:
               # this is either inuse or abort, just set a flag
               bt( TR_WARN, "aborting expired session with id", bv( k ) )
               v.poolstatus = v.poolStatusAbort
      for k in expired:
         del self.authenSessions[ k ]

   def addAuthenSession( self, authenSession ):
      with self.mutex:
         self.maybePurgeExpiredAuthenSessions()
         if len( self.authenSessions ) >= self.maxAuthenSessions:
            Logging.log( AAA_INCOMPLETE_LOGINS_LIMIT )
            return False
         self.authenSessions[ authenSession.id ] = authenSession
         return True

   def removeOlderSessionsWithSameTty( self, session ):
      # If this is a real tty, remove dead sessions with the same tty
      # Other ttys are OK to share (e.g., "ssh" or "command-api" ).
      if not realTtyNameRe.match( session.tty ):
         return
      sessionsToDelete = [ s for s in self.status.session.values()
                           if s != session and s.tty == session.tty and
                           s.startTime <= session.startTime ]
      for sess in sessionsToDelete:
         bt( TR_WARN, "purging session with tty", bv( sess.tty ),
             "and id", bv( sess.id ) )
         if sess.sessionPid:
            del self.status.pidToSession[ sess.sessionPid ]
         del self.status.session[ sess.id ]

   def purgeOldPendingSessions( self ):
      # self.mutex must be held
      pendingSessions = [ s for s in self.status.session.values()
                          if s.state == 'pending' ]
      pendingSessions.sort( key=operator.attrgetter( 'startTime' ) )

      minSessionsToDelete = max( len( pendingSessions ) - self.maxPendingSessions,
                                 0 )

      cutoffTime = Tac.now() - self.maxPendingTimeout
      purgeCount = 0

      for sess in pendingSessions:
         if purgeCount < minSessionsToDelete or sess.startTime < cutoffTime:
            bt( TR_WARN, "purging pending session with id", bv( sess.id ) )
            del self.status.session[ sess.id ]
            purgeCount += 1
         else:
            break

   # pylint: disable-next=inconsistent-return-statements
   def maybeAddPendingSession( self, authenSession, sessionData ):
      if authenSession.type == 'authnTypeEnable':
         # The Cli never opens/closes sessions, so don't create pending
         # sessions for enable requests.
         return
      traceX( TR_AUTHEN, "Adding pending session for user:", authenSession.user,
              "service:", authenSession.service, "method:", authenSession.methodName,
              "session id:", authenSession.id, "session data:", sessionData )
      with self.mutex:
         self.purgeOldPendingSessions()
         s = self.status.session.newMember( authenSession.id )
         s.account = self.status.account[ authenSession.user ]
         s.startTimeInUtc = Tac.utcNow()
         s.startTime = Tac.now()
         s.authenMethod = authenSession.methodName
         s.service = authenSession.service
         s.tty = authenSession.tty
         s.remoteHost = authenSession.remoteHost
         s.remoteUser = authenSession.remoteUser
         if authenSession.authenServer:
            s.authenServer = authenSession.authenServer
         self.createSessionData( s, sessionData )
         s.state = 'pending'
         s.localAddr = authenSession.localAddr or ""
         s.localPort = authenSession.localPort or 0
         s.remotePort = authenSession.remotePort or 0
         s.sshPrincipal = authenSession.sshPrincipal or ""
         s.authenSource = authenSession.authenSource
         self.removeOlderSessionsWithSameTty( s )
      return True

   def getShellFromAccount( self, account ):
      # treat it as local authentication by default (e.g., sshkey
      # users have other method name).
      plugin = ( self.pluginForMethod( account.authenMethod ) or
                 self.pluginForMethod( "local" ) )
      return plugin.getUserShell( account.userName ) or self.config.shell

   def maybeLogFailure( self, authenSession, state ):
      traceX( TR_WARN, "maybeLogFailure", state.status )
      user = authenSession.user
      if ( authenSession.type == 'authnTypeLogin' and agent and agent.config and
           user not in systemUsers ):
         config = agent.config
         if state.status in ( 'fail',
                              'unavailable',
                              'unknown' ):
            if config.loggingOnFailure:
               Logging.log( AAA_LOGIN_FAILED, user,
                            authenSession.remoteHost,
                            authenSession.service,
                            authenSession.loginFailureReason( state ) )
            else:
               bt( TR_ERROR, "user", bv( user ), "login failed from",
                   bv( authenSession.remoteHost ),
                   bv( authenSession.service ),
                   bv( authenSession.loginFailureReason( state ) ) )

            # Accounting for failed authentication
            self.sendFailedShellAcct( authenSession.user, authenSession.localAddr,
                                      authenSession.localPort,
                                      authenSession.remoteHost,
                                      authenSession.remotePort, authenSession.tty,
                                      authenSession.privLevel,
                                      authenSession.sshPrincipal,
                                      authenSession.authenSource,
                                      authenSession.loginFailureReason( state ) )

         # For remote logins, keep track of number of consecutive failed logins
         # and user's last failed login
         if ( state.status == 'fail' and authenSession.service != 'login' and
              config.lockoutTime != config.lockoutDisabled ):
            traceX( TR_WARN, "incrementing account lockout counter" )
            currTime = Tac.now()
            with self.lockoutMutex:
               userLockStatus = self.status.newUserLockoutStatus( user )
               if currTime > userLockStatus.lastFailedLogin + config.lockoutWindow:
                  userLockStatus.numFailedLogins = 1
               else:
                  userLockStatus.numFailedLogins += 1
               userLockStatus.lastFailedLogin = currTime

   def findAuthzMethodList( self, name, service ):
      cfg = self.config
      authzMethod = cfg.authzMethod.get( name )
      if not authzMethod:
         return None
      defaultMethod = authzMethod.defaultMethod
      if service == 'login':
         consoleMethod = authzMethod.consoleMethod
         if consoleMethod:
            return consoleMethod
         elif not cfg.consoleAuthz:
            # if consoleAuthz is disabled, allow everything
            return { 0: 'none' }
      return defaultMethod          

   def findAuthenMethodList( self, type, service ):
      cfg = self.config
      if type == 'authnTypeLogin':
         ml = cfg.loginMethodList.get( service, cfg.defaultLoginMethodList )
      elif type == 'authnTypeEnable':
         ml = cfg.enableMethodList.get( service, cfg.defaultEnableMethodList )
      else:
         ml = None
         Logging.log( AAA_INVALID_AUTHENTYPE, type )
      return ml

   def findAllLoginMethods( self ):
      methodSet = set()
      # pylint: disable-next=unused-variable
      for service, methodList in self.config.loginMethodList.items():
         for method in methodList.method.values():
            methodSet.add( method )

      # Add default login methods as well
      for method in self.config.defaultLoginMethodList.method.values():
         methodSet.add( method )

      return methodSet

   def findNextPlugin( self, ml, authenSession ):
      traceX( TR_INFO, "find next plugin starting", authenSession.nextMethodIndex,
              "total", len( ml.method ) )
      # pylint: disable-next=superfluous-parens
      while ( authenSession.nextMethodIndex < len( ml.method ) ):
         m = ml.method.get( authenSession.nextMethodIndex )
         if not m:
            # CLI might be modifying it
            return None
         authenSession.nextMethodIndex += 1
         # find the plugin
         plugin = self.pluginForMethod( m )
         if not plugin:
            bt( TR_ERROR, "no plugin found for authen method", bv( m ) )
            Logging.log( AAA_INVALID_AUTHN_METHODLIST, authenSession.service, m )
         elif plugin.ready():
            authenSession.setMethod( m )
            return plugin
         else:
            Logging.log( AAA_AUTHN_PLUGIN_NOT_READY, m )
      # no match
      return None

   def authenticate( self, authSession, *responses ):
      """In case of fallback, we may already have some information
      (username/password) from the unavailable method. In this case
      we auto-feed those responses directly into the current method
      without going back to pam. In case we do not have the information,
      we mark a flag and remember to track the response from pam.
      """
      while True:
         result = authSession.authenticator.authenticate( *responses )
         if result.get( 'status' ) != 'inProgress':
            break
         messages = result.get( 'messages', [] )
         responses = authSession.response.processAuthenMessages( messages )
         if responses is None:
            break
      return result

   def _lockoutRemoteUser( self, type, service, user ):
      # Returns True if we should lock out account (there are too many consecutive 
      # failed logins within lockoutWindow time period). Only affects remote login.
      if type == 'authnTypeLogin' and service != 'login':
         currTime = Tac.now()
         with self.lockoutMutex:
            userLockoutStatus = self.status.userLockoutStatus.get( user )
            if userLockoutStatus:
               lastFailedLogin = userLockoutStatus.lastFailedLogin
               numFailedLogins = userLockoutStatus.numFailedLogins
               bt( TR_ERROR, "failed login counter:", bv( numFailedLogins ) )
               if lastFailedLogin + agent.config.lockoutWindow < currTime:
                  bt( TR_WARN, "resetting lockout status: lockout expired" )
                  self.unlockAccount( user )
               elif numFailedLogins >= agent.config.maxLoginAttempts:
                  if currTime >= lastFailedLogin + agent.config.lockoutTime:
                     bt( TR_WARN, "unlocking account: enough time has elapsed" )
                     self.unlockAccount( user )
                  else:
                     bt( TR_ERROR, "authentication aborted: too many failed logins" )
                     return True
      return False

   def _maybeUnlockAccount( self, type, service, user ):
      # Upon successful remote login, reset lockout counters
      # ('login' service refers to console login)
      if ( type == 'authnTypeLogin' and service != 'login' and
           user in self.status.userLockoutStatus ):
         bt( TR_WARN, "resetting lockout status: successful login" )
         self.unlockAccount( user )

   def unlockAccount( self, user ):
      with self.lockoutMutex:
         del self.status.userLockoutStatus[ user ]

   def _startNewAuthenticator( self, methodList, authenSession ):
      traceX( TR_INFO, "In _startNewAuthenticator" )
      authenState = Tac.Value( "AaaApi::AuthenState" )
      authenState.id = authenSession.id
      authenState.status = 'unavailable'
      authenState.user = authenSession.user
      authenState.authToken = authenSession.authToken

      while True:
         plugin = self.findNextPlugin( methodList, authenSession )
         if plugin is None:
            break
         try:
            authenSession.response.enableAutoFeed( )
            a = plugin.createAuthenticator( authenSession.methodName,
                                            authenSession.type,
                                            authenSession.service,
                                            authenSession.remoteHost,
                                            authenSession.remoteUser,
                                            authenSession.tty,
                                            authenSession.user,
                                            authenSession.privLevel )
            authenSession.setAuthenticator( a )
            r = self.authenticate( authenSession )
            self.applyPluginResult( r, authenState )
            status = authenState.status
            traceX( TR_AUTHEN, "authenticator returned status", status,
                    "user", authenState.user )
            authenSession.updateState( authenState )
            if authenState.status == 'success':
               authenSession.authenServer = r.get( 'authenServer' )
         except:
            import traceback
            exc = traceback.format_exc()
            bt( TR_ERROR, "authentication exception:", bv( str( exc ) ) )
            raise
         # I must handle a status of unavailable differently than other
         # results because I must fall back to the next authentication
         # method in the list .
         if status != 'unavailable':
            break
         authenSession.logFallbackMessage( )

      if authenState.status == 'inProgress':
         added = self.addAuthenSession( authenSession )
         if not added:
            del authenSession
            authenState = Tac.Value( "AaaApi::AuthenState",
                                     id=id, status='unavailable' )
      elif authenState.status == 'success':
         self._maybeUnlockAccount( authenSession.type, authenSession.service, 
                              authenSession.user )
         self.handleLogin( authenSession )
         self.maybeAddPendingSession( authenSession, r.get( 'sessionData' ) )
      elif authenState.status == 'unavailable':
         # HACK: If we have never asked for a password, and for some reason
         # we fail with 'unavailable', we want to return 'unknown' so PAM would
         # try the next module (pam_unix.so) which would ask for a password.
         # This way we don't reveal too much information. See system-auth.ac.Eos.
         if authenSession.response.response.get( 'promptPassword' ) is None:
            bt( TR_WARN, "no password asked, return 'unknown' to PAM" )
            authenState.status = 'unknown'
      return authenState

   def userLoginAllowed( self, service, user ):
      if service != "login":
         return True

      matchListStringName = self.consoleConfig.consoleUserMatchList
      matchList = self.matchListConfig.matchStringList.get( matchListStringName )

      return not matchList or matchList.match( user )

   # AaaApi.py contains the wrapper functions that call these functions:
   # startAuthenticate, continueAuthenticate, abortAuthenticate, openSession,
   # closeSession.  The expected usage is that pam_aaa or nss_aaa will use
   # PyClient to invoke the functions in AaaApi.py, which will then invoke
   # these methods on the global Aaa instance.
   def startAuthenticate( self, type, service, tty, remoteHost, remoteUser,
                          user, uid=None, privLevel=0, sessionId=None,
                          method=None, localAddr=None, localPort=None,
                          remotePort=None ):
      bt( TR_AUTHEN, "startAuthenticate for type", bv( type ),
          "service", bv( service ), "tty", bv( tty or "" ),
          "sessionId", bv( sessionId ), "user", bv( user ) )
      tty = UtmpDump.sanitizeTtyName( tty or "" )

      # look up the method list config for service
      authenSession = None
      if sessionId is not None:
         id = sessionId
      else:
         id = self.allocateAuthenSessionId()
      authenState = Tac.Value( "AaaApi::AuthenState" )
      authenState.id = id
      authenState.status = 'fail'
      authenState.user = user
      authenState.authToken = ""

      if not self.userLoginAllowed( service, user ):
         return authenState

      if uid is not None:
         assert not user
         if uid == 0 and type == 'authnTypeEnable':
            bt( TR_INFO, "Allowing uid 0 to enable" )
            authenState.user = "root"
            authenState.status = 'success'
            return authenState

         user = self.getUserById( uid )
         if not user:
            bt( TR_ERROR, "authentication aborted: unknown uid", bv( uid ) )
            authenState.message0 = Tac.Value(
               "AaaApi::AuthenMessage",
               style='error', text='Cannot authenticate unknown uid %d' %( uid ) )
            return authenState
         authenState.user = user
      session = None
      with self.mutex:
         if sessionId is not None:
            session = self.status.session.get( sessionId )
            if not session:
               bt( TR_ERROR, "authentication aborted: unknown sessionId", 
                                                                    bv( sessionId ) )
               authenState.message0 = Tac.Value( "AaaApi::AuthenMessage",
                  style='error',
                  text='Cannot authenticate for unknown sessionId %s' %(
                     sessionId ) )
               return authenState
            if not service:
               service = session.service
         else:
            # use the newly allocated id as session id
            sessionId = id
         if session:
            if not session.account:
               bt( TR_ERROR, "authentication aborted: session user deleted" )
               authenState.message0 = Tac.Value( "AaaApi::AuthenMessage",
                  style='error',
                  text="user deleted in sessionId %d" % ( sessionId ) )
               return authenState
            if uid is not None and session.account.id != uid:
               bt( TR_ERROR, "authentication aborted: uid", bv( uid ),
                   "doesn't match session's uid", bv( session.account.id ) )
               authenState.message0 = Tac.Value( "AaaApi::AuthenMessage",
                  style='error',
                  text="uid doesn't match the uid in sessionId %d" % ( sessionId ) )
               return authenState
            if session.remoteHost and not remoteHost:
               remoteHost = session.remoteHost
            if session.remoteUser and not remoteUser:
               remoteUser = session.remoteUser
            if session.tty and not tty:
               tty = session.tty

      if method is not None:
         ml = Tac.newInstance( "Aaa::AuthenMethodList", "testAaaMethodList" )
         ml.method[ 0 ] = method
      else:
         ml = self.findAuthenMethodList( type, service )

      traceX( TR_INFO, "service is:", service)
      traceX( TR_INFO, "method list is:", ','.join( ml.method.values() ) )
      if ml:
         authenSession = AuthenSession( id, type, service, tty, remoteHost,
                                        remoteUser, user, privLevel, sessionId,
                                        localAddr=localAddr, localPort=localPort,
                                        remotePort=remotePort,
                                        authenSource=Tac.enumValue(
                                           "Aaa::AuthenSource",
                                           authenSourcePassword ) )
         authenState = self._startNewAuthenticator( ml, authenSession )
      self.maybeLogFailure( authenSession, authenState )
      return authenState

   def continueAuthenticate( self, authenStateId, *responses, **keywords ):
      traceX( TR_AUTHEN, "continueAuthenticate for authenStateId", authenStateId )
      try:
         with self.mutex:
            sess = self.authenSessions.get( authenStateId )
            if not sess:
               bt( TR_ERROR, "no authenSession found for id", bv( authenStateId ) )
               authenState = Tac.Value( "AaaApi::AuthenState",
                                        id=authenStateId, status='fail' )
               return authenState
            # set the flag so we are not removed from the pool
            sess.poolstatus = sess.poolStatusBusy

            # Lockout account from remote access if applicable. This check is in
            # continueAuthenticate because we might not know the username until
            # now (in case of telnet)
            if self._lockoutRemoteUser( sess.type, sess.service, sess.user ):
               Logging.log( AAA_LOGIN_FAILED, sess.user,
                            sess.remoteHost, sess.service,
                            LOCKOUT_MESSAGE )
               # Accounting for failed authentication
               self.sendFailedShellAcct( sess.user, sess.localAddr, sess.localPort,
                                         sess.remoteHost, sess.remotePort, sess.tty,
                                         sess.privLevel, sess.sshPrincipal,
                                         sess.authenSource, LOCKOUT_MESSAGE )
               del self.authenSessions[ authenStateId ]
               sess.poolstatus = sess.poolStatusIdle
               return Tac.Value( "AaaApi::AuthenState",
                                 id=authenStateId,
                                 status='fail',
                                 message0=Tac.Value( "AaaApi::AuthenMessage",
                                                     style='error',
                                                     text=LOCKOUT_MESSAGE ) )

         # keep track of username/password
         sess.response.processAuthenResponses( *responses )
         r = self.authenticate( sess, *responses )
         self.applyPluginResult( r, sess.state )
         traceX( TR_AUTHEN, "authenticator returned status", sess.state.status,
                 "user", sess.state.user )
         sess.updateState( sess.state )
         if sess.state.status == 'success':
            sess.authenServer = r.get( 'authenServer' )
            self._maybeUnlockAccount( sess.type, sess.service, sess.user )
            self.handleLogin( sess )
            self.maybeAddPendingSession( sess, r.get( 'sessionData' ) )

         ml = None
         method = keywords.get( 'groupName' )
         with self.mutex:
            if sess.poolstatus == sess.poolStatusAbort:
               # this session has been aborted, just return error
               bt( TR_ERROR, "authSession", bv( authenStateId ), "has been aborted" )
               del self.authenSessions[ authenStateId ]
               return Tac.Value( "AaaApi::AuthenState",
                                 id=authenStateId, status='fail' )
            if sess.state.status != 'inProgress':
               del self.authenSessions[ authenStateId ]
               sess.poolstatus = sess.poolStatusIdle
               if sess.state.status == 'unavailable':
                  if method is None:
                     ml = self.findAuthenMethodList( sess.type, sess.service )
                  else:
                     ml = Tac.newInstance( "Aaa::AuthenMethodList",
                                           "testAaaMethodList" )
                     ml.method[ 0 ] = method
            else:
               # still need to keep it in the queue, mark as idle
               sess.poolstatus = sess.poolStatusIdle

         if ml:
            sess.logFallbackMessage( )
            # we need to find the next method and try again with the
            # credentials we've collected so far
            sess.response.enableAutoFeed( )
            authenState = self._startNewAuthenticator( ml, sess )
         else:
            authenState = sess.state
         self.maybeLogFailure( sess, authenState )
         return authenState
      except:
         import traceback
         exc = traceback.format_exc()
         bt( TR_ERROR, "authentication exception:", bv( str( exc ) ) )
         # delete the session
         with self.mutex:
            del self.authenSessions[ authenStateId ]
         raise

   def abortAuthenticate( self, authenStateId ):
      bt( TR_ERROR, "abortAuthenticate for id", bv( authenStateId ) )
      id = int( authenStateId )
      with self.mutex:
         sess = self.authenSessions.get( id )
         if sess is None:
            return
         if self.config.loggingOnFailure:
            Logging.log( AAA_LOGIN_FAILED, sess.user, sess.remoteHost,
                         sess.service, "Authentication aborted" )
         # Accounting for failed authentication
         self.sendFailedShellAcct( sess.user, sess.localAddr, sess.localPort,
                                   sess.remoteHost, sess.remotePort, sess.tty,
                                   sess.privLevel, sess.sshPrincipal,
                                   sess.authenSource, "Authentication aborted" )

         if sess.poolstatus == sess.poolStatusIdle:
            del self.authenSessions[ id ]
         else:
            # Another thread is still using this session --
            # tell that thread to abort when continueAuthenticate
            # returns
            sess.poolstatus = sess.poolStatusAbort

   def createSession( self, user, service, tty=None,  remoteHost=None,
                      remoteUser=None, authenMethod="", authenSource='',
                      localAddr=None, localPort=None, remotePort=None,
                      sshPrincipal=None ):
      bt( TR_SESSION, "createSession for user", bv( user ), "tty", bv( tty or "" ) )
      if not user:
         return Tac.Value( "AaaApi::Session", status="failed" )
      tty = UtmpDump.sanitizeTtyName( tty or "" )
      sessionId = self.allocateAuthenSessionId()
      authenSession = AuthenSession( sessionId, 'authnTypeLogin', service, tty,
                                     remoteHost, remoteUser, user, 0, sessionId,
                                     localAddr=localAddr, localPort=localPort,
                                     remotePort=remotePort,
                                     sshPrincipal=sshPrincipal,
                                     authenSource=authenSource )
      authenSession.setMethod( authenMethod )
      # In case authentication was not handled by AAA (ex SSH keys), check
      # lockout status.
      if self._lockoutRemoteUser( 'authnTypeLogin', service, user ):
         Logging.log( AAA_LOGIN_FAILED, user, remoteHost or "", service,
                      LOCKOUT_MESSAGE )
         # Accounting for failed authentication
         self.sendFailedShellAcct( authenSession.user, authenSession.localAddr,
                                   authenSession.localPort, authenSession.remoteHost,
                                   authenSession.remotePort, authenSession.tty,
                                   authenSession.privLevel,
                                   authenSession.sshPrincipal,
                                   authenSession.authenSource, LOCKOUT_MESSAGE )

         return Tac.Value( "AaaApi::Session", id=sessionId, status='failed',
                           pamError=AaaDefs.PAM_MAXTRIES,
                           message=LOCKOUT_MESSAGE )
      else:
         # We validated user successfully and apparently we haven't reached enough
         # consecutive failed logins to lock out, so reset counter
         self._maybeUnlockAccount( 'authnTypeLogin', service, user )
      self.handleLogin( authenSession )
      self.maybeAddPendingSession( authenSession, None )
      return self.openSession( sessionId )

   def createSessionData( self, session, attrs ):
      assert type( attrs ) is not str # pylint: disable=unidiomatic-typecheck
      if attrs is not None:
         prop = session.property.newMember( session.authenMethod )
         for k, v in attrs.items():
            if isinstance( v, bytes ):
               # attributes stored as byte: Radius classAttr
               prop.attrByte[ k ] =  v
            else:
               prop.attr[ k ] = str( v )

   def userSessionCount( self, username ):
      # count all established sessions of specified user (excluding console)
      return sum( session.service != 'login' and
                  session.account and
                  session.account.userName == username and
                  session.state == 'established'
                  for session in self.status.session.values() )

   def openSession( self, authenStateId ):
      bt( TR_SESSION, "openSession for id", bv( authenStateId ) )
      r = Tac.Value( "AaaApi::Session", id=authenStateId )
      user = ''
      with self.mutex:
         sess = self.status.session.get( authenStateId )
         if sess and sess.state == 'pending':
            username = sess.account.userName
            if sess.service == 'login':
               # no limit for console
               userLimit = -1
            else:
               userLimit = self.mgmtAcctConfig.userSessionLimit( username )

            if ( userLimit == 0 or
                 ( userLimit > 0 and # pylint: disable=chained-comparison
                   userLimit <= self.userSessionCount( username ) ) ):
               # user over limit
               Logging.log( AAA_USER_SESSION_LIMIT, username,
                            userLimit )
               r.status = 'failed'
               r.pamError = AaaDefs.PAM_PERM_DENIED
               r.message = "Too many logins for %s" % username
               # delete session
               del self.status.session[ sess.id ]

               if agent.config.loggingOnFailure:
                  Logging.log( AAA_LOGIN_FAILED,
                               username,
                               sess.remoteHost,
                               sess.service,
                               'Session limit reached' )
               # Accounting for failed authentication
               self.sendFailedShellAcct( sess.user, sess.localAddr, sess.localPort,
                                         sess.remoteHost, sess.remotePort, sess.tty,
                                         sess.privLevel, sess.sshPrincipal,
                                         sess.authenSource, "Sesion limit reached" )
            else:
               sess.state = 'established'
               r.status = 'open'
               user = sess.userName

               if agent.config.loggingOnSuccess:
                  Logging.log( AAA_LOGIN,
                               sess.account.userName,
                               sess.remoteHost,
                               sess.service )
         elif sess:
            bt( TR_ERROR, "non-pending session exists for id", bv( authenStateId ) )
            r.status = 'failed'
            r.pamError = AaaDefs.PAM_PERM_DENIED
            r.message = "Session already opened"
         else:
            bt( TR_ERROR, "no pending session for id", bv( authenStateId ) )
            r.status = 'failed'
            r.pamError = AaaDefs.PAM_USER_UNKNOWN

      if r.status == 'open':
         # do accounting outside mutex
         self.sendShellAcct( user, authenStateId, "start" )
      return r

   def closeSession( self, sessionId ):
      bt( TR_SESSION, "closeSession for session id", bv( sessionId ) )
      sess = self.status.session.get( sessionId )
      if not sess:
         bt( TR_ERROR, "no session found for session id", bv( sessionId ) )
         return False

      self.sendShellAcct( sess.userName, sessionId, "stop" )
      if agent.config.loggingOnSuccess:
         Logging.log( AAA_LOGOUT,
                      sess.userName,
                      sess.remoteHost,
                      sess.service )

      with self.mutex:
         if sess and sess.sessionPid:
            del self.status.pidToSession[ sess.sessionPid ]
         del self.status.session[ sessionId ]
      return True

   def getSessionRoles( self, sessionId ):
      session = self.status.session.get( sessionId )
      if not session:
         return '[]'

      sessionData = session.property.get( session.authenMethod )
      return sessionData.attr.get( 'roles', '[]' )

   def getUserById( self, uid ):
      acct = self.status.userid.get( uid )
      return acct.userName if acct else None

   def getPwEnt( self, name=None, uid=None, force=True ):
      traceX( TR_DEBUG, "getPwEnt name:", name, "uid:", uid, "force", force )
      acct = None
      pwEnt = None
      if name is not None:
         accounts = self.status.account
         acct = accounts.get( name )
         if not acct and force:
            # No user with this name has logged in since this Aaa process
            # started, but it may be known to one of my plugins. "force" forces
            # Aaa to learn from its plugins, which is undesirable for certain 
            # programs like useradd (bug27964).

            # Assume we are logging in if we are in the case where we are given
            # a name but the account doesn't exist. Since we don't know the
            # service, get the set of all methods according to all configured
            # method lists
            methodSet = self.findAllLoginMethods()

            for method in methodSet:
               plugin = self.pluginForMethod( method )
               if not plugin:
                  bt( TR_ERROR, "no plugin found for authen method", bv( method ) )
               elif plugin.hasUnknownUser():
                  traceX( TR_INFO, "plugin", method, "has unknown user" )

                  # Unknown user (ex TACACS+, RADIUS). The correct thing to do here
                  # would be to assume that the account doesn't exist, but that
                  # policy causes problems for sshd (see BUG3197). So instead claim
                  # that these accounts exist and generate a temporary uid which I
                  # don't store. Note that sshd will actually use this pwent for the
                  # first login of any given user, so it should be as close to
                  # correct as possible.
                  # I'd like to advertise a shell of /bin/nologin for these bogus
                  # users, but sshd uses the result of the getpwnam that it performed
                  # prior to the PAM session, so I have to give a valid shell here.
                  pwEnt = makePwEntry( name, self.config.shell )
                  traceX( TR_INFO, "getPwEnt: unknown user", name,
                          "; generating bogus pwent" )

                  # Generate a UID for the user if it exists or can exist
                  uid = self.generateUid( name )
                  pwEnt.userId = uid
                  traceX( TR_INFO, "getPwEnt: no account found for user", name,
                          "; returning pwent with uid", uid )

      elif uid is not None:
         userids = self.status.userid
         intUid = int( uid )
         acct = userids.get( intUid )

      if acct:
         traceX( TR_DEBUG, "getPwEnt: found account", acct.authenMethod )
         shell = self.getShellFromAccount( acct )
         pwEnt = makePwEntry( acct.userName, shell, acct.authenMethod )
         pwEnt.userId = acct.id

      # Set the groupId for the pwEnt that we found
      if pwEnt is not None:
         try:
            pwEnt.groupId = grp.getgrnam( "eosadmin" ).gr_gid
         except: # pylint: disable=bare-except
            pwEnt.groupId = pwEnt.userId # arbitrary fall-back
         return pwEnt
      traceX( TR_INFO, "getPwEnt: no matching account found" )
      return None

   def getUserIdGroupId( self, userName ):
      acct = self.status.account.get(userName)
      userId = acct.id if acct else self.generateUid( userName )
      groupId = self.groupId if self.groupId is not None else userId

      return (userId, groupId)

   def _doAuthz( self, methodList, r, sessionId, func, simple, user, mlName ):
      """Helper for authorizeShell and authorizeShellCommand."""
      for _, m in sorted( methodList.items() ):
         plugin = self.pluginForMethod( m )
         if not plugin:
            Logging.log( AAA_INVALID_AUTHZ_METHODLIST, mlName, m )
         elif plugin.ready():
            try:
               r.status, r.message, av = func( m, plugin )
               if not simple:
                  for i, ( a, v ) in enumerate( av.items() ):
                     traceX( TR_AUTHZ, "authorizeShell", i, a, v )
                     r.av[ i ] = Tac.Value( "AaaApi::AvPair", name=a, val=repr( v ) )

               with self.mutex:
                  if sessionId is not None:
                     session = self.status.session.get( sessionId )
                     if not session:
                        # session is deleted while we are doing authorization
                        bt( TR_WARN, "session closed during authz:",
                            bv( sessionId ) )
                     elif r.status == 'allowed':
                        if session.sessionPid:
                           self.status.pidToSession[ session.sessionPid ] = session
                        self.createSessionData( session, av )
            except:
               import traceback
               exc = traceback.format_exc()
               bt( TR_ERROR, "authorization exception:", bv( str( exc ) ) )
               raise
            # I must handle a status of unavailable differently than other
            # results because I must fall back to the next method in the list.
            if r.status != 'allowed':
               bt( TR_ERROR, "authorization from", bv( mlName ),
                   bv( r.status ) )
            if r.status != 'authzUnavailable':
               break
            if plugin.logFallback():
               Logging.log( AAA_AUTHZ_FALLBACK, m, mlName, user )
         else:
            Logging.log( AAA_AUTHZ_PLUGIN_NOT_READY, m )

   def _updateTty( self, session, tty ):
      # update tty in case of ssh
      if tty and session and session.tty == "ssh":
         session.tty = tty
         self.removeOlderSessionsWithSameTty( session )

   def authorizeShell( self, uid, sessionId, user=None, sessionPid=None, tty=None ):
      tty = UtmpDump.sanitizeTtyName( tty or "" )

      r = Tac.Value( "AaaApi::AuthzResult", status='denied', message="" )
      if uid == 0:
         # Let root do whatever it wants without authorization
         bt( TR_AUTHZ, "authorizeShell for root allowed" )
         r.status = 'allowed'
         return r

      if not user:
         user = self.getUserById( uid )
         if not user:
            bt( TR_ERROR, "authorization denied: unknown uid", bv( uid ) )
            r.message = 'Cannot authorize shell for unknown uid %d' % uid
            return r

      if sessionId is None:
         bt( TR_ERROR, "exec authorization denied: unspecified sessionId",
             bv( sessionId ) )
         r.message = 'Cannot authorize shell for unspecified sessionId'
         return r


      bt( TR_AUTHZ, "authorizeShell for user", bv( user ),
          "sessionId", bv( sessionId ), "tty", bv( tty or "" ) )

      session = None
      with self.mutex:
         if sessionId is not None:
            session = self.status.session.get( sessionId )
            if not session:
               bt( TR_ERROR, "exec authorization aborted: unknown sessionId",
                   bv( sessionId ) )
               r.status = 'aborted'
               r.message = 'Cannot authorize shell for unknown sessionId %d' % \
                           sessionId
               return r
            self._updateTty( session, tty )
            if sessionPid:
               session.sessionPid = sessionPid
      ml = self.findAuthzMethodList( 'exec', session.service )
      if not ml:
         r.message = "No authorization method list found for 'exec'"
         return r

      def _authzShell( method, plugin ):
         return plugin.authorizeShell( method, user, session )

      self._doAuthz( ml, r, sessionId, _authzShell, False, user, 'exec' )
      return r

   def authorizeShellCommand( self, uid, mode, privlevel, tokens, sessionId ):
      r = Tac.Value( "AaaApi::AuthzResultSimple", status='denied', message="" )
      if uid == 0:
         # Let root do whatever it wants without authorization
         r.status = 'allowed'
         return r
      user = self.getUserById( uid )
      if not user:
         bt( TR_ERROR, "authorization aborted: unknown uid", bv( uid ) )
         r.message = 'Cannot authorize a command for unknown uid %d' % uid
         return r
      session = None
      with self.mutex:
         if sessionId is not None:
            try:
               session = self.status.session[ sessionId ]
               if session.service == 'login' and not self.config.consoleAuthz:
                  # authz is disabled on console, just return
                  traceX( TR_DEBUG, 'Console authorization is disabled' )
                  r.status = 'allowed'
                  return r
            except KeyError:
               bt( TR_ERROR, "cmd authorization aborted: unknown sessionId",
                   bv( sessionId ) )
               r.status = 'aborted'
               r.message = 'Cannot authorize command for unknown sessionId %d' % \
                           sessionId
               return r

      cfg = self.config

      mlname = "command%02d" % privlevel
      try:
         ml = cfg.authzMethod[ mlname ].defaultMethod
      except KeyError:
         bt( TR_ERROR, "authorization aborted: unknown method list", bv( mlname ) )
         r.message = "Authorization method list '%s' not found" % mlname
         return r

      def _authzShellCommand( method, plugin ):
         return plugin.authorizeShellCommand( method, user, session, mode, privlevel,
                                              tokens )

      self._doAuthz( ml, r, sessionId, _authzShellCommand, True, user, mlname )
      return r

   def sendShellAcct( self, user, sessionId, action ):
      traceX( TR_ACCT, "sendShellAcct user =", user,
              "sessionId =", sessionId, "action =", action )

      session = None
      if sessionId is not None:
         session = self.status.session.get( sessionId )
         if not session:
            bt( TR_ERROR, "exec accounting aborted: unknown sessionId",
                bv( sessionId ) )
            return

      # Go through all the extra accounting methods configured,
      # find the registered plugin and call them.
      for method in self.status.extraAcctMethods:
         plugin = self.pluginForMethod( method )
         if plugin:
            try:
               plugin.sendShellAcct( method, user, session, action, Tac.utcNow() )
            except Exception as e:  # pylint: disable=broad-except
               bt( TR_ERROR, f"sendShellAcct failed {str(e)}" )
         else:
            bt( TR_ERROR, "sendShellAcct plugin for method", bv( method ),
                  "not registered" )

      mlname = "exec"
      ml = self.config.acctMethod.get( mlname )
      if ml is None:
         bt( TR_ERROR, "accounting aborted: unknown method list", bv( mlname ) )
         return
      method, methodAction = ml.defaultMethod, ml.defaultAction
      if session and session.service == 'login' and ml.consoleUseOwnMethod:
         method, methodAction = ml.consoleMethod, ml.consoleAction

      if methodAction == "none":
         return
      elif action == "start" and methodAction == "stopOnly":
         return

      def _sendShellAcct( methodName, plugin ):
         elapsedTime = None
         if action == "stop":
            elapsedTime = Tac.now() - session.startTime
         return plugin.sendShellAcct( methodName, user, session, action,
                                      session.startTimeInUtc, elapsedTime )
      acctQElement = AcctQueueElement( user, mlname, method, _sendShellAcct )
      traceX( TR_DEBUG, "Enqueing accounting element", acctQElement,
              'action=', action )
      self._enqueueAcctRequest( acctQElement )

   def sendCommandAcct( self, uid, privLevel, tokens, sessionId, cmdType=None,
                        **kwargs ):
      '''
         Sent for command accounting. kwargs can be
         1. gRPCType: gnmi, gnoi, gnsi, gribi, p4Runtime
         2. gRPCName: Name of the gRPC
         3. gRPCPayload: Payload of the gRPC
         4. gRPCPayloadTruncated: A boolean to indicate if the payload was truncated
      '''
      traceX( TR_ACCT, "SendCommandAcct uid = ", uid, " sessionId = ", sessionId,
              " tokens = ", tokens )

      if uid == 0:
         traceX( TR_DEBUG, "Aborting command accounting for root user" )
         return

      user = self.getUserById( uid ) or f"unknown,uid={uid}"

      session = None
      if sessionId is not None:
         session = self.status.session.get( sessionId )
         if not session:
            bt( TR_ERROR, "cmd accounting aborted: unknown sessionId",
                bv( sessionId ) )
            self.status.counters.acctError += 1
            return

      # Go through all the extra accounting methods configured,
      # find the registered plugin and call them.
      for method in self.status.extraAcctMethods:
         plugin = self.pluginForMethod( method )
         if plugin:
            try:
               plugin.sendCommandAcct( method, user, session, privLevel,
                                       Tac.utcNow(), tokens, cmdType=cmdType,
                                       **kwargs )
            except Exception as e:  # pylint: disable=broad-except
               bt( TR_ERROR, f"sendCommandAcct failed {str(e)}" )
         else:
            bt( TR_ERROR, "sendCommandAcct plugin for method", bv( method ),
                  "not registered" )

      mlname = "command%02d" % privLevel
      ml = self.config.acctMethod.get( mlname )
      if ml is None:
         bt( TR_ERROR, "accounting aborted: unknown method list", bv( mlname ) )
         self.status.counters.acctError += 1
         return
      method, methodAction = ml.defaultMethod, ml.defaultAction
      if session and session.service == 'login' and ml.consoleUseOwnMethod:
         method, methodAction = ml.consoleMethod, ml.consoleAction

      if methodAction == 'none':
         return

      def _sendCommandAcct( methodName, plugin ):
         return plugin.sendCommandAcct( methodName, user, session, privLevel,
                                        Tac.utcNow(), tokens )
      acctQElement = AcctQueueElement( user, mlname, method, _sendCommandAcct )
      self._enqueueAcctRequest( acctQElement )

   def sendFailedShellAcct( self, user, localAddr, localPort, remoteHost, remotePort,
                            tty, privLevel, sshPrincipal, authenSource,
                            failureCause ):
      '''
         Sent to indicate failed authentication event
      '''
      traceX( TR_ACCT, "SendFailedShellAcct user = ", user, " remoteAddr = ",
            remoteHost, " authenSource = ", authenSource, " failureCause = ",
            failureCause )

      # Go through all the extra accounting methods configured,
      # find the registered plugin and call them.
      for method in self.status.extraAcctMethods:
         plugin = self.pluginForMethod( method )
         if plugin:
            try:
               plugin.sendFailedShellAcct( user, localAddr, localPort, remoteHost,
                                           remotePort, tty, privLevel, sshPrincipal,
                                           authenSource, failureCause, Tac.utcNow() )
            except Exception as e:  # pylint: disable=broad-except
               bt( TR_ERROR, f"sendFailedShellAcct failed {str(e)}" )
         else:
            bt( TR_ERROR, "sendFailedShellAcct plugin for method", bv( method ),
                  "not registered" )

   def sendFailedCommandAcct( self, user, privLevel, sessionId, tokens, cmdType,
                              authzDetail, **kwargs ):
      '''
         Send to indicate failed authorization event. kwargs can be,
         1. gRPCType: gnmi, gnoi, gnsi, gribi, p4Runtime
         2. gRPCName: Name of the gRPC
         3. gRPCPayload: Payload of the gRPC
         4. gRPCPayloadTruncated: A boolean to indicate if the payload was truncated
      '''
      traceX( TR_ACCT, "SendFailedCommandAcct user = ", user, " sessionId = ",
            sessionId, " tokens = ", tokens, " authzDetail = ", authzDetail )

      session = None
      if sessionId is not None:
         session = self.status.session.get( sessionId )
         if not session:
            bt( TR_ERROR, "cmd accounting aborted: unknown sessionId",
                bv( sessionId ) )
            return

      # Go through all the extra accounting methods configured,
      # find the registered plugin and call them.
      for method in self.status.extraAcctMethods:
         plugin = self.pluginForMethod( method )
         if plugin:
            try:
               plugin.sendFailedCommandAcct( user, privLevel, session, tokens,
                                             cmdType, Tac.utcNow(), authzDetail,
                                             **kwargs )
            except Exception as e:  # pylint: disable=broad-except
               bt( TR_ERROR, f"sendFailedCommandAcct failed {str(e)}" )
         else:
            bt( TR_ERROR, "sendFailedCommandAcct plugin for method", bv( method ),
                  "not registered" )

   def sendSystemAcct( self, reason, action ):
      bt( TR_ACCT, "SendSystemAcct action", bv( action ), "reason", bv( reason ) )

      event = 'sys_acct'
      mlname = "system"
      ml = self.config.acctMethod.get( mlname )
      if ml is None:
         return
      method, methodAction = ml.defaultMethod, ml.defaultAction
      if methodAction == "none":
         return
      elif action == "start" and methodAction == "stopOnly":
         traceX( TR_ACCT, "action == 'start' but methodAction == 'stopOnly'. Skip." )
         return

      def _sendSystemAcct( methodName, plugin ):
         return plugin.sendSystemAcct( methodName, event,
                                       Tac.utcNow(), reason, action )
      acctQElement = AcctQueueElement( "unknown", mlname, method, _sendSystemAcct )
      self._enqueueAcctRequest( acctQElement )

   def _enqueueAcctRequest( self, element ):
      """ The function enqueues accounting request in the accounting queue.
      If the queue is full then half the oldest entries are purged and new
      request is enqueued. """

      done = False
      while not done:
         try:
            # numPendingAcctRequests should be set here before enqueuing because
            # numPendingAcctRequests is also set in dispatcher thread. If we set
            # numPendingAcctRequests after enqueuing, a race exists where after 
            # we get acctQueue.size() and before we set it to numPendingAcctRequests
            # here, dispatcher might dequeue the request and set 
            # numPendingAcctRequests which we will end up overriding here.
            self.status.counters.numPendingAcctRequests = acctQueue.qsize() + 1
            acctQueue.put_nowait( element )
            done = True
         except queue.Full:
            numElementsPurged = acctQueue.purge( finalQSize=_acctQueueMaxSize // 2 )
            self.status.counters.acctError += numElementsPurged
            Logging.log( AAA_ACCT_QUEUE_FULL, numElementsPurged )

   def stopAcct( self ):
      # used for tests
      self.acctDispatcherThread_.stop()
      self.acctDispatcherThread_.join()

class AcctQueue( queue.Queue ):

   def purge( self, finalQSize ):
      ''' Removes older entries until the queue size is less than
      or equal to finalQSize. '''
      numPurged = 0
      while self.qsize() > finalQSize:
         try:
            self.get_nowait()
            self.task_done()
            numPurged += 1
         except queue.Empty:
            break
      return numPurged

class AcctDispatcher( threading.Thread ):
   ''' Responsible for dequeing the acct requests from queue. '''
   def run( self ):
      prevDispatchResult = True
      while True:
         item = acctQueue.get()
         if not item:
            # signal for exiting
            bt( TR_ACCT, "accounting thread exiting" )
            acctQueue.task_done()
            break
         user, mlname, acctMethod, _sendFunc = item
         agent.status.counters.numPendingAcctRequests = acctQueue.qsize()
         traceX( TR_DEBUG, "Dispatching accounting request user=", user, " mlname=", 
                mlname, " _sendFunc=", _sendFunc.__name__ )
         dispatchResult = self.dispatch( user, mlname, acctMethod, _sendFunc )
         traceX( TR_DEBUG, "Dispatched accounting request mlname=", mlname,
                 "acctMethod=", acctMethod, "_sendFunc=", _sendFunc.__name__,
                 "result=", dispatchResult )
         if not dispatchResult:
            agent.status.counters.acctError += 1
         else:
            agent.status.counters.acctSuccess += 1
         # logging
         if dispatchResult != prevDispatchResult:
            prevDispatchResult = dispatchResult
            if dispatchResult == False: # pylint: disable=singleton-comparison
               Logging.log( AAA_ACCT_MSG_DROP )
            else:
               Logging.log( AAA_ACCT_MSG_RESUME )
         acctQueue.task_done()

   def dispatch( self, user, mlname, method, func ):
      """ Helper function used by the dispatcher thread. """
      for _, m in sorted( method.items() ):
         plugin = agent.pluginForMethod( m )
         if not plugin:
            Logging.log( AAA_INVALID_ACCT_METHODLIST, mlname, m )
         elif plugin.ready():
            status = 'acctError'
            try:
               status, message = func( m, plugin ) # pylint: disable=unused-variable
            except: # pylint: disable=bare-except
               import traceback
               exc = traceback.format_exc()
               bt( TR_ERROR, "accounting exception:", bv( str( exc ) ) )
            # Falling back to the next method if status is not success
            if status == 'acctSuccess':
               return True
            if plugin.logFallback():
               Logging.log( AAA_ACCT_FALLBACK, m, mlname, user )
         else:
            Logging.log( AAA_ACCT_PLUGIN_NOT_READY, m )
      return False

   def stop( self ):
      # used for tests to gracefully shutdown Aaa agent
      bt( TR_ACCT, "wait for accounting queue to flush", bv( acctQueue.qsize() ) )
      acctQueue.join()
      bt( TR_ACCT, "stopping accouting thread" )
      acctQueue.put_nowait( None )

def main():
   # Setting this variable causes libnss_aaa to short-circuit and return
   # "not found".  This is necessary to avoid deadlocks caused by processes
   # spawned by the Aaa agent attempting to PyClient back into the agent
   # while the agent's activity thread is blocked waiting for the child process
   # to exit.
   from LocalUserLib import getNssAaaDisableFunc
   f = getNssAaaDisableFunc()
   f( 1 )
   container = Agent.AgentContainer(
      [ Aaa ], pyServerUnixAddr='/var/run/AaaSocket-%d' % os.getpid() )
   container.runAgents()
