# Copyright (c) 2008, 2009, 2010, 2011  Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=c-extension-no-member
import Tac, Tracing, Aaa
import ArPyUtils
import Logging
import sys

AAA_EXEC_AUTHZ_FAILED = Logging.LogHandle(
      "AAA_EXEC_AUTHZ_FAILED",
      severity=Logging.logWarning,
      fmt="User %s failed authorization to start a shell on %s%s",
      explanation="AAA server denied authorization to start a shell.",
      recommendedAction="Confirm that the user settings on authorization "
                        "server(s) are correct." )

AAA_CMD_AUTHZ_FAILED = Logging.LogHandle(
      "AAA_CMD_AUTHZ_FAILED",
      severity=Logging.logWarning,
      fmt="User %s failed authorization to execute command '%s'%s",
      explanation="AAA denied authorization for the specified command.",
      recommendedAction="Confirm that the user settings on authorization "
                        "server(s) are correct." )

AAA_AUTHEN_UNEXPECTED_MESSAGE = Logging.LogHandle(
      "AAA_AUTHEN_UNEXPECTED_MESSAGE",
      severity=Logging.logError,
      fmt="Unexpected message for login service '%s' user '%s' (%s)",
      explanation="AAA received unexpecte message for authentication",
      recommendedAction="AAA server should only ask for username/password" )

t0 = Tracing.trace0
authenSourceEnum = Tac.Type( "Aaa::AuthenSource" )

# For some APIs, there are two versions:
# A normal version that is supposed to be called by PyClient's eval method,
# which returns a TAC type, and an Exec version that is supposed to be called by
# PyClient's exec method, which returns output for the caller to parse. The Exec
# APIs are used when the client does not have or want libtac.

def _waitForAaa( ):
   '''Wait for Aaa to initialize before proceeding (especially important if
   Aaa restarts). 30 seconds should be plenty, and if we timeout the client 
   will get an exception.

   Use sleep=True as the client should use execModeThreadPerConnection to
   connect to Aaa.
   '''
   ArPyUtils.waitFor( lambda: Aaa.agent and Aaa.agent.initialized,
                description="Aaa to initialized", timeout=30, firstCheck=0 )

def _printAuthenStateExec( state ):
   # print fields in declarative order
   sys.stdout.write( '\x00'.join( str( x ) for x in
                                ( "id",
                                  state.id,
                                  "status",
                                  state.status,
                                  "style0",
                                  state.message0.style,
                                  "text0",
                                  state.message0.text,
                                  "style1",
                                  state.message1.style,
                                  "text1",
                                  state.message1.text,
                                  "style2",
                                  state.message2.style,
                                  "text2",
                                  state.message2.text,
                                  "style3",
                                  state.message3.style,
                                  "text3",
                                  state.message3.text,
                                  "user",
                                  state.user,
                                  "authToken",
                                  state.authToken ) ) )

def _printAuthenResultExec( result ):
   sys.stdout.write( '\x00'.join( str( x ) for x in
                                ( result.status,
                                  result.sessionId,
                                  result.av,
                                  result.uid,
                                  result.gid ) ) )

def valueToStrep( func ):
   # decorator to return Tac.valueToStrep() of a value unless returnOriginalType=True
   # is specified.
   def convert( *args, **kwargs ):
      returnOriginalType = kwargs.pop( 'returnOriginalType', False )
      r = func( *args, **kwargs )
      if returnOriginalType:
         return r
      else:
         return Tac.valueToStrep( r )
   return convert

@valueToStrep
def startLoginAuthenticate( service, tty=None, remoteHost=None, remoteUser=None,
                            user="", method=None, localAddr=None, localPort=None,
                            remotePort=None ):
   """
      The authentication conversation for a login starts here, with an optional
      username.  The 'service' parameter specifies the service to which the user is
      authenticating, eg. "sshd".
      All the other parameters are optional, but should be provided if they are
      available to the caller.  Returns an AaaApi::AuthenState instance that includes
      the id used to continue the conversation.

      Arguments
      service   : service to which user is authenticating
      tty       : TTY of the connection
      remoteHost: Address of the remote connection
      remoteUser: Remote user initiating the connection
      user      : username of the connection
      method    : authentication method
      localAddr : Address of the local endpoint
      localPort : Port of the local endpoint
      remotePort: Port of the remote connection
   """
   _waitForAaa()
   state = Aaa.agent.startAuthenticate( 'authnTypeLogin', service, tty, remoteHost, 
                                         remoteUser, user, method=method,
                                         localAddr=localAddr, localPort=localPort,
                                         remotePort=remotePort )
   Aaa.agent.incAuthnCounter( state.status, service )
   return state

def startLoginAuthenticateExec( service, tty=None, remoteHost=None,
                                remoteUser=None, user="", method=None,
                                localAddr=None, localPort=None, remotePort=None ):
   """
      Exec version of startLoginAuthenticate()

      Arguments
      service   : service to which user is authenticating
      tty       : TTY of the connection
      remoteHost: Address of the remote connection
      remoteUser: Remote user initiating the connection
      user      : username of the connection
      method    : authentication method
      localAddr : Address of the local endpoint
      localPort : Port of the local endpoint
      remotePort: Port of the remote connection
   """
   # pylint: disable=unexpected-keyword-arg
   state = startLoginAuthenticate( service, tty, remoteHost, remoteUser, user,
                                   method=method, returnOriginalType=True,
                                   localAddr=localAddr, localPort=localPort,
                                   remotePort=remotePort )
   # pylint: enable=unexpected-keyword-arg
   _printAuthenStateExec( state )

def startEnableAuthenticate( service, uid, privLevel, sessionId=None ):
   """The authentication conversation for the CLI enable command starts here.
   The uid is required, and the effective uid of the calling process should be
   passed.  The type of authentication is assumed to be 'authnTypeEnable'.
   Returns an AaaApi::AuthenState instance that includes the authenStateId used
   to continue the conversation."""
   _waitForAaa()
   state = Aaa.agent.startAuthenticate( 'authnTypeEnable', service, 
                                        tty=None, remoteHost=None,
                                        remoteUser=None, user="", uid=uid,
                                        privLevel=privLevel,
                                        sessionId=sessionId )
   Aaa.agent.incAuthnCounter( state.status, service )
   return Tac.valueToStrep( state )

@valueToStrep
def continueAuthenticate( authenStateId, *responses, **keywords ):
   """After starting an authentication conversation, the client continues
   calling this function as long as the returned AuthenState instance has a
   status of inProgress.  After each call the messages in the message list
   should be displayed to the user, and responses received for any prompts in
   the list.  Then the responses should be passed in order as arguments to the
   next continueAuthenticate invokation.  Returns an AaaApi::AuthenState
   instance."""
   _waitForAaa()
   service = None
   session = Aaa.agent.authenSessions.get( authenStateId )
   if session:
      service = session.service
   state = Aaa.agent.continueAuthenticate( authenStateId, *responses, 
                                           **keywords )
   Aaa.agent.incAuthnCounter( state.status, service )
   return state

def continueAuthenticateExec( authenStateId, *responses, **keywords ):
   """Exec version of continueAuthenticate()"""
   keywords[ 'returnOriginalType' ] = True
   state = continueAuthenticate( authenStateId, *responses, **keywords )
   _printAuthenStateExec( state )

def abortAuthenticate( authenStateId ): # pylint: disable=useless-return
   """Ends the authentication conversation. Causes the AAA agent to forget
   about this AuthenState id.  Returns nothing."""
   _waitForAaa()
   Aaa.agent.abortAuthenticate( authenStateId )
   Aaa.agent.incAuthnCounter( 'fail' )
   return

def abortAuthenticateExec( authenStateId ):
   """Ends the authentication conversation. Causes the AAA agent to forget
   about this AuthenState id.  Returns nothing."""
   abortAuthenticate( authenStateId )

def _printSessionStateExec( state ):
   # print fields in declarative order
   sys.stdout.write( '\x00'.join( str( x ) for x in
                                ( 'id',
                                  state.id,
                                  'status',
                                  state.status,
                                  'pamError',
                                  state.pamError,
                                  'message',
                                  state.message ) ) )

def sendFailedShellAcct( user, localAddr, localPort, remoteHost, remotePort,
                         tty, privLevel, sshPrincipal=None, failureCause=None,
                         authenSource='' ):
   '''
      Used to record the failed authentication event

      Arguments
      user        : username of the connection
      localAddr   : Address of the local endpoint
      localPort   : Port of the local endpoint
      remoteHost  : Address of the remote connection
      remotePort  : Port of the remote connection
      tty         : TTY of the connection
      privLevel   : privilege level of the user
      sshPrincipal: Matched SSH principal in the client SSH certificate
      failureCause: Reason for the failed authentication
      authenSource: Authentication source, it can be one of the string values
                    define in Aaa::AuthenSource
                    a. Password
                    b. SSH key
                    c. SSH certificate
                    d. TLS certificate
   '''
   _waitForAaa()
   Aaa.agent.sendFailedShellAcct( user, localAddr, localPort, remoteHost,
                                  remotePort, tty, privLevel, sshPrincipal,
                                  verifyAuthenSource( authenSource ), failureCause )
   # this function only returns None
   return Tac.valueToStrep( None )

def sendFailedCommandAcct( user, privLevel, sessionId, tokens, cmdType,
                           authzDetail, **kwargs ):
   '''
      Used to record the failed authorization event

      Arguments
      user       : username of the connection
      privlevel  : the session's current privilege level
      sessionId  : the session id
      tokens     : a list of tokens that compose the command.
      cmdType    : indicates the type of command, "cli", "gRPC" etc
      timestamp  : time of the event
      authzDetail: details of the authorization failure
      kwargs     : gRPCType: gnmi, gnoi, gnsi, gribi, p4Runtime
                   gRPCName: Name of the gRPC
                   gRPCPayload: In case of gRPC's, complete/partial payload
                   gRPCPayloadTruncated: boolean flag to indicate if payload
                                         is truncated
   '''
   _waitForAaa()
   Aaa.agent.sendFailedCommandAcct( user, privLevel, sessionId, tokens, cmdType,
                                    authzDetail, **kwargs )
   # this function only returns None
   return Tac.valueToStrep( None )

def createSession( user, service, tty=None, remoteHost=None, remoteUser=None,
                   authenMethod="", localAddr=None, localPort=None, remotePort=None,
                   sshPrincipal=None, validUserOnly=False, authenSource='' ):
   """Creates a session for a user in case the authentication was not handled
   by AAA.
   
   validUserOnly: only allow when the user is a valid EOS user. A valid EOS user is
   one of the following:
   - a user that has logged in once (known to Aaa)
   - a user that is locally configured (known to Aaa)
   - any user if a remote authentication  method is configured (assumed to be valid)

   For example, services using certificate based authentication might allow users
   not recognized by AAA. On the other hand, services like sshd need to validate
   the user as they need to run the login shell with a valid uid/gid, so they should
   specify validUserOnly=True.

   Arguments:
   user        : username of the connection
   service     : service requesting session
   tty         : TTY of the connection
   remoteHost  : Address of the remote connection
   remoteUser  : Remote user initiating the connection
   authenMethod: Authentication method
   localAddr   : Address of the local endpoint
   localPort   : Port of the local endpoint
   remotePort  : Port of the remote connection
   sshPrincipal: Matched SSH principal in the client SSH certificate
   authenSource: Authentication source, it can be one of the string values
                 defined in Aaa::AuthenSource
                 a. Password
                 b. SSH key
                 c. SSH certificate
                 d. TLS certificate
   """
   _waitForAaa()
   if validUserOnly and not Aaa.agent.getPwEnt( user ):
      session = Tac.Value( "AaaApi::Session", status="failed",
                           message="Unknown user" )
   else:
      session = Aaa.agent.createSession( user, service, tty, remoteHost, remoteUser,
                                         authenMethod=authenMethod,
                                         authenSource=verifyAuthenSource(
                                            authenSource ),
                                         localAddr=localAddr,
                                         localPort=localPort, remotePort=remotePort,
                                         sshPrincipal=sshPrincipal )
   return Tac.valueToStrep( session )

def createSessionExec( user, service, tty=None, remoteHost=None, remoteUser=None,
                       authenMethod="", localAddr=None, localPort=None,
                       remotePort=None, sshPrincipal=None, validUserOnly=False,
                       authenSource='' ):
   """Exec version of createSession()

      Arguments:
      user        : username of the connection
      service     : service requesting session
      tty         : TTY of the connection
      remoteHost  : Address of the remote connection
      remoteUser  : Remote user initiating the connection
      authenMethod: Authentication method
      localAddr   : Address of the local endpoint
      localPort   : Port of the local endpoint
      remotePort  : Port of the remote connection
      sshPrincipal: Matched SSH principal in the client SSH certificate
      authenSource: Authentication source, it can be one of the string values
                    defined in Aaa::AuthenSource
                    a. Password
                    b. SSH key
                    c. SSH certificate
                    d. TLS certificate
   """
   _waitForAaa()
   if validUserOnly and not Aaa.agent.getPwEnt( user ):
      session = Tac.Value( "AaaApi::Session", status="failed",
                           message="Unknown user" )
   else:
      session = Aaa.agent.createSession( user, service, tty, remoteHost, remoteUser,
                                         authenMethod=authenMethod,
                                         localAddr=localAddr, localPort=localPort,
                                         remotePort=remotePort,
                                         sshPrincipal=sshPrincipal,
                                         authenSource=verifyAuthenSource(
                                            authenSource ) )
   _printSessionStateExec( session )

def openSession( authenStateId ):
   """Opens a session for a previously authenticated user. The caller passes in
   the AuthenState id for the conversation in which the user authenticated.
   Returns an AaaApi::Session instance."""
   _waitForAaa()
   session = Aaa.agent.openSession( authenStateId )
   return Tac.valueToStrep( session )

def openSessionExec( authenStateId ):
   """Exec version of openSession()"""
   _waitForAaa()
   session = Aaa.agent.openSession( authenStateId )
   _printSessionStateExec( session )

def closeSession( sessionId ):
   """Closes a previously opened session.  Returns string.
   As AaaPamHelper tries to parse the return value we have to return something."""
   _waitForAaa()
   result = Aaa.agent.closeSession( sessionId )
   return Tac.valueToStrep( result )

def closeSessionExec( sessionId ):
   """Exec version of closeSession()"""
   _waitForAaa()
   Aaa.agent.closeSession( sessionId )

def getPwEnt( name=None, uid=None, force=True ):
   "Returns an AaaApi::PasswdResult instance for the name or uid specified."
   _waitForAaa()
   result = Tac.Value( "AaaApi::PasswdResult" )
   entry = Aaa.agent.getPwEnt( name, uid, force )
   if entry is not None:
      result.found = True
      result.passwordEntry = entry
   else:
      result.found = False
   # remove the extra back slashes added by "valueToStrep" using "decode"
   # python3's equivalent to python2's .decode( 'string_escape' )
   return Tac.valueToStrep( result ).encode(
         'utf-8' ).decode( 'unicode_escape' ).encode( 'latin1' ).decode( 'utf8' )

def getPwEntExec( name=None, uid=None, force=True ):
   """Exec version of getPwEnt."""
   _waitForAaa()
   entry = Aaa.agent.getPwEnt( name, uid, force )
   if entry is not None:
      # now print the fields
      sys.stdout.write( '\x00'.join( str( x ) for x in ( "1", # found
                                                         entry.userName,
                                                         "x", # fake password
                                                         entry.userId,
                                                         entry.groupId,
                                                         entry.realName,
                                                         entry.homeDir,
                                                         entry.shell ) ) )
   else:
      sys.stdout.write( "0" )

def getSessionRoles( sessionId ):
   "Returns the roles for the current session."
   _waitForAaa()
   roles = Aaa.agent.getSessionRoles( sessionId )
   return roles

@valueToStrep
def authorizeShell( uid, user=None, sessionId=None, sessionPid=None, tty=None ):
   """Determines whether the user specified is authorized to start a new shell.
   For this to succeed the Aaa agent must have an active login session for this
   uid. This is related to the "aaa authorization exec" config.

   Returns an AaaApi::AuthzResult."""
   _waitForAaa()
   r = Aaa.agent.authorizeShell( uid, user=user, sessionId=sessionId,
                                 sessionPid=sessionPid, tty=tty )
   Aaa.agent.incAuthzCounter( r.status )
   if r.status == 'denied':
      if not user:
         ent = Aaa.agent.getPwEnt( uid=uid )
         if ent :
            user = ent.userName
         else:
            user = "UID:%d" % uid # pylint: disable=consider-using-f-string
      Logging.log( AAA_EXEC_AUTHZ_FAILED, user, tty or "unknown tty",
                   # pylint: disable-next=consider-using-f-string
                   " (%s)" % r.message if r.message else "" )
      
   return r

def authorizeShellExec( uid, sessionId=None, sessionPid=None,
                        tty=None ):
   """The Exec version of authorizeShell"""
   # pylint: disable=unexpected-keyword-arg
   r = authorizeShell( uid, sessionId=sessionId, sessionPid=sessionPid,
                       tty=tty, returnOriginalType=True )
   # pylint: enable=unexpected-keyword-arg
   # result is ( status, privLevel, autocmd, message )
   status = str( r.status )
   message = r.message
   privLevel = '1'
   autocmd = ''
   for v in r.av.values():
      if v.name == "privilegeLevel":
         privLevel = v.val
      elif v.name == "autocmd":
         # value is repr() of the autocmd string (see _doAuthz() in Aaa.py),
         # so strip that
         autocmd = v.val[ 1: -1 ]
   sys.stdout.write( '\x00'.join( ( 'status',
                                    status,
                                    'message',
                                    message,
                                    'privLevel',
                                    privLevel,
                                    'autocmd',
                                    autocmd ) ) )

def _authorizeShellCommand( uid, mode, privLevel, tokens, sessionId=None,
                            cmdType=None, **kwargs ):
   """Determines whether the user is authorized to run a certain command in
   a certain mode.  token is a list of tokens that compose the command.
   Returns an AaaApi::AuthzResult."""
   _waitForAaa()
   r = Aaa.agent.authorizeShellCommand( uid, mode, privLevel, tokens, sessionId )
   Aaa.agent.incAuthzCounter( r.status )
   if r.status == 'denied':
      ent = Aaa.agent.getPwEnt( uid=uid )
      if ent :
         user = ent.userName
      else:
         user = "UID:%d" % uid # pylint: disable=consider-using-f-string
      Logging.log( AAA_CMD_AUTHZ_FAILED, user, " ".join( tokens ),
                   # pylint: disable-next=consider-using-f-string
                   " (%s)" % r.message if r.message else "" )
      # Account for failed authorization event
      failureMessage = f": {r.message}" if r.message else ""
      Aaa.agent.sendFailedCommandAcct( user, privLevel, sessionId, tokens, cmdType,
                                       "authorization failed" + failureMessage,
                                       **kwargs )
   return r

def authorizeShellCommand( uid, mode, privLevel, tokens, sessionId=None,
                           cmdType=None, **kwargs ):
   r = _authorizeShellCommand( uid, mode, privLevel, tokens, sessionId,
                               cmdType=cmdType, **kwargs )
   return Tac.valueToStrep( r )

def authorizeShellCommandEx( uid, mode, privLevel, tokens, sessionId=None,
                             cmdType=None, **kwargs ):
   r = _authorizeShellCommand( uid, mode, privLevel, tokens, sessionId,
                               cmdType=cmdType, **kwargs )
   return '\x00'.join( (r.status, r.message) )

def sendCommandAcct( uid, privLevel, tokens, sessionId=None, cmdType=None,
                     **kwargs ):
   """Sends command accounting to the accounting module. Tokens is a list
   of tokens that compose the command. Returns an AaaApi::AcctResult."""
   _waitForAaa()
   Aaa.agent.sendCommandAcct( uid, privLevel, tokens, sessionId,
                              cmdType=cmdType, **kwargs )
   return Tac.valueToStrep( None )

def authenticateAndAuthorizeLocalCommand( user, mode, tokens,
                                          privLevel=1 ):
   """Authenticate and Authorize a local shell command in the same call to avoid
   multiple PyClient RPCs.
   Returns an AaaApi::AuthzResultSimple that includes authorization result."""
   _waitForAaa()

   result = Tac.Value( 'AaaApi::AuthzResultSimple', status='denied', message="" )

   state = Aaa.agent.startAuthenticate( 'authnTypeLogin', 'shell', None,
                                        None, None, user, None )
   Aaa.agent.incAuthnCounter( state.status, 'shell' )

   if state.status == 'fail':
      result.message = "Unauthorized client"
      return Tac.valueToStrep( result )
   elif state.status == 'unavailable':
      result.message = "Authentication service is currently unavailable"
      return Tac.valueToStrep( result )
   elif state.status == 'unknown':
      result.message = "Unknown client identity"
      return Tac.valueToStrep( result )

   entry = Aaa.agent.getPwEnt( name=user )

   if entry is None:
      result.message = "Unauthorized client"
      return Tac.valueToStrep( result )

   ar = Aaa.agent.authorizeShellCommand( entry.userId,
                                         mode,
                                         privLevel,
                                         tokens,
                                         state.id )
   Aaa.agent.incAuthzCounter( ar.status )

   if ar.status == 'denied':
      Logging.log( AAA_CMD_AUTHZ_FAILED, user, " ".join( tokens ) )

   result.status = ar.status
   result.message = ar.message
   return Tac.valueToStrep( result )

def _authenticate( user, password, service, remoteHost, remoteUser=None, tty=None,
                   localAddr=None, localPort=None, remotePort=None ):
   """This API performs non-interactive authentication based on username/password.
   The AAA server is only expected to ask for username and password (which is
   typically the case). This is NOT used by Linux applications that use PAM (such as
   login/sshd/telnet).

   Returns: AuthenState
   """
   # pylint: disable=unexpected-keyword-arg
   state = startLoginAuthenticate( service, tty=tty, remoteHost=remoteHost,
                                   remoteUser=remoteUser, user=user,
                                   returnOriginalType=True, localAddr=localAddr,
                                   localPort=localPort, remotePort=remotePort )
   # pylint: enable=unexpected-keyword-arg
   answers = []

   authenId = state.id
   if state.status == 'inProgress':
      # we should only be asked for password
      for i in range( 4 ):
         attr = "message%d" % ( i ) # pylint: disable=consider-using-f-string
         m = getattr( state, attr )
         if m.style == 'invalid': # pylint: disable=no-else-break
            break
         elif m.style == 'promptEchoOff':
            answers.append( password )
            break
         else:
            Logging.log( AAA_AUTHEN_UNEXPECTED_MESSAGE, service, user, m.text )
            state.status = 'fail'
            break

      if state.status == 'inProgress':
         # pylint: disable=unexpected-keyword-arg
         state = continueAuthenticate( authenId, *tuple( answers ),
                                       returnOriginalType=True )
         # pylint: enable=unexpected-keyword-arg
         if state.status == 'inProgress':
            Logging.log( AAA_AUTHEN_UNEXPECTED_MESSAGE, service, user,
                         '(inProgress)' )
            state.status = 'fail'

   return state

def _authorizeSession( user, userEnt, session, tty, skipAuthz ):
   result = Tac.Value( "AaaApi::AuthenAndAuthzResult" )

   if session.status != 'open':
      result.status = 'sessionFailed'
      return result

   if not userEnt:
      userEnt = Aaa.agent.getPwEnt( user )
      if not userEnt:
         # did someone just delete the user?
         Aaa.agent.closeSession( session.id )
         result.status = 'authenFailed'
         return result

   # For now, we don't pass in sessionPid. This means we will not support session
   # timeout for those sessions, which might be actually desirable.
   # pylint: disable=unexpected-keyword-arg
   if skipAuthz:
      r = Tac.Value( "AaaApi::AuthzResult" )
   else:
      r = authorizeShell( result.uid, user=user, sessionId=session.id,
                          tty=tty, returnOriginalType=True )
   # pylint:enable=unexpected-keyword-arg

   if r.status != "allowed":
      # failed, let's close the session
      Aaa.agent.closeSession( session.id )
      result.status = 'authzFailed'
   else:
      result.status = 'allSuccess'
      result.sessionId = session.id
      result.uid = userEnt.userId
      result.gid = userEnt.groupId
      for v in r.av.values():
         if v.name:
            result.av[ v.name ] = v.val
         else:
            break

   return result

def _authenticateAndAuthorizeSessionCommon( user, password, service, remoteHost,
                                            remoteUser=None, tty=None,
                                            localAddr=None,
                                            localPort=None, remotePort=None,
                                            skipAuthz=False ):
   _waitForAaa()
   if isinstance( user, bytes ):
      user = user.decode()
   if isinstance( password, bytes ):
      password = password.decode()
   authenState = _authenticate( user, password, service, remoteHost,
                                remoteUser=remoteUser, tty=tty, localAddr=localAddr,
                                localPort=localPort, remotePort=remotePort )
   if authenState.status != 'success':
      return Tac.Value( "AaaApi::AuthenAndAuthzResult", status="authenFailed" )

   # now open session
   session = Aaa.agent.openSession( authenState.id )
   return _authorizeSession( user, None, session, tty, skipAuthz )

@valueToStrep
def authenticateAndOpenSession( user, password, service, remoteHost,
                                remoteUser=None, tty=None, localAddr=None,
                                localPort=None, remotePort=None ):
   """This API does password authentication and opens a session. Particularly,
   it does the following:

   1. _authenticate(): performs password authentication.
   2. openSession(): opens a session for the authenticated user.

   Parameters:

   service: a lower-case string that identifies the service.
   remoteHost: For remote services, specify the remote host address. It's important
               to pass this in so AAA can identify the service as a "remote service"
               which has security implications (e.g., whether allow the default
               passwordless 'admin' user to login). If it's a local service,
               specify None.

   remoteUser: username from the remote host. Leave as None unless you know.
   tty: real or pseudo tty name if available.

   Returns AaaApi::AuthenAndAuthzResult string representation.

   status: allSuccess/authenFailed/sessionFailed/authzFailed
   sessionId: the session id which should be used for AaaApi.closeSession().
   av: always empty.

   If status is 'allSuccess', a session is opened and sessionId is returned.
   The caller should call closeSession() with it later to close the session.
   If status is any of the failed codes, no cleanup is required and all other
   fields are undefined.
   """
   return _authenticateAndAuthorizeSessionCommon( user, password, service,
                                                  remoteHost,
                                                  remoteUser=remoteUser,
                                                  tty=tty,
                                                  localAddr=localAddr,
                                                  localPort=localPort,
                                                  remotePort=remotePort,
                                                  skipAuthz=True )

@valueToStrep
def authenticateAndOpenSessionExec( user, password, service, remoteHost,
                                    remoteUser=None, tty=None, localAddr=None,
                                    localPort=None, remotePort=None ):
   """
   Exec version of authenticateAndOpenSession()
   """
   result =  _authenticateAndAuthorizeSessionCommon( user, password, service,
                                                     remoteHost,
                                                     remoteUser=remoteUser,
                                                     tty=tty,
                                                     localAddr=localAddr,
                                                     localPort=localPort,
                                                     remotePort=remotePort,
                                                     skipAuthz=True )
   _printAuthenResultExec( result )

@valueToStrep
def authenticateAndAuthorizeSession( user, password, service, remoteHost,
                                     remoteUser=None, tty=None, localAddr=None,
                                     localPort=None, remotePort=None ):
   """This API does password authentication, opens a session and performs exec
   authorization. Particularly, it does the following:

   1. _authenticate(): performs password authentication.
   2. openSession(): opens a session for the authenticated user.
   3. authorizeShell(): determines if user is authorized to start a shell, and if so,
      what kind of privileges or roles should be provided.

   It basically does authenticateAndOpenSession() plus EXEC (shell) authorization.

   Parameters and return value: see authenticateAndOpenSession().

   Extra information in AaaApi::AuthenAndAuthzResult returned by EXEC authorization:

   av: av pairs returned by the server from authorization. Known avpair names
       can be found at the begining of AaaPluginLib. The most commonly used
       avpair is "privilegeLevel" with a value of an integer string between 1
       and 15.
   """
   return _authenticateAndAuthorizeSessionCommon( user, password, service,
                                                  remoteHost,
                                                  remoteUser=remoteUser,
                                                  tty=tty,
                                                  localAddr=localAddr,
                                                  localPort=localPort,
                                                  remotePort=remotePort,
                                                  skipAuthz=False )

@valueToStrep
def createAndAuthorizeSession( user, service, authenMethod, remoteHost,
                               remoteUser=None, tty=None, validUserOnly=True,
                               localAddr=None, localPort=None, remotePort=None ):
   """This API skips authentication, and is typically called after certificate based
   authentication. Particularly, it does:

   1. createSession(): creates a session for the user.
   2. authorizeShell(): determines if user is authorized to start a shell, and if so,
      what kind of privileges or roles should be provided.

   Most parameters are documented in the authenticateAndAuthorizeSession() API.

   authenMethod: specifies the non-AAA authentication method used for the user prior
                 to this call. Use a string separate from valid AAA authentication
                 methods. Examples are "sshkey" or "certificate".

   validUserOnly: see comments under createSession(). Note the default is true here,
   as we are doing AAA authorization.

   Returns AaaApi::AuthenAndAuthzResult string representation. See
   authenticateAndAuthorizeSession() for details.

   If the service does not want to perform AAA EXEC authorization, use the
   createSession() API instead.
   """
   _waitForAaa()

   userEnt = None
   if validUserOnly:
      userEnt = Aaa.agent.getPwEnt( user )
      if not userEnt:
         return Tac.Value( "AaaApi::AuthenAndAuthzResult", status="authenFailed" )

   tlsCert = authenSourceEnum.authenSourceTlscert
   session = Aaa.agent.createSession( user, service, tty, remoteHost, remoteUser,
                                      authenMethod=authenMethod,
                                      authenSource=Tac.enumValue(
                                         "Aaa::AuthenSource", tlsCert ),
                                      localAddr=localAddr,
                                      localPort=localPort, remotePort=remotePort )
   return _authorizeSession( user, userEnt, session, tty, False )

def verifyAuthenSource( authenSource ):
   '''
      Helper function to validate the authenSource value
   '''
   if authenSource:
      return Tac.enumValue( "Aaa::AuthenSource", authenSource )
   return Tac.enumValue( "Aaa::AuthenSource", authenSourceEnum.authenSourceNone )
