#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import grp
import os
import AaaDefs
from AaaPluginLib import TR_INFO
from BothTrace import traceX as bt
import Logging
import Tac
from Tracing import traceX

AAA_USERINFO_IO_ERROR = Logging.LogHandle(
      "AAA_USERINFO_IO_ERROR",
      severity=Logging.logError,
      fmt="Error generating user info file (%s)",
      minInterval=60,
      explanation="An attempt to generate user info file failed "
                  "because of a filesystem error.",
      recommendedAction="Check to see if the filesystem has run out of disk space. "
                        "Contact TAC Support if this problem persists " )

syncInterval = 0.1
errorSyncInterval = 1

class UserInfoSynchronizer:
   """This implements a service that synchronizes aaa/status to
   /etc/AaaUserInfo file for nss_aaa to process getpwname() and friends.
   This is much faster than having to PyClient into Aaa."""

   def __init__( self, agent ):
      self.agent_ = agent
      self.status_ = agent.status
      self.config_ = agent.config
      self.activity_ = Tac.ClockNotifiee()
      self.activity_.handler = self.syncUserInfoFile
      self.activity_.timeMin = Tac.endOfTime
      self.syncPending_ = False
      try:
         self.gid = grp.getgrnam( "eosadmin" ).gr_gid
      except KeyError:
         self.gid = None
      self.scheduleSync()

   def writeLine( self, f, userName, userId, groupId, realName, homeDir,
                  shell ):
      # the format should be consistent with AaaPam
      # pylint: disable-next=consider-using-f-string
      f.write( "%s:x:%d:%d:%s:/home/%s:%s\n" %
               ( userName, userId, groupId, realName, homeDir, shell ) )

   def syncUserInfoFile( self ):
      """synchronize aaa/status and aaa/local/config to user info file"""
      if not self.agent_.initialized:
         # just reschedule
         self.scheduleSync()
         return
      bt( TR_INFO, "synchronize user info file" )
      with self.agent_.mutex:
         try:
            tmp = f"{AaaDefs.userInfoFileName}-{os.getpid()}"
            with open( tmp, "w" ) as f:
               # write logged in users
               for user, acct in self.status_.account.items():
                  self.writeLine( f, user, acct.id,
                                  acct.id if self.gid is None else self.gid,
                                  f"Eos-{user} ({acct.authenMethod})",
                                  user,
                                  self.agent_.getShellFromAccount( acct ) )
               for plugin in self.agent_.plugins.values():
                  plugin.writeUserInfo( f )
            os.rename( tmp, AaaDefs.userInfoFileName )
            self.syncPending_ = False
         except OSError as e:
            Logging.log( AAA_USERINFO_IO_ERROR, e.strerror )
            # reschedule
            self.scheduleSync( interval=errorSyncInterval )

   def scheduleSync( self, interval=syncInterval ):
      traceX( TR_INFO, "schedule user info file sync" )
      with self.agent_.mutex:
         self.syncPending_ = True
         self.activity_.timeMin = Tac.now() + interval

   def syncPending( self ):
      return self.syncPending_

   @Tac.handler( 'account' )
   def handleAccount( self, name ):
      self.scheduleSync()

