# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# An accounting provider that sends all the accounting records to syslog.
#
import AaaPluginLib
import Logging
import Tac
import Tracing

t0 = Tracing.trace0

ACCOUNTING_CMD = Logging.LogHandle(
      "ACCOUNTING_CMD",
      severity=Logging.logInfo,
      fmt="%s task_id=%d start_time=%d timezone=%s service=%s priv-lvl=%d cmd=%s",
      explanation="Command accounting message.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

ACCOUNTING_EXEC = Logging.LogHandle(
      "ACCOUNTING_EXEC",
      severity=Logging.logNotice,
      fmt="%s task_id=%d start_time=%d timezone=%s service=%s%s",
      explanation="Exec accounting message.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

ACCOUNTING_SYSTEM = Logging.LogHandle(
      "ACCOUNTING_SYSTEM",
      severity=Logging.logNotice,
      fmt="%s task_id=%d start_time=%d timezone=%s service=%s event=%s reason=%s",
      explanation="System accounting message.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

taskId = 0
def getTaskId():
   global taskId
   taskId = taskId + 1
   return taskId

def tty( session ):
   if session and session.tty:
      return session.tty
   return 'unknown'

def remoteAddr( session ):
   if session and session.remoteHost:
      return session.remoteHost
   return 'unknown'

def timezone():
   import time # pylint: disable=import-outside-toplevel
   isdst = time.localtime().tm_isdst
   timezone = None # pylint: disable=redefined-outer-name
   if isdst:
      timezone = time.tzname[ 1 ]
   else:
      timezone = time.tzname[ 0 ]
   return timezone

class LoggingAcctPlugin( AaaPluginLib.Plugin ): # pylint: disable=abstract-method
   def __init__( self, aaaConfig ):
      AaaPluginLib.Plugin.__init__( self, aaaConfig, "logging" )

   def ready( self ):
      t0( "LoggingAcctPlugin ready()" )
      return True
   
   def sendCommandAcct( self, method, user, session, privlevel, timestamp, tokens,
                        cmdType=None, **kwargs ):
      t0( "sendCommandAcct for method", method, "user", user, "privlevel",
          privlevel, "timestamp", timestamp, "tokens", tokens )
      assert method == self.name
      logEntry = { 'user' : user,
                   'tty' : tty( session ),
                   'remoteAddr' : remoteAddr( session ),
                   'action' : 'stop',
                   'priv-lvl' : privlevel,
                   'timezone' : timezone(),
                   'start_time' : timestamp,
                   'task_id' : getTaskId(),
                   'service' : 'shell',
                   'cmd' : ' '.join( tokens ) + ' <cr>',
                   }

      # sample for formatting
      # pylint: disable-next=pointless-string-statement
      '''ACCOUNTING-6-CMD: arastra tty1 172.17.18.47 stoptask_id=528
      start_time=1280796967 timezone=PDT service=shell priv-lvl=15
      cmd=show running-config <cr>'''
      Logging.log( ACCOUNTING_CMD,
                   ' '.join( ( logEntry[ 'user' ],
                               logEntry[ 'tty' ],
                               logEntry[ 'remoteAddr' ],
                               logEntry[ 'action' ] ) ),
                   logEntry[ 'task_id' ],
                   logEntry[ 'start_time' ],
                   logEntry[ 'timezone' ],
                   logEntry[ 'service' ],
                   logEntry[ 'priv-lvl' ],
                   logEntry[ 'cmd' ] )
      return 'acctSuccess', ''

   def sendShellAcct( self, method, user, session, action, startTime,
                      elapsedTime=None ):
      t0( "sendShellAcct for method", method, "user", user, "action",
          action, "startTime", startTime, "elapsedTime", elapsedTime )
      assert method == self.name
      logEntry = { 'user' : user,
                   'tty' : tty( session ),
                   'remoteAddr' : remoteAddr( session ),
                   'action' : action,
                   'timezone' : timezone(),
                   'service' : 'shell',
                   'start_time' : int( startTime ),
                   }
      if action == 'start':
         assert session
         logEntry[ 'task_id' ] = getTaskId()
         session.execAcctTaskId = logEntry[ 'task_id' ]
      if action == 'stop':
         assert session
         if session.execAcctTaskId:
            logEntry[ 'task_id' ] = session.execAcctTaskId
         else:
            logEntry[ 'task_id' ] = getTaskId()
         assert elapsedTime is not None
         logEntry[ 'elapsed_time' ] = elapsedTime

      # sample for formatting
      # pylint: disable-next=pointless-string-statement
      '''ACCOUNTING-5-EXEC: arastra tty2 172.17.18.47 start
      task_id=118 start_time=1281566358 timezone=PDT service=shell '''
      
      elapsedTimeStr = ""
      if action == 'stop':
         # pylint: disable-next=consider-using-f-string
         elapsedTimeStr = " elapsed_time=%s" % logEntry[ 'elapsed_time' ]
         
      Logging.log( ACCOUNTING_EXEC,
                   ' '.join( ( logEntry[ 'user' ],
                               logEntry[ 'tty' ],
                               logEntry[ 'remoteAddr' ],
                               logEntry[ 'action' ] ) ),
                   logEntry[ 'task_id' ],
                   logEntry[ 'start_time' ],
                   logEntry[ 'timezone' ],
                   logEntry[ 'service' ],
                   elapsedTimeStr )
      return 'acctSuccess', ''

   def sendSystemAcct( self, method, event, startTime, reason, action ):
      t0( "sendSystemAcct for method", method, "action",
          action, "startTime", startTime, "reason", reason )
      assert method == self.name
      logEntry = { 'user' : 'unknown',
                   'tty' : 'unknown',
                   'remoteAddr' : 'unknown',
                   'action' : action,
                   'service' : 'system',
                   'task_id' : getTaskId(),
                   'start_time' : int( startTime ),
                   'timezone': timezone(),
                   'reason' : reason,
                   'event' : event,
                   }
      
      # samples for formatting
      # pylint: disable-next=pointless-string-statement
      '''ACCOUNTING-5-SYSTEM: unknown unknown unknown
      stop    task_id=265 start_time=1041611719   timezone=EST    service=system
      event=sys_acct  reason=shutdown

      ACCOUNTING-5-SYSTEM:  unknown unknown unknown
      start  task_id=1  start_time=1041611719 timezone=EST    service=system
      event=sys_acct reason=reload'''

      Logging.log( ACCOUNTING_SYSTEM,
                   ' '.join( ( logEntry[ 'user' ],
                               logEntry[ 'tty' ],
                               logEntry[ 'remoteAddr' ],
                               logEntry[ 'action' ] ) ),
                   logEntry[ 'task_id' ],
                   logEntry[ 'start_time' ],
                   logEntry[ 'timezone' ],
                   logEntry[ 'service' ],
                   logEntry[ 'event' ],
                   logEntry[ 'reason' ] )
      return 'acctSuccess', ''

def Plugin( ctx ):
   return LoggingAcctPlugin( ctx.aaaAgent.config )

