# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing
import TacSigint
import socket, sys
import threading
from errno import ECONNRESET, ECONNABORTED

t0 = Tracing.trace0

# Send command to specified socket and print received response. Interacts 
# directly with owner of the socket (Agent) without Sysdb interaction.
def runCommand( sockName, command, stringBuff=None, timeout=600 ):
   """
   sockName is the name of the socket created by Agent during startup using 
   Ark AgentSocketHandler interface. 

   First word in the command determines commandType, all remaining words are 
   passed as arguments to corresponding handler. For example "diag show units"
   command invokes "diag" handler with "show units" as command arguments.

   If stringBuff (a file-like object) is specified, all output will be sent to it
   rather than to stdout.

   Return 0 on success
          1 on connection error (Agent not started or doesn't exist)
   """

   def handleOutput( output ):
      if stringBuff:
         stringBuff.write( output )
      else:
         sys.stdout.write( output )
         sys.stdout.flush()

   # socket in abstract name space
   abssockName = "\0s%s" % sockName # pylint: disable=consider-using-f-string
   # pylint: disable-next=consider-using-f-string
   t0( "Connecting to: %s Command: %s" % ( abssockName, command ) )

   s = socket.socket( socket.AF_UNIX, socket.SOCK_STREAM )
   s.settimeout( timeout )
   try:
      s.connect( abssockName )
   except OSError:
      handleOutput( "Cli connection error\n" )
      return 1

   t0( "Sending command" )
   s.send( command.encode( "utf-8" ) )

   with TacSigint.immediateMode( threading.current_thread().getName() ==
                                 'MainThread' ):
      try:
         while True : 
            data = s.recv( 1024 * 1024 ).decode( "utf-8" )
            if data:
               handleOutput( data )
            else:
               t0( "Receive completed" )
               break
      except KeyboardInterrupt: # pylint: disable=try-except-raise
         raise
      except OSError as why:
         if why.errno not in [ ECONNRESET, ECONNABORTED ]:
            raise
      finally:
         t0( "Closing connection" )
         s.close()
   return 0
