#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# This script runs directly on the dut and creates a tar archive containing
# troubleshooting/debug information for any agent. A gzipped tar archive with the
# name "<agent>-debug-{date}.tgz will be placed under /tmp

import datetime
import os
import optparse # pylint: disable=deprecated-module
import tarfile
import tempfile
import glob
import time
import Tac
import re
import sys

def openTempTarGzFile():
   '''create a temporary tar.gz file'''
   ( tmpFd, tmpFileName ) = tempfile.mkstemp( suffix='.tar.gz' )
   os.close( tmpFd )
   # pylint: disable-next=consider-using-with
   tmpTarFile = tarfile.open( name=tmpFileName, mode='w:gz', dereference=True )
   return tmpTarFile

def closeTarGzFile( tmpTarFile ):
   '''close the tar.gz file'''
   tmpTarFile.close()

def fileListSortedByModTime( fileList ):
   '''return list of files sorted by last modified time'''
   sortedFileList = []
   if not fileList:
      return sortedFileList
   for fl in fileList:
      stats = os.stat( fl )
      lastModTime = time.localtime( stats[ 8 ] )
      sortedFileList.append( ( lastModTime, fl ) )
      sortedFileList.sort()
   return [ f for ( _, f ) in sortedFileList ]

def scp( src, dst ):
   argv = [ 'scp', src, dst ]
   try:
      Tac.run( argv, stderr=Tac.CAPTURE )
   except Tac.SystemCommandError as e:
      output = re.sub(
         "Warning: Permanently added '.+' \\([RD]SA\\) to "
         "the list of known hosts.", "",
         e.output ).strip()
      if re.match( r"Permission denied \(.+\)\.", output ):
         # Probably incorrect username or password.
         output = "Permission denied"
      elif output.startswith( 'ssh: ' ):
         # An error such as 'Name or service not known'.
         output = output[ len( 'ssh: ' ) : ]
      elif output.startswith( 'scp: ' ):
         # An error such as 'No such file or directory'.
         output = output[ len( 'scp: ' ) : ]
      print( "Error: %s" % output )
      return False
   return True

def addFileToTar( tarFile, file ):
   try:
      tarFile.add( file )
      print( file )
   except PermissionError as e:
      print( f'Skipping { file } : { e }' )

def main( agent, agentLogFunction=None ):
   AgentLogFiles = [ '/var/log/qt/%s*' % agent,
                     '/var/log/agents/%s*' % agent,
                     '/var/log/messages' ]
   AgentDumpFiles = [ '/var/log/%s_dump.*' %
                      ( 'ribd' if agent == 'Rib' else agent ) ]
   AgentCoreFiles = [ '/var/core/core.*.%s*' % agent ]
   op = optparse.OptionParser( prog='%s' % sys.argv[ 0 ],
                               usage='%s [options]' % sys.argv[ 0 ],
                               description='Script for gathering %s agent '
                               'debugging information' % agent )
   op.add_option( '-n', '--no-core', action='store_true',
                  help='Do not include %s core files as part of tar archive'
                  % agent )
   op.add_option( '-r', '--no-%s-dump' % agent.lower(), action='store_true',
                  dest='no_agent_dump',
                  help='Do not include %s dump files as part of tar archive'
                  % agent )
   op.add_option( '-c', '--clean', action='store_true',
                  help='remove the %s dump and core files after archival' )
   op.add_option( '-s', '--scp', action='store',
                  help='scp the tar archive to a remote server' )
   op.add_option( '-o', '--output', action='store',
                  help='save the .tar.gz to a particular file' )
   ( opts, _ ) = op.parse_args()

   agentTmpTarFile = openTempTarGzFile()
   print( "Generating %s-debug tar archive" % agent )
   print( "Adding the following files ..." )

   for agentLogDir in AgentLogFiles:
      fl = glob.glob( agentLogDir )
      for f in fileListSortedByModTime( fl ):
         addFileToTar( agentTmpTarFile, f )

   if not opts.no_agent_dump:
      # Call the method implemented by the agent for specific state dumps
      # (if it exists) to run commands and collect agent runtime state
      if agentLogFunction:
         agentLogFunction()

      for f in fileListSortedByModTime( glob.glob( AgentDumpFiles[ 0 ] ) ):
         addFileToTar( agentTmpTarFile, f )

   if not opts.no_core:
      for f in fileListSortedByModTime( glob.glob( AgentCoreFiles[ 0 ] ) ):
         addFileToTar( agentTmpTarFile, f )

   now = datetime.datetime.now()
   hostName = os.uname()[ 1 ]
   agentDebugTarFile = opts.output
   if not opts.output:
      agentDebugTarFile = '/tmp/%s-debug-' % agent + '%s-' % hostName + \
         now.strftime( "%Y%m%d_%H%M%S" ) + '.tar.gz'
   closeTarGzFile( agentTmpTarFile )
   os.rename( agentTmpTarFile.name, agentDebugTarFile )
   os.chmod( agentDebugTarFile, 0o644 )

   print( "to %s" % agentDebugTarFile )

   if opts.clean:
      print( "cleaning up following %s debug files ..." % agent )
      for f in fileListSortedByModTime( glob.glob( AgentDumpFiles[ 0 ] ) ):
         os.remove( f )
         print( f )

      for f in fileListSortedByModTime( glob.glob( AgentCoreFiles[ 0 ] ) ):
         os.remove( f )
         print( f )

   if opts.scp:
      print( f"copying {agentDebugTarFile} to {opts.scp}" )
      if scp( agentDebugTarFile, opts.scp ):
         os.remove( agentDebugTarFile )
