# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Tac
import Tracing
import CliSession
from CliSession import PreCommitHandlerError
from GenericReactor import GenericReactor
from TypeFuture import TacLazyType
from abc import abstractmethod

t0 = Tracing.Handle( "AirStreamLib" ).trace0

RedundancyMode = TacLazyType( 'Redundancy::RedundancyMode' )

# ToExternalSmWrapper keeps external entities in synced with native entities
class ToExternalSmWrapperBase( object ):
   # Varaiables to be set:
   # - nativePathAndType is a list of tuples of Sysdb path and type of the
   #   native entities
   # - externalPathAndType is a list of tuples of Sysdb path and type of external
   #   entities
   # - smType is the type of the state-machine that synchrozies the external entities
   #   with the native entities
   #
   # Once the constructor runs,
   # - nativePaths will be mounted in 'r' mode
   # - externalPathList will be mounted in 'w' mode
   #
   # Developer must specify these variables and implement createSm() that instantiate
   # the actual state machine using self.nativeEntity and self.externalEntity methods
   # These methods take index to return the mounted entity corresponding to the
   # respective **PathAndType list variables
   #
   # The string in the format "nativePath + smType" will be used as the key to
   # register the sm to StatefulSmManager. The pair must be unique.
   nativePathAndType = []
   externalPathAndType = []
   smType = ''

   def __init__( self, em, mg, agent ):
      self.toExternalSm = None
      self.mountedEntities_ = {}
      self.em = em
      self.doMounts( mg )
      self.agent = agent
      self.redundancyReactor_ = None

   def doMounts( self, mg ):
      for path, typ in self.nativePathAndType:
         assert not path.startswith( '/' ), "Path shouldn't start with /"
         self.mountedEntities_[ path ] = mg.mount( path, typ, 'r' )
      for path, typ in self.externalPathAndType:
         assert not path.startswith( '/' ), "Path shouldn't start with /"
         self.mountedEntities_[ path ] = mg.mount( path, typ, 'w' )

   def nativeEntity( self, idx=0 ):
      return self.mountedEntities_.get( self.nativePathAndType[ idx ][ 0 ] )

   def externalEntity( self, idx=0 ):
      return self.mountedEntities_.get( self.externalPathAndType[ idx ][ 0 ] )

   @abstractmethod
   def createSm( self ):
      # Developer must create the sm and set it to self.toExternalSm
      raise NotImplementedError(
         "Child class must implement createSm()" )

   def run( self ):
      self.redundancyReactor_ = GenericReactor(
            self.agent.redundancyStatus(), [ 'mode' ],
            self.handleRedundancyMode, callBackNow=True )

   def handleRedundancyMode( self, notifiee=None ):
      if self.agent.redundancyStatus().mode != RedundancyMode.active:
         return

      # Some cohab btests load the same ConfigAgent plugin multiple times
      # Create and register the SM only if it's not already done.
      if not self.alreadyCreated():
         t0( 'create and register ToExternalSm with key ' + self.key )
         self.createSm()
         if self.toExternalSm:
            self.registerSm( self.toExternalSm )
      else:
         t0( 'skip creating SM with key ' + self.key )

   @property
   def key( self ):
      assert self.nativePathAndType and self.smType
      return self.nativePathAndType[ 0 ][ 0 ] + "," + self.smType

   def registerSm( self, sm ):
      """Register the sm to the StatefulSmManager
      The key is the '%nativePath,%smType'.  This is to prevent any same SM to
      be registered to the same path, which does not make sense.
      """
      assert sm
      ssm = Tac.singleton( 'AirStream::StatefulSmManager' )
      key = self.key
      assert key not in ssm.sm, "Register same SM with the same native path."
      t0( 'register SM with key ' + key )
      ssm.sm[ key ] = sm

   def alreadyCreated( self ):
      ssm = Tac.singleton( 'AirStream::StatefulSmManager' )
      return self.key in ssm.sm

# An error to be raised when one-time-toNativeSyncher found issues
# The userMsg is the clean error to the user
# The debugMsg is more debug info to be written to quick trace
class ToNativeSyncherError( PreCommitHandlerError ):
   def __init__( self, sessionName, syncherName, userMsg, debugMsg=None ):
      fullDebugMsg = "Session %s has problem while running %s: %s : %s" % (
            sessionName, syncherName, userMsg, debugMsg )
      PreCommitHandlerError.__init__( self, userMsg, fullDebugMsg )

def getSessionEntity( entMan, sessionName, path ):
   sessionDir = entMan.root().entity[ 'session/sessionDir/' + sessionName ]
   return sessionDir.entity[ path ]

# Register a custom entity copy handler for OpenConfig paths
# This is required for all external entities that use stateful sync method.
# dirTypeName is the type of HandlerDir.
# path/typeName are the path/type that are applicable to this copy handler.
# See CliSession.registerCustomCopyHandler() for more details of path/typeName.
def registerCopyHandler( entMan, uniqueName,
                         dirTypeName="AirStream::ExternalEntityCopyHandlerDir",
                         path="", typeName="", commitExternal=False ):
   assert path or typeName, \
         "Entity copy handler should be registered by path or type or both"
   handlerDir = CliSession.registerCopyHandlerDir( entMan,
                  uniqueName, dirTypeName )
   handlerDir.copyHandler = ()
   handler = handlerDir.copyHandler
   handler.copyOnCommit = commitExternal
   handler.initialize()

   CliSession.registerCustomCopyHandler( entMan, path, typeName, handler )
