#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import difflib
import argparse

def chooseCommand( choice, commands ):
   '''Match the user's command against a list of known commands. If it
   doesn't match exactly, a fuzzy match is applied, and if we get a
   unique fuzzy match we return that. Otherwise, we give a list of
   likely matches for the user to try again.
   '''

   # de-dup commands, so that the close and prefix matches
   # can work properly even if the caller passed duplicates
   commands = set( commands )

   if choice in commands:
      return choice

   prefixMatches = [ c for c in commands if c.startswith( choice ) ]
   if len( prefixMatches ) == 1:
      # pylint: disable-next=consider-using-f-string
      print( "Assuming you meant: %r" % prefixMatches[ 0 ] )
      return prefixMatches[ 0 ]

   close = difflib.get_close_matches( choice, commands, cutoff=0.80 )
   if len( close ) == 1:
      # pylint: disable-next=consider-using-f-string
      print( "Assuming you meant: %r" % close[ 0 ] )
      return close[ 0 ]

   if close:
      print( "Maybe you meant one of:", ", ".join( close ) )
   elif prefixMatches:
      print( "Maybe you meant one of:", ", ".join( prefixMatches ) )

   return None


class OnlyOnce( argparse.Action ):
   '''
   This can be used as the action in an argparse add_argument call, to
   allow the option to be allowed a single time. parser error() is generated
   otherwise.
   '''
   def __init__( self, option_strings, dest, nargs=None, **kwargs ):
      if nargs is not None:
         raise ValueError( "nargs not allowed" )
      super().__init__( option_strings, dest,
                        **kwargs )
      self._called = False

   def __call__( self, parser, namespace, values, option_string=None ):
      if self._called:
         parser.error( f"{option_string} specified multiple times " )
      self._called = True
      setattr( namespace, self.dest, values )

