# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ast
import re
import subprocess
import yaml

def getPythonDocString( pathToFile ):
   '''
   Returns the doc string inside the python file
   without executing it, by parsing into an AST. The function
   does not validate pathToFile and will result in an exception
   if the path is invalid
   This function will raise exception if the file open fails or
   the ast parse fails.
   '''
   try:
      with open( pathToFile ) as f:
         buf = f.read()
      return ast.get_docstring( ast.parse( buf, filename=pathToFile ) )
   except SyntaxError:
      # Stestd still uses this to parse python2 files that aren't py3 compatible in
      # older releases
      cmd = [ "python2", "/usr/bin/DocString", pathToFile ]
      output = subprocess.check_output( cmd, text=True )
      return output.strip()

def getYamlInfoFromDocString( pathToFile, tagsToParse=None ):
   '''
   This function parses a python docstring into different
   sections and returns the YAML information for each section
   as a dictionary, keyed by the identified tag.
   For this purpose it uses AID2821 information to identify
   the tags and section information. It enforces that
   the start of a new section is marked by '@' in the beginning
   of a new line.
   If tagsToParse is not None, it is expected to be a list. In this case
   the content is parsed as YAML only for those tags that are in the list.
   This allows for cases where there are some tag content that are useful
   to be parsed as YAML and some that are not.
   This function will raise exception if the YAML parsing fails.
   '''
   if tagsToParse:
      assert isinstance( tagsToParse, list )

   info = {}
   docStr = getPythonDocString( pathToFile )
   if not docStr:
      return info
   else:
      try:
         return getYamlInfoFromDocStringBuffer( docStr, tagsToParse )
      except yaml.parser.ParserError:
         print( f"Failed parsing yaml in {pathToFile}:\n{docStr}\n" )
         raise

def getYamlInfoFromDocStringBuffer( docStr, tagsToParse=None ):
   '''
   This function parses a python docstring that has already been read into memory.
   '''

   # starts with @<tag> on a new line followed by lines which represent
   # data to be parsed as YAML data. It also accepts @<tag>@ as long as
   # the line ends on '@'. Unfortunately we have mix of both in our
   # code, so be generous.
   info = {}
   regex = re.compile( r'^(\n?@)((?:(?!@$)\S)+)(@$)?((?:(?!^@).)*)',
                       re.DOTALL | re.M )
   parsed = regex.findall( docStr )
   if not parsed:
      return info

   for p in parsed:
      tag = p[ 1 ]
      if tagsToParse and tag not in tagsToParse:
         continue
      content = p[ 3 ]
      structured = yaml.safe_load( content )
      # If there isn't any yaml to be found, yaml.load will return None.
      # Convert that to an empty list so callers can handle it without special
      # code.
      info[ tag ] = structured or []
   return info
