# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

''' This module provides ersatz versions of stuff from numpy, that is good
enough for the minimal use cases we have for them, and fits the idiomatic use
we have made of them in Arista. '''

import struct

def base_repr( val, base=36, zero='0' ):
   ''' return a string representation of 'bar', in base 'base'. '''
   return zero if val == 0 else base_repr( val // base, base, "" ) + \
         '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'[ val % base ]

def asFloat32( f ):
   ''' truncate a float value to fit in a 32-bit IEEE float. '''
   if not isinstance( f, float ):
      f = float( f )
   return struct.unpack( 'f', struct.pack( 'f', f ) )[ 0 ]

def array_split( array, partitions ):
   ''' Split an array into "partitions" chunks of contiguous items. Nominally,
   there are len(array) / partitions items in each chunk, but if the length of
   the list is not a multiple of the number of partitions, the initial chunks
   have one additional item each to account. If the number of partitions
   exceeds the size of the list, there may be empty partitions in the result
   '''

   if partitions == 0:
      raise ValueError( "cannot divide a list into 0 partitions" )

   smallchunk, residue = divmod( len( array ), partitions )

   result = []
   pos = 0
   for i in range( partitions ):
      nextpos = pos + smallchunk + ( 1 if i < residue else 0 )
      result.append( array[ pos : nextpos ] )
      pos = nextpos
   return result
